/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
// Copier.cpp
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//

#include "DataIterator.H"
#include "IntVect.H"
#include "Copier.H"
#include "MayDay.H"
#include "LayoutIterator.H"
#include "SPMD.H"

using std::ostream;

Copier::Copier(const DisjointBoxLayout& a_level, const BoxLayout& a_dest)
{
  define(a_level, a_dest, IntVect::TheZeroVector());
}

Copier::Copier(const DisjointBoxLayout& a_level,
	       const BoxLayout& a_dest,
	       const IntVect& a_destGhost)
{
  define(a_level, a_dest, a_destGhost);
}

void Copier::clear()
{
  for(unsigned int i=0; i<m_localMotionPlan.size(); ++i)
    {
      delete m_localMotionPlan[i];
    }
  for(unsigned int i=0; i<m_fromMotionPlan.size(); ++i)
    {
      delete m_fromMotionPlan[i];
    }
  for(unsigned int i=0; i<m_toMotionPlan.size(); ++i)
    {
      delete m_toMotionPlan[i];
    }

  m_localMotionPlan.resize(0);
  m_fromMotionPlan.resize(0);
  m_toMotionPlan.resize(0);
}

void Copier::define(const DisjointBoxLayout& a_level,
		    const BoxLayout& a_dest)
{
  define(a_level, a_dest, IntVect::TheZeroVector());
}

void Copier::define(const DisjointBoxLayout& a_level,
		    const BoxLayout& a_dest,
		    const IntVect& a_ghost)
{
  assert(a_level.isClosed());
  assert(a_dest.isClosed());

  clear();

  //bool self = a_dest == a_level;  
  const DisjointBoxLayout& level= a_level;
  const BoxLayout&         dest = a_dest;

  unsigned int myprocID = procID();
  
  for(LayoutIterator to(a_dest.layoutIterator()); to.ok(); ++to)
    {
      Box ghost(dest[to()]);
      ghost.grow(a_ghost);
      unsigned int toProcID = dest.procID(to());
      for(LayoutIterator from(a_level.layoutIterator()); from.ok(); ++from)
	{
	  unsigned int fromProcID = level.procID(from());
	  //   if(self && from() == to()){
	    // don't copy onto self valid region when performing exchange.
	    // a_dest == a_level is my signal that I'm performing an
	    // 'exchange' type operation.
	  
	  if(toProcID != myprocID && fromProcID != myprocID){
	    // don't keep track of copy operations that don't involve this processor
	  }
	  else{
	    const Box& fromBox = level[from()];
	    if (ghost.intersectsNotEmpty(fromBox))
		  {
			Box box(ghost);
			box&=fromBox;
			MotionItem* item = new MotionItem(DataIndex(from()), DataIndex(to()), box);
			if(item == NULL){
			  MayDay::Error("Out of Memory in copier::define");
			}
			if(toProcID == fromProcID) // local move
			  m_localMotionPlan.push_back(item);
			else if(fromProcID == myprocID)
			  {
				item->procID = toProcID;
				m_fromMotionPlan.push_back(item);
			  }
			else
			  {
				item->procID = fromProcID;
				m_toMotionPlan.push_back(item);
			  }
	      }
	    if(fromBox.smallEnd(0) > ghost.bigEnd(0)){
	      //can skip the rest, since we know that the smallEnd
	      // of all the remaining boxes
	      // are lexigraphically beyond this ghosted box.
	      from.end();
	    }
						
	  }
	}
    }
}

int Copier::print() const
{
  std::cout << *this;
  return 0;
}

ostream& operator<<(ostream& os, const Copier& copier)
{
  os << "local("<<procID()<<"): ";
  for(CopyIterator it(copier, CopyIterator::LOCAL); it.ok(); ++it)
    {
      os << it().region;
    }
  os << "\nfrom("<<procID()<<"): ";
  for(CopyIterator it(copier, CopyIterator::FROM); it.ok(); ++it)
    {
      os << it().region<<"["<<it().procID<<"]";
    }
  os << "\nto("<<procID()<<"): ";
  for(CopyIterator it(copier, CopyIterator::TO); it.ok(); ++it)
    {
      os << it().region<<"["<<it().procID<<"]";
    }
  os<<"\n";
  return os;
}

Copier::~Copier()
{
  clear();
}

