/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.

#ifndef DATAITERATOR_H
#define DATAITERATOR_H

#include "DataIndex.H"
#include "BoxLayout.H"
#include "SPMD.H"

///An Iterator based on a BoxLayout object. 
/**
   An Iterator based on a BoxLayout object.  It does not
   support a dereferencing operation, since it is intended
   to work with all of BoxLayouts, DisjointBoxLayouts, BoxLayoutDatas
   LevelDatas, and any object that is built on top
   of a BoxLayout object.  DataIterator access the data in a BoxLayout-based
   object in a data-parallel manner. Order is not defined.


*/


class DataIterator
{
public:
  // default copy constructor should be fine

  /// a null constructed DataIterator will return false on ok()
  DataIterator();

  ///
  inline const DataIndex& operator()() const;

  ///
  inline void operator++();

  ///
  inline bool ok() const;

  ///
  inline void reset();

  ///
  inline void begin();

  ///
  /**
     skip the iterator to the end. ok() will return false after this
     method is called.
  */
  void end();

private:

  friend class BoxLayout;
  friend class DisjointBoxLayout;

  DataIterator(const BoxLayout& boxlayout, const int* layoutID);

  //int m_begin, m_end;

  const BoxLayout m_layout;

  unsigned int m_index;

  DataIndex m_current;

  unsigned int m_procID;

};

inline DataIterator::DataIterator()
  : m_index(0), m_procID(0)
{}


inline const DataIndex& DataIterator::operator()() const
{
  assert(ok());
  return m_current;
}

inline void DataIterator::operator++()
{
  const BoxLayout::Entry* box;
  while(++m_index < m_layout.size())
    {
      box = &(*(m_layout.m_boxes))[m_index];
      if(box->m_procID == m_procID)
	{
	  m_current.m_index = box->index;
	  return;
	}
    }

}

inline bool DataIterator::ok() const
{
  return m_index < m_layout.size();
}



inline void DataIterator::reset()
{
  begin();
}

inline void DataIterator::begin()
{
  m_index = 0; 
  const BoxLayout::Entry* box;
  while(m_index < m_layout.size())
    {
      box = &(*(m_layout.m_boxes))[m_index];
      if(box->m_procID == m_procID)
	{
	  m_current.m_index = box->index;
	  return;
	}
      ++m_index;
    }
}


#endif 
