/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/

//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//


#include "BitSet.H"
#include "IntVect.H"
#include "Box.H"


class DenseIntVectSetIterator;
///
/**  Performance-oriented class that reproduces most of the
  functionality of IntVectSet.  The one function that is
  not supported is union (|= ).  This object stores a dense
  representation of an IntVectSet over the box of definition*/
class DenseIntVectSet
{
public:
  ///
  DenseIntVectSet(){;}
  
  ///
  /** you can either have the domain begin all true, or all false */
  DenseIntVectSet(const Box& a_domain, bool init = true);
  // copy, and operator= should be fine

  /// 
  /**
     make this set the complement of itself and the input
   */
  DenseIntVectSet& operator-=(const Box& box);

  /// 
  /**
     make this set the complement of itself and the input
   */
  DenseIntVectSet& operator-=(const IntVect& intvect);

  /// 
  /**
     make this set the complement of itself and the input
   */
  DenseIntVectSet& operator-=(const DenseIntVectSet& ivs);

  /// 
  /**
     make this set the intersection of itself and the input
   */
  DenseIntVectSet& operator&=(const Box& box);

  /// 
  /**
     make this set the intersection of itself and the input
   */
  DenseIntVectSet& operator&=(const DenseIntVectSet& ivs);

  ///
  /**
     Shift every cell in the set by the input.
   */
  void shift(const IntVect& iv);

  //
  /* slower, constant time pointwise access.
     You should not build a BoxIterator, then access this
     method if you intend to iterate over the objects true
     or false terms.  you should build a DenseIntVectSetIterator */
  bool operator[](const IntVect& index) const;

  ///
  /** returns 'true' if the entire set of points specified
	  by the box 'b' is a member of the IntvectSet */
  bool contains(const Box& box) const;

  ///
  /**
     coarsen the set by the input
   */
  void coarsen(int iref);

  ///
  /**
     refine the set by the input
   */
  void refine(int iref);

  ///
  /**
     grow the set by the input
   */
  void grow(int igrow);

  ///
  /**
     grow the set by the input
   */
  void grow(int idir, int igrow);


  inline const Box& box() const;

  ///
  /** see IntVectSet
   */
  void nestingRegion(int radius, const Box& domain);

  ///
  /**
     return true if the set has no points
   */
  bool isEmpty() const; //cheaper than numPts by wide margin
  
  ///
  /**
     return true if the set consists of a set of a full box
   */
  bool isFull() const;  // cheaper than numPts, bu significant margin

  ///
  /**
     return the number of points in the set
   */
  int  numPts() const;

private:
  void grow(const IntVect& iv);
  Box m_domain;
  BitSet m_bits;
  friend DenseIntVectSetIterator;
};

///
/** DenseIntVectSetIterator iterates over all the 'true'
  members of a DenseIntVectSet set.*/
class DenseIntVectSetIterator
{
public:

  ///
  DenseIntVectSetIterator();
  
  ///
  DenseIntVectSetIterator(const DenseIntVectSet& ivs);

  ///
  void define(const DenseIntVectSet& ivs);
  //default null, assignment, copy and destructor should work fine.

  ///
  const IntVect& operator()() const ;

  ///
  bool ok() const;
  
  ///
  void operator++();

  ///
  void begin();

  ///
  void end();
private:
  BitSetIterator m_iterator;
  IntVect        m_current;
  const DenseIntVectSet*    m_ivsPtr;
  int isize, ijsize, bigi, bigj;

  void nextIntVect();
  void nextIntVect(int skip);
  static DenseIntVectSet    emptyDenseIntVectSet;
};

//===================================================================

// inline functions


inline
const Box& DenseIntVectSet::box() const { return m_domain;}



inline
DenseIntVectSet& DenseIntVectSet::operator-=(const IntVect& intvect)
{
  if(m_domain.contains(intvect))
	m_bits.setFalse(m_domain.index(intvect));
  return *this;
}

inline void DenseIntVectSet::shift(const IntVect& iv)
{
  m_domain.shift(iv);
}

inline
DenseIntVectSetIterator::DenseIntVectSetIterator()
  : m_ivsPtr(&emptyDenseIntVectSet)
{
  ;
}

inline
void DenseIntVectSetIterator::define(const DenseIntVectSet& ivs)
{
  m_ivsPtr = &ivs;
  begin();
}

inline
DenseIntVectSetIterator::DenseIntVectSetIterator(const DenseIntVectSet& ivs)
  : m_ivsPtr(&ivs)
{
  begin();
}

inline
const IntVect& DenseIntVectSetIterator::operator()() const
{
  return m_current;
}

inline
bool DenseIntVectSetIterator::ok() const
{
  return m_iterator.ok();
}


inline 
void DenseIntVectSetIterator::end()
{
  m_iterator.end();
}
	
//  inline 
//  void DenseIntVectSetIterator::operator++()
//  {
//    ++m_iterator;
//    nextIntVect();
//    while(!m_iterator() && m_iterator.ok())
//      {
//        ++m_iterator; // next bit please
//        nextIntVect();
//      }
//  }
inline 
void DenseIntVectSetIterator::operator++()
{
  ++m_iterator;
  int i=1;
  while(m_iterator.ok() && !m_iterator())
    {
      ++m_iterator; // next bit please
      ++i;
    }
  nextIntVect(i);
}
	
