
#include "DenseIntVectSet.H"
#include "BoxIterator.H"
#include "MayDay.H"

DenseIntVectSet::DenseIntVectSet(const Box& a_domain, bool init)
  :
  m_domain(a_domain),
  m_bits(a_domain.numPts(), init)
{}

bool DenseIntVectSet::operator[](const IntVect& index) const
{
  if(!m_domain.contains(index)) return false;
  return m_bits[m_domain.index(index)];
}

DenseIntVectSet& DenseIntVectSet::operator-=(const Box& b)
{
  if(isEmpty()) return *this;
  if(!m_domain.intersects(b)) return *this;
  BoxIterator it(b & m_domain);
  for(it.begin(); it.ok(); ++it) this->operator-=(it());
  return *this;
}

DenseIntVectSet& DenseIntVectSet::operator&=(const Box& b)
{
  if(b.contains(m_domain)) return *this;
  if(isEmpty()) return *this;
  if(!m_domain.intersects(b))
	{
	  *this = DenseIntVectSet();
	  return *this;
	}
  BoxIterator bit(m_domain);
  int i=0;
  for(bit.begin(); bit.ok();++i, ++bit)
	{
	  if(b.contains(bit())) { }// do nothing;
	  else m_bits.setFalse(i);
	}
  return *this;
}

bool DenseIntVectSet::contains(const Box& box) const
{
  if(!m_domain.contains(box)) return false;
  for(BoxIterator bit(box); bit.ok(); ++bit)
	if(!this->operator[](bit())) return false;
  return true;
}

DenseIntVectSet& DenseIntVectSet::operator&=(const DenseIntVectSet& ivs)
{
  if(isEmpty()) return *this;
  if(ivs.isEmpty())
	{
	  *this = DenseIntVectSet();
	  return *this;
	}
  if(!m_domain.intersectsNotEmpty(ivs.m_domain))
	{
	  *this = DenseIntVectSet();
	  return *this;
	}
  BoxIterator bit(m_domain);
  int i=0;
  for(bit.begin(); bit.ok();++i, ++bit)
	{
	  if(m_bits[i])
		{
		  if(ivs[bit()]) { } // do nothing
		  else
			m_bits.setFalse(i);
		}
	}
  return *this;
}

void DenseIntVectSet::coarsen(int iref)
{
  if(iref == 1) return;
  assert(iref >= 1);
  int refinements = iref/2;
  assert(refinements*2 == iref); // check iref for power of 2

  Box newDomain(m_domain);
  newDomain.coarsen(iref);
  DenseIntVectSet newSet(newDomain, false);
  BoxIterator bit(m_domain);
  int count=0;
  for(bit.begin(); bit.ok(); ++bit, ++count)
	{
	  if(m_bits[count])
		{
		  IntVect iv(bit());
		  iv.coarsen(iref);
		  long index = newDomain.index(iv);
		  newSet.m_bits.setTrue(index);
		}
	}

  *this = newSet;
}


void DenseIntVectSet::refine(int iref)
{
  if(iref == 1) return;
  if(isEmpty()) return;
  assert(iref >= 1);
  int refinements = iref/2;
  assert(refinements*2 == iref); // check iref for power of 2 
  Box newDomain(m_domain);
  newDomain.refine(iref);
  DenseIntVectSet newSet(newDomain, false);
  IntVect iv;
  BoxIterator bit(newDomain);
  int count=0;
  for(bit.begin(); bit.ok(); ++bit, ++count)
	{
	  iv = bit();
	  iv.coarsen(iref);
	  if(this->operator[](iv))
		{
		  newSet.m_bits.setTrue(count);
		}
	}
  *this = newSet;
}

void DenseIntVectSet::nestingRegion(int radius, const Box& domain)
{
  assert(radius > 0);
  Box clobberBox(IntVect::Zero, IntVect::Zero);
  IntVect center(IntVect::Zero);
  clobberBox.grow(radius);
  
  DenseIntVectSet tmp(*this);
  BoxIterator bit(m_domain);
  int i=0;
  for(bit.begin(); bit.ok();++i, ++bit)
	{
	  if(!(tmp.m_bits[i])) // was there a zero at this bit ?
		{
		  if(domain.contains(bit()))
			 {
			   clobberBox.shift(bit()-center);
			   *this -= clobberBox;
			   center=bit();
			 }
		}
	}
}
  

void DenseIntVectSet::grow(int igrow)
{
  if(igrow >= 1)
    {
      IntVect range(IntVect::Unit);
      range*=igrow;
      grow(range);
    }
  else if(igrow < 0)
    {
      for(int idir = 0; idir < SpaceDim; idir++)
        {
          grow(idir, igrow);
        }
    }
  else
    {
      //nothing to do if igrow == 0
      return;
    }

}

void DenseIntVectSet::grow(int idir, int igrow)
{
  assert(idir >= 0);
  assert(idir < SpaceDim);
  if(igrow >= 1)
    {
      IntVect range(IntVect::Zero);
      range[idir] = igrow;
      grow(range);
    }
  else if(igrow < 0)
    {
      IntVect shiftvec= igrow*BASISV(idir);
      DenseIntVectSet ivsShiftPlus = *this;
      DenseIntVectSet ivsShiftMinu = *this;
      ivsShiftPlus.shift(shiftvec);
      ivsShiftMinu.shift(-shiftvec);
      *this &= ivsShiftPlus;
      *this &= ivsShiftMinu;
    }
  else
    {
      //nothing to do if igrow == 0
      return;
    }
}

void DenseIntVectSet::grow(const IntVect& range)
{
  Box newDomain(m_domain);
  newDomain.grow(range);
  DenseIntVectSet newSet(newDomain, false);

  int index = 0;

#if CH_SPACEDIM == 2
 
  for(int j = 0; j<m_domain.size(1); ++j)
	{
	  for(int i=0; i<m_domain.size(0); ++i, ++index)
		{
		  if(m_bits[index])
			{
			  for(int jj=0; jj<2*range[1] +1 ; ++jj)
				{
				  int indexNew = i + newDomain.size(0)*(j+jj);
				  for(int ii = 0; ii<2*range[0] + 1; ++ii, ++indexNew)
					newSet.m_bits.setTrue(indexNew);
				}
			}
		}
	}
#endif

#if CH_SPACEDIM == 3
  for(int k=0; k<m_domain.size(2); ++k)
	for(int j = 0; j<m_domain.size(1); ++j)
	  for(int i = 0; i<m_domain.size(0); ++i, ++index)
		{
		  if(m_bits[index])
			{
			   for(int kk=0; kk<2*range[2]+1; ++kk)
				 for(int jj=0; jj<2*range[1] +1 ; ++jj)
				   {
					 int indexNew = i+
					   newDomain.size(0)*(j+jj) +
					   newDomain.size(1)*newDomain.size(0)*(k+kk);
					 for(int ii = 0; ii<2*range[0] + 1; ++ii, ++indexNew)
					   newSet.m_bits.setTrue(indexNew);
				   }
			}
		}
  
#endif

  *this = newSet;
}

DenseIntVectSet& DenseIntVectSet::operator-=(const DenseIntVectSet& ivs)
{
  BoxIterator bit(m_domain);
  int i=0;
  for(bit.begin(); bit.ok();++i, ++bit)
	{
	  if(m_bits[i])
		{
		  if(ivs[bit()])
			m_bits.setFalse(i);
		}
	}
  return *this;
}

bool DenseIntVectSet::isEmpty() const
{
  return m_bits.isEmpty();
}
bool DenseIntVectSet::isFull() const
{
  return m_bits.isFull();
}

int DenseIntVectSet::numPts() const
{
  int n = m_domain.numPts();
  int c=0;
  for(int i=0; i<n; ++i)
	if(m_bits[i]) ++c;
  return c;
}

DenseIntVectSet  DenseIntVectSetIterator::emptyDenseIntVectSet;


void DenseIntVectSetIterator::nextIntVect()
{
  const IntVect& big = m_ivsPtr->m_domain.bigEnd();
  const IntVect& small = m_ivsPtr->m_domain.smallEnd();
  ++(m_current[0]);
#if CH_SPACEDIM >= 2
  if(m_current[0] > big[0])
    {
      m_current[0] = small[0];
      ++m_current[1];
    }
#endif
#if CH_SPACEDIM == 3
  if(m_current[1] > big[1])
    {
      m_current[1] = small[1];
      ++m_current[2];
    }
#endif
}

void DenseIntVectSetIterator::nextIntVect(int skip)
{
  if(skip == 1) {nextIntVect(); return;}

#if CH_SPACEDIM == 2

  int j_jump = skip / isize;
  m_current[1] += j_jump;
  m_current[0] += skip - j_jump*isize;
  if(m_current[0] > bigi)
	{
	  ++(m_current[1]);
	  m_current[0] -= isize;
	}
#endif
#if CH_SPACEDIM == 3
  
  int jump = skip / ijsize;
  m_current[2] += jump;
  skip -=   jump*ijsize;
  jump  =   skip / isize;
  m_current[1] += jump;
  skip -=   jump*isize;
  m_current[0] += skip;

  if(m_current[0] > bigi)
	{
	  ++(m_current[1]);
	  m_current[0] -= isize;
	}
  if(m_current[1] > bigj)
	{
	  ++(m_current[2]);
	  m_current[1] -= ijsize/isize;
	}
  
  // hack for now to make sure code actually works....
  //for(;skip>0; --skip) nextIntVect();

#endif

}
	  
void DenseIntVectSetIterator::begin()
{
  if(m_ivsPtr->m_bits.isEmpty())
	{
	  m_iterator.end();
	  return;
	}
  isize =  m_ivsPtr->m_domain.size(0);
  bigi  =   m_ivsPtr->m_domain.bigEnd()[0];
#if CH_SPACEDIM > 2
  ijsize = isize * m_ivsPtr->m_domain.size(1);
  bigj  =   m_ivsPtr->m_domain.bigEnd()[1];
#endif
  m_iterator = BitSetIterator(m_ivsPtr->m_bits);
  m_current = m_ivsPtr->m_domain.smallEnd();
  int i=0;
  while(m_iterator.ok() && !m_iterator())
    {
      ++i;
      ++m_iterator;
    }
  nextIntVect(i);
}

