/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
// DisjointBoxLayout.H
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
#ifndef LEVELPLAN_H
#define LEVELPLAN_H

#include "Vector.H"
#include "BoxLayout.H"

///A BoxLayout that has a concept of disjointedness
/**
   DisjointBoxLayout is-a BoxLayout that has a concept of disjointedness
   of the boxes.  DisjointBoxLayout is only different from BoxLayout in
   that it has a method isDisjoint().  
*/

class DisjointBoxLayout: public BoxLayout
{
public:

  // Constructors, destuctors, defines
  // =================================

  ///
  DisjointBoxLayout();

  ///
  /** Construct from a Vector of Boxes and optionally a Vector of
      processor assignments.  If the processor assignment Vector
      is present, it must be either zero-length or the same length
      as the Box Vector.  On exit, the DisjointBoxLayout will be closed.
  */
  DisjointBoxLayout(const Vector<Box>& a_boxes,
                    const Vector<int>& a_procIDs);

  ///
  virtual
  ~DisjointBoxLayout(){;}

  ///
  /** Shallow define. Only way to promote a BoxLayout.  If BoxLayout
      has been closed, then this method checks isDisjoint and throws an
      error if not disjoint.  If a_layout is disjoint, then this object
      becomes a closed DisjointBoxLayout object
  */
  void
  define(BoxLayout& a_layout);

  ///
  /** Define this DisjointBoxLayout from a Vector of Boxes.  This is a
      wrapper for a call to BoxLayout::define, required because
      DisjointBoxLayout::define is overloaded and the compiler will not
      look into the base class for name resolution.
  */
  virtual void
  define(const Vector<Box>& a_boxes,
         const Vector<int>& a_procIDs);



  // other member functions
  // ======================

  ///
  /** Mark this DisjointBoxLayout as complete and unchangeable.
   */
  virtual void
  close();

  ///
  /** checks that the source BoxLayout isDisjoint, throws an error
      if it is not. Otherwise, same as BoxLayout::deepCopy */
  virtual void
  deepCopy(const BoxLayout& a_source);

  ///
  /** Returns true if this object contains a disjoint
      union of Boxes. 
       The Disjoint testing algorithm assumes that the
      boxes are CELL-Centered boxes.  We can expand the algorithm
      when someone needs it.  bvs
  */
  bool
  isDisjoint() const;

};

#endif
