/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
// DisjointBoxLayout.cpp
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//


#include "Vector.H"
#include "DisjointBoxLayout.H"
#include "MayDay.H"
#include "DataIterator.H"

DisjointBoxLayout::DisjointBoxLayout()
  :BoxLayout()
{
}


DisjointBoxLayout::DisjointBoxLayout(const Vector<Box>& a_boxes,
                                     const Vector<int>& a_procIDs)
  :BoxLayout(a_boxes,a_procIDs)
{
}


void
DisjointBoxLayout::define(const Vector<Box>& a_boxes,
                          const Vector<int>& a_procIDs)
{
  BoxLayout::define(a_boxes,a_procIDs);
}


void
DisjointBoxLayout::define(BoxLayout& a_layout)
{
  ((BoxLayout*)this)->operator=(a_layout);
  if(*m_closed)
    {
      if(!isDisjoint())
        {
          MayDay::Error("define of DisjointBoxLayout from non-disjoint boxes");
        }
    }
}

void
DisjointBoxLayout::close()
{
  if(!*m_closed)  //do nothing if already closed
    {
      sort();
      if(!isDisjoint())
        {
          MayDay::Error("construction of DisjointBoxLayout from non-disjoint boxes");
        }
      *m_closed = true;
	  m_dataIterator = new DataIterator(*this, m_layout);
    }
}

void
DisjointBoxLayout::deepCopy(const BoxLayout& a_layout)
{
  BoxLayout::deepCopy(a_layout);
  if(!a_layout.isClosed())
	sort();
  if(!isDisjoint())
    {
      MayDay::Error("construction of DisjointBoxLayout from non-disjoint boxes");
    }
}

bool
DisjointBoxLayout::isDisjoint() const
{	
 
  for(unsigned int i=0; i<size(); ++i){
    for(unsigned int j=i+1; j<size(); ++j){
      bool disjoint = false;
      const Box& a = m_boxes->operator[](i).box;
      const Box& b = m_boxes->operator[](j).box;
      D_EXPR(disjoint = disjoint || a.bigEnd(0) < b.smallEnd(0) || a.smallEnd(0) > b.bigEnd(0),
             disjoint = disjoint || a.bigEnd(1) < b.smallEnd(1) || a.smallEnd(1) > b.bigEnd(1),
             disjoint = disjoint || a.bigEnd(2) < b.smallEnd(2) || a.smallEnd(2) > b.bigEnd(2));
      if(!disjoint)
        {
          return false;
        }
      if(b.smallEnd(0) > a.bigEnd(0)){
		//can skip the rest, since we know that the smallEnd of all the remaining boxes
		// are going to also return true for this test.
        j=size();
      }
    }
  }
  return true;
}
