/*
** This software is copyright (C) by the Lawrence Berkeley National
** Laboratory.  Permission is granted to reproduce this software for
** non-commercial purposes provided that this notice is left intact.
**  
** It is acknowledged that the U.S. Government has rights to this
** software under Contract DE-AC03-765F00098 between the U.S.  Department
** of Energy and the University of California.
**  
** This software is provided as a professional and academic contribution
** for joint exchange. Thus it is experimental, is provided ``as is'',
** with no warranties of any kind whatsoever, no support, no promise of
** updates, or printed documentation. By using this software, you
** acknowledge that the Lawrence Berkeley National Laboratory and Regents
** of the University of California shall have no liability with respect
** to the infringement of other copyrights by any part of this software.
**  
*/

#ifndef CH_FARRAYBOX_H
#define CH_FARRAYBOX_H

//
// $Id: FArrayBox.H,v 1.1.1.1 2000/09/05 23:55:56 graves Exp $
//

#include <iostream>

//#include <Pointers.H>
#include "Box.H"
#include "BaseFab.H"
#include "REAL.H"
#include "SPACE.H"



//@Man:
//@Memo: A Fortran Array of Reals
/*@Doc: 

  Fortran Array Boxess (generally called FABs) are objects constructed
  to interface with arrays in Fortran.  Useful operations can be performed
  upon FABs in C++, and they provide a convenient interface to
  Fortran when it is necessary to retreat into that language for
  doing arithmetic operations when performance matters.

  FArrayBox is derived from BaseFab<Real>.
  FArrayBox adds additional useful capabilities which make sense
  for Real types, such as I/O and L**p norms.

  The C pre-processor macro `CH\_SPACEDIM' must be defined to use
  this class.  The internal precision of FArrayBox objects is
  set by defining either `CH\_USE\_FLOAT' or `CH\_USE\_DOUBLE' 

  This class does NOT provide a copy constructor or assignment operator.  */

class FArrayBox
  : public BaseFab<Real>
{
public:

/// {\bf constructors and destructor}

  ///
  /**
     Constructs an invalid FArrayBox with no memory.
  */
  FArrayBox ();
  
  /*@ManDoc: Constructs an initial FArrayBox with the data space
    allocated but not inititialized. ncomp is the number of components
    (variables) at each data point in the Box.  

*/
  FArrayBox (const Box& b, int ncomp);

///
/**
   Constructs FArrayBox by reading it from istream.
*/
  explicit FArrayBox (std::istream& is);

  ///
/**
  The (virtual) destructor.
*/
  virtual ~FArrayBox ();

///{\bf norm functions}

  /*@ManDoc: Returns the Lp-norm of this FAB using components
    (comp : comp+numcomp-1).  p < 0  -> ERROR.
    p = 0  -> infinity norm (max norm).
    p = 1  -> sum of ABS(FAB)
    p > 1  -> Lp-norm
  */
  virtual Real norm (int p=2,
                     int comp=0,
                     int numcomp=1) const;

  /*@ManDoc: Returns the Lp-norm of this FAB using components (comp :
    comp+numcomp-1) and within the subbox.  p < 0 -> ERROR.  p = 0 ->
    infinity norm (max norm).  p = 1 -> sum of ABS(FAB) p > 1 ->
    Lp-norm 
*/
  virtual Real norm (const Box& subbox,
                     int        p=2,
                     int        comp=0,
                     int        numcomp=1) const;

  /**  
       Returns sum of pow(fab[i,c],p): i in subbox, comp<=c<comp+numcomp
       p>=2 only 

*/
  virtual Real sumPow(const Box& subbox,
                     int        p=2,
                     int        comp=0,
                     int        numcomp=1) const;

  ///{\bf min and max functions}

  ///
  /**
     Returns the minimum value of given component of this FArrayBox.
  */
  Real min (int comp = 0) const;

  ///
/**
   Returns the minimum value of given component of this FArrayBox in
   given subbox.  

*/
  Real min (const Box& subbox,
            int        comp = 0) const;

  ///
  /**
     Returns the maximum value of given component of this FArrayBox.
  */
  Real max (int comp = 0) const;

  ///
/**
   Returns the maximum value of given component of this FArrayBox in
   given subbox.  

*/
  Real max (const Box& subbox,
            int        comp = 0) const;

  ///
/**
  Finds location of minimum value in given component of this
  FArrayBox.  */
  IntVect minIndex (int comp = 0) const;

  /*@ManDoc: Returns location of minimum value in given component  of this FArrayBox in
    given subbox.
  */
  IntVect minIndex (const Box& subbox,
                    int        comp = 0) const;
  ///
  /**
     Returns location of maximum value in given component  of this FArrayBox.
  */
  IntVect maxIndex (int comp = 0) const;
 
  /*@ManDoc: Returns location of maximum value in given component of
    this FArrayBox in given subbox.  

*/
  IntVect maxIndex (const Box& subbox,
                    int        comp = 0) const;

///{\bf mask functions}

///
  /**
     Computes mask array with value of 1 in cells where this FArrayBox
     has value less than val, 0 otherwise.  mask is resized by this
     function.  The number of cells marked with 1 returned.  

*/
  int maskLT (BaseFab<int>& mask,
              Real             val,
              int           comp = 0) const;
 
///
  /**
     Computes mask array with value of 1 in cells where this FArrayBox
     has value less than or equal to val, 0 otherwise.  mask is
     resized by this function.  The number of cells marked with 1
     returned.  

*/
  int maskLE (BaseFab<int>& mask,
              Real             val,
              int           comp = 0) const;
///
  /**
     Computes mask array with value of 1 in cells where this FArrayBox
     has value equal to val, 0 otherwise.  mask is resized by this
     function.  The number of cells marked with 1 returned.  

*/
  int maskEQ (BaseFab<int>& mask,
              Real             val,
              int           comp = 0) const;
///
  /**
     Computes mask array with value of 1 in cells where this FArrayBox
     has value greater than val, 0 otherwise.  mask is resized by this
     function.  The number of cells marked with 1 returned.  

*/
  int maskGT (BaseFab<int>& mask,
              Real             val,
              int           comp = 0) const;
 
///
  /**
     Computes mask array with value of 1 in cells where this FArrayBox
     has value greater than or equal to val, 0 otherwise.  mask is
     resized by this function.  The number of cells marked with 1
     returned.

*/
  int maskGE (BaseFab<int>& mask,
              Real             val,
              int           comp = 0) const;


  /// {\bf Arithmetic functions}

  ///
  /**
     Modifies this FArrayBox by replacing eahc value with its absolute value.

  */
  void abs ();

  ///
  /**
     Modifies this FArrayBox by replacing eahc value with its absolute value,
     for components (comp: comp+numcomp-1).

  */
  void abs (int comp,
            int numcomp=1);

  ///
  /**
     Modifies this FArrayBox by replacing eahc value with its absolute value,
     for components (comp: comp+numcomp-1) and within the subbox.

  */
  void abs (const Box& subbox,
            int        comp = 0,
            int        numcomp=1);

///
/**
   Returns sum of given component of FArrayBox.
*/
  Real sum (int comp,
            int numcomp = 1) const;

  /*@ManDoc: Returns sum of component of this FArrayBox in given
    subbox.  

*/
  Real sum (const Box& subbox,
            int        comp,
            int        numcomp = 1) const;

  ///
/**
   Modifies this FArrayBox by replacing each value x with v/x.  For
   given range of components and within given subbox.  

*/
  FArrayBox& invert (Real          v,
                     const Box& subbox,
                     int        comp=0,
                     int        numcomp=1);
  ///
/**
   Modifies this FArrayBox by replacing each value x with v/x.  For
   given range of components.

*/
  FArrayBox& invert (Real   v,
                     int comp,
                     int numcomp=1);

  ///
/**
   Modifies this FArrayBox by replacing each value x with v/x.  

*/
  FArrayBox& invert (Real v);

  ///
/**
   Modifies this FArrayBox by replacing each value with its additive
   inverse.  For given range of components and within given subbox.  

*/
  FArrayBox& negate (const Box& subbox,
                     int        comp=0,
                     int        numcomp=1);
  ///
/**
   Modifies this FArrayBox by replacing each value with its additive
   inverse.  For given range of components.

*/
  FArrayBox& negate (int comp,
                     int numcomp=1);
  ///
/**
   Modifies this FArrayBox by replacing each value with its additive
   inverse.  

*/
  FArrayBox& negate ();

///
/**
   Modifies this FArrayBox by adding the scalar Real r to all values.  For
   given range of components and within given subbox. 

*/
  FArrayBox& plus (Real          r,
                   const Box& b,
                   int        comp=0,
                   int        numcomp=1);

///
/**
   Modifies this FArrayBox by adding the scalar Real r to all values.  For
   given range of components. 

*/
  FArrayBox& plus (Real   r,
                   int comp,
                   int numcomp=1);
///
/**
   Modifies this FArrayBox by adding the scalar Real r to all values.

*/
  FArrayBox& plus (Real r);

///
/**
   Modifies this FArrayBox by adding the scalar Real r to all values.

*/
  FArrayBox& operator+= (Real r);

///
/**
   Modifies this FArrayBox by pointwise addition of the values of the
   argument FArrayBox.  You might come to grief if the domains of the
   FArrayBoxes don't match.

*/
   FArrayBox& operator+= (const FArrayBox& f);

///
/**
   Modifies this FArrayBox by pointwise addition of the values of the
   argument FArrayBox.  You might come to grief if the domains of the
   FArrayBoxes don't match.  The same as += operator.  

*/
  FArrayBox& plus (const FArrayBox& src);

///
  /**
     Modifies this FArrayBox by pointwise scaled addition of the
     argument FArrayBox (a[i] <- a[i] + scale * src[i]).  Uses domain
     of the intersection of these two FArrayBoxes.

 */
  FArrayBox& plus (const FArrayBox& src, const Real& scale);
///
  /**
     Modifies this FArrayBox by pointwise scaled addition of the
     argument FArrayBox (a[i] <- a[i] + scale * src[i]).  Uses domain
     of the intersection of these two FArrayBoxes.
 */
  FArrayBox& plus (const FArrayBox& src, const Real& scale, 
				   int                srccomp,
                   int                destcomp,
                   int                numcomp=1);

  /*@ManDoc: Modifies this FArrayBox by pointwise addition of values
    in the argument FArrayBox.  Adds src's components
    (srccomp:srccomp+numcomp-1) to this FArrayBox's components
    (destcomp:destcomp+numcomp-1) where the domains of the two
    FArrayBoxes intersect.  

*/
  FArrayBox& plus (const FArrayBox& src,
                   int                srccomp,
                   int                destcomp,
                   int                numcomp=1);

  /*@ManDoc: Modifies this FArrayBox by pointwise addition of values
    in the argument FArrayBox.  Adds src's components
    (srccomp:srccomp+numcomp-1) to this FArrayBox's components
    (destcomp:destcomp+numcomp-1) where the domain of this FArrayBox
    intersects the subbox.  NOTE: subbox must be contained in this
    FAB.  

*/
  FArrayBox& plus (const FArrayBox& src,
                   const Box&         subbox,
                   int                srccomp,
                   int                destcomp,
                   int                numcomp=1);

  /*@ManDoc: Modifies this FArrayBox by pointwise addition of values
               in the argument FArrayBox.  Adds src's components
               (srccomp:srccomp+numcomp-1) in the Box srcbox to this
               FArrayBox's components (destcomp:destcomp+numcomp-1) in
               the Box destbox.  Corresponding locations within the
               two FArrayBoxes are indexed relative to srcbox and
               destbox, and will in general not be the same.  The
               srcbox and destbox must be same size.  The results are
               UNDEFINED if the src and dest FArrayBoxes are the same
               and the srcbox and destbox overlap.

  */
  FArrayBox& plus (const FArrayBox& src,
                   const Box&         srcbox,
                   const Box&         destbox,
                   int                srccomp,
                   int                destcomp,
                   int                numcomp=1);

///
/**
   Modifies this FArrayBox by subtracting the scalar Real r to all values.
    Note: use plus(-r) for more general operations.
  */
  FArrayBox& operator-= (Real r);

///
/**
   Modifies this FArrayBox by pointwise subtraction of the values of the
   argument FArrayBox.  You might come to grief if the domains of the
   FArrayBoxes don't match.

*/
  FArrayBox& operator-= (const FArrayBox& f);

///
/**
   Modifies this FArrayBox by pointwise subtraction of the values of the
   argument FArrayBox.  You might come to grief if the domains of the
   FArrayBoxes don't match.  The same as -= operator.

  */
  FArrayBox& minus (const FArrayBox& src);

  /*@ManDoc: Modifies this FArrayBox by pointwise subtraction of values
    in the argument FArrayBox.  Subtracts src's components
    (srccomp:srccomp+numcomp-1) from this FArrayBox's components
    (destcomp:destcomp+numcomp-1) where the domains of the two
    FArrayBoxes intersect.  

*/
  FArrayBox& minus (const FArrayBox& src,
                    int                srccomp,
                    int                destcomp,
                    int                numcomp=1);

  /*@ManDoc: Modifies this FArrayBox by pointwise subtraction of values
    in the argument FArrayBox.  Subtracts src's components
    (srccomp:srccomp+numcomp-1) from this FArrayBox's components
    (destcomp:destcomp+numcomp-1) where the domain of this FArrayBox
    intersects the subbox.  NOTE: subbox must be contained in this
    FAB.  

*/
  FArrayBox& minus (const FArrayBox& src,
                    const Box&         subbox,
                    int                srccomp,
                    int                destcomp,
                    int                numcomp=1);

  /*@ManDoc: Modifies this FArrayBox by pointwise subtraction of values
               in the argument FArrayBox.  Subtracts src's components
               (srccomp:srccomp+numcomp-1) in the Box srcbox from this
               FArrayBox's components (destcomp:destcomp+numcomp-1) in
               the Box destbox.  Corresponding locations within the
               two FArrayBoxes are indexed relative to srcbox and
               destbox, and will in general not be the same.  The
               srcbox and destbox must be same size.  The results are
               UNDEFINED if the src and dest FArrayBoxes are the same
               and the srcbox and destbox overlap.

  */
  FArrayBox& minus (const FArrayBox& src,
                    const Box&         srcbox,
                    const Box&         destbox,
                    int                srccomp,
                    int                destcomp,
                    int                numcomp=1);

///
/**
   Modifies this FArrayBox by multiplying all values by the scalar Real r.
  */
  FArrayBox& operator*= (Real r);

///
/**
   Modifies this FArrayBox by multiplying all values by the scalar Real r.
  */
  FArrayBox& mult (Real r);

///
/**
   Modifies this FArrayBox by multiplying all values by the scalar
   Real r.  For given range of components.

  */
  FArrayBox& mult (Real   r,
                   int comp,
                   int numcomp=1);

///
/**
   Modifies this FArrayBox by multiplying all values by the scalar
   Real r.  For given range of components and within given subbox.

  */
  FArrayBox& mult (Real          r,
                   const Box& b,
                   int        comp=0,
                   int        numcomp=1);

///
/**
   Modifies this FArrayBox by pointwise multiplication of the values by the
   argument FArrayBox.  You might come to grief if the domains of the
   FArrayBoxes don't match.

*/
  FArrayBox& operator*= (const FArrayBox& f);

///
/**
   Modifies this FArrayBox by pointwise multiplication by the values in the
   argument FArrayBox.  You might come to grief if the domains of the
   FArrayBoxes don't match.  The same as the *= operator.

*/
  FArrayBox& mult (const FArrayBox& src);

  /*@ManDoc: Modifies this FArrayBox by pointwise multiplication by values
    in the argument FArrayBox.  Multiplies src's components
    (srccomp:srccomp+numcomp-1) by this FArrayBox's components
    (destcomp:destcomp+numcomp-1) where the domains of the two
    FArrayBoxes intersect.  

*/
  FArrayBox& mult (const FArrayBox& src,
                   int                srccomp,
                   int                destcomp,
                   int                numcomp=1);

  /*@ManDoc: Modifies this FArrayBox by pointwise multiplication by values
    in the argument FArrayBox.  Multiplies src's components
    (srccomp:srccomp+numcomp-1) by this FArrayBox's components
    (destcomp:destcomp+numcomp-1) where the domain of this FArrayBox
    intersects the subbox.  NOTE: subbox must be contained in this
    FAB.  

*/
  FArrayBox& mult (const FArrayBox& src,
                   const Box&         subbox,
                   int                srccomp,
                   int                destcomp,
                   int                numcomp=1);

  /*@ManDoc: Modifies this FArrayBox by pointwise multiplication by values
               in the argument FArrayBox.  Multiplies src's components
               (srccomp:srccomp+numcomp-1) in the Box srcbox by this
               FArrayBox's components (destcomp:destcomp+numcomp-1) in
               the Box destbox.  Corresponding locations within the
               two FArrayBoxes are indexed relative to srcbox and
               destbox, and will in general not be the same.  The
               srcbox and destbox must be same size.  The results are
               UNDEFINED if the src and dest FArrayBoxes are the same
               and the srcbox and destbox overlap.

  */
  FArrayBox& mult (const FArrayBox& src,
                   const Box&         srcbox,
                   const Box&         destbox,
                   int                srccomp,
                   int                destcomp,
                   int                numcomp=1);

///
/**
   Modifies this FArrayBox by dividing all values by the scalar Real r.
  */
  FArrayBox& operator/= (Real r);

///
/**
   Modifies this FArrayBox by dividing all values by the scalar Real r.
  */
  FArrayBox& divide (Real r);

///
/**
   Modifies this FArrayBox by dividing all values by the scalar Real r.
   For given range of components.
*/
  FArrayBox& divide (Real   r,
                     int comp,
                     int numcomp=1);

///
/**
   Modifies this FArrayBox by dividing all values by the scalar Real
   r.  For given range of components and within given subbox.  

*/
  FArrayBox& divide (Real          r,
                     const Box& b,
                     int        comp=0,
                     int        numcomp=1);

///
/**
   Modifies this FArrayBox by pointwise division of the values by the
   argument FArrayBox.  You might come to grief if the domains of the
   FArrayBoxes don't match.

*/
  FArrayBox& operator/= (const FArrayBox& src);

///
/**
   Modifies this FArrayBox by pointwise division by the values in the
   argument FArrayBox.  You might come to grief if the domains of the
   FArrayBoxes don't match.  The same as the /= operator.

*/
  FArrayBox& divide (const FArrayBox& src);

  /*@ManDoc: Modifies this FArrayBox by pointwise division by values
    in the argument FArrayBox.  Divides this FArrayBox's components
    (destcomp:destcomp+numcomp-1) by src's components
    (srccomp:srccomp+numcomp-1) where the domains of the two
    FArrayBoxes intersect.  

*/
  FArrayBox& divide (const FArrayBox& src,
                     int                srccomp,
                     int                destcomp,
                     int                numcomp=1);

  /*@ManDoc: Modifies this FArrayBox by pointwise division by values
    in the argument FArrayBox.  Divides this FArrayBox's components
    (destcomp:destcomp+numcomp-1) by src's components
    (srccomp:srccomp+numcomp-1) where the domain of this FArrayBox
    intersects the subbox.  NOTE: subbox must be contained in this
    FAB.  

*/
  FArrayBox& divide (const FArrayBox& src,
                     const Box&         subbox,
                     int                srccomp,
                     int                destcomp,
                     int                numcomp=1);

  /*@ManDoc: Modifies this FArrayBox by pointwise division by values
               in the argument FArrayBox.  Divides this FArrayBox's
               components (destcomp:destcomp+numcomp-1) in the Box
               destbox by src's components (srccomp:srccomp+numcomp-1)
               in the Box srcbox.  Corresponding locations within the
               two FArrayBoxes are indexed relative to srcbox and
               destbox, and will in general not be the same.  The
               srcbox and destbox must be same size.  The results are
               UNDEFINED if the src and dest FArrayBoxes are the same
               and the srcbox and destbox overlap.

  */
  FArrayBox& divide (const FArrayBox& src,
                     const Box&         srcbox,
                     const Box&         destbox,
                     int                srccomp,
                     int                destcomp,
                     int                numcomp=1);

protected:

  virtual void performCopy (const BaseFab<Real>& src,
							const Box&        srcbox,
							int               srccomp,
							const Box&        destbox,
							int               destcomp,
							int               numcomp);

private:
  //
  // These are disallowed.
  //
  FArrayBox (const FArrayBox&);
  FArrayBox& operator= (const FArrayBox&);
};


#endif /*CH_FARRAYBOX_H*/
