/*
** This software is copyright (C) by the Lawrence Berkeley National
** Laboratory.  Permission is granted to reproduce this software for
** non-commercial purposes provided that this notice is left intact.
**  
** It is acknowledged that the U.S. Government has rights to this
** software under Contract DE-AC03-765F00098 between the U.S.  Department
** of Energy and the University of California.
**  
** This software is provided as a professional and academic contribution
** for joint exchange. Thus it is experimental, is provided ``as is'',
** with no warranties of any kind whatsoever, no support, no promise of
** updates, or printed documentation. By using this software, you
** acknowledge that the Lawrence Berkeley National Laboratory and Regents
** of the University of California shall have no liability with respect
** to the infringement of other copyrights by any part of this software.
**  
*/

//
// $Id: FArrayBox.cpp,v 1.2 2000/10/25 21:56:10 graves Exp $
//

#include <cstdlib>
#include <iostream>
#include <iomanip>
#include <cfloat>
#include <cmath>
#include <cstring>
using std::cin;
using std::cout;
using std::cerr;
using std::setw;
using std::setprecision;
using std::ios;
#include <cassert>

#include "Misc.H"
#include "FArrayBox.H"
#include "FArrayBox_F_F.H"
#include "MayDay.H"


FArrayBox::FArrayBox ()
  :
  BaseFab<Real>()
{
}

FArrayBox::FArrayBox (const Box& b,
                      int        n)
  :
  BaseFab<Real>(b,n)
{
}




FArrayBox::~FArrayBox () {}


Real
FArrayBox::norm (const Box& subbox,
                 int        p,
                 int        comp,
                 int        numcomp) const
{
  assert(comp >= 0 && comp+numcomp <= nComp());
  assert(p >= 0);

  Real* tmp = 0;
  int tmplen = 0;
  Real nrm = 0;
  if (p == 0)
    {
      //here  begins a normed fab function piece
      ForAllThisCPencil(Real,subbox,comp,numcomp)
        {
          const Real* row = &thisR;
          if (tmp == 0)
            {
              tmp = new Real[thisLen];
              tmplen = thisLen;
              for (int i = 0; i < thisLen; i++)
                tmp[i] = Abs(Real(row[i]));
            }
          else
            {
              for (int i = 0; i < thisLen; i++)
                tmp[i] = Max(tmp[i],Real(Abs(row[i])));
            }
        } EndForPencil
            nrm = tmp[0];
      for (int i = 1; i < tmplen; i++)
        nrm = Max(nrm, tmp[i]);
      ForAllThisCPencil(Real,subbox,comp,numcomp)
        {
          const Real* row = &thisR;
          if (tmp == 0)
            {
              tmp = new Real[thisLen];
              tmplen = thisLen;
              for (int i = 0; i < thisLen; i++)
                tmp[i] = Abs(Real(row[i]));
            }
          else
            {
              for (int i = 0; i < thisLen; i++)
                tmp[i] = Max(tmp[i],Real(Abs(row[i])));
            }
        } EndForPencil
        nrm = tmp[0];
        for (int i = 1; i < tmplen; i++)
            nrm = Max(nrm, tmp[i]);
        //here it ends
    }
  else if (p == 1)
    {
    //here begins a normed fab function piece
        ForAllThisCPencil(Real,subbox,comp,numcomp)
        {
            const Real* row = &thisR;
            if (tmp == 0)
            {
                tmp = new Real[thisLen];
                tmplen = thisLen;
                for (int i = 0; i < thisLen; i++)
                    tmp[i] = Abs(Real(row[i]));
            }
            else
            {
                for (int i = 0; i < thisLen; i++)
                    tmp[i] += Abs(Real(row[i]));
            }
        } EndForPencil
        nrm = tmp[0];
        for (int i = 1; i < tmplen; i++)
            nrm += tmp[i];
    //here it ends
    }
  else if (p == 2)
    {
      nrm = sqrt(sumPow(subbox, 2, comp, numcomp));
    }
  else //so standard norms weren't good enough for you?
    {
      Real invpwr = 1.0/p;
      nrm = pow(sumPow(subbox, p, comp, numcomp),invpwr);
    }
  delete [] tmp;
  return nrm;
}

// utility function used in norms and such
// only works for p>=2

Real FArrayBox::sumPow(const Box& subbox,
		       int        p,
		       int        comp,
		       int        numcomp) const
{
  assert(p>=2);
  Real sum = 0;
  Real* tmp = NULL;
  int tmplen = 0;
  if (p == 2)
    {
      ForAllThisCPencil(Real,subbox,comp,numcomp)
        {
          const Real* row = &thisR;
          if (tmp == 0)
            {
              tmp = new Real[thisLen];
              tmplen = thisLen;
              for (int i = 0; i < thisLen; i++)
                tmp[i] = row[i]*row[i];
            }
          else
            {
              for (int i = 0; i < thisLen; i++)
                tmp[i] += row[i]*row[i];
            }
        } EndForPencil
            sum = tmp[0];
	for (int i = 1; i < tmplen; i++)
	  sum += tmp[i];
    }
  else //so standard norms weren't good enough for you?
    {
      Real pwr = Real(p);
      ForAllThisCPencil(Real,subbox,comp,numcomp)
        {
          const Real* row = &thisR;
          if (tmp == 0)
            {
              tmp = new Real[thisLen];
              tmplen = thisLen;
              for (int i = 0; i < thisLen; i++)
                tmp[i] = pow(row[i],pwr);
            }
          else
            {
              for (int i = 0; i < thisLen; i++)
                tmp[i] += pow(row[i],pwr);
            }
        } EndForPencil
            sum = tmp[0];
	for (int i = 1; i < tmplen; i++)
	  sum += tmp[i];
 
    }
  delete [] tmp;
  return sum;
}

//here is where the orderedfab functions start


Real
FArrayBox::min (int comp) const
{
  Real *_min_row = 0;
  int _X_len = 0;
  ForAllThisCPencil(Real,domain,comp,1)
    {
      const Real* _row = &thisR;
      if (_min_row == 0)
        {
          _min_row = new Real[thisLen];
          _X_len = thisLen;
          for (int i = 0; i < thisLen; i++)
            _min_row[i] = _row[i];
        }
      else
        {
          for (int i = 0; i < thisLen; i++)
            _min_row[i] = Min(_row[i],_min_row[i]);
        }
    } EndForPencil;

  Real _min = _min_row[0];
  for (int i = 1; i < _X_len; i++)
    _min = Min(_min,_min_row[i]);

  delete [] _min_row;

  return _min;
}


Real
FArrayBox::min (const Box& subbox,
                int        comp) const
{
  Real *_min_row = 0;
  int _X_len = 0;
  ForAllThisCPencil(Real,subbox,comp,1)
    {
      const Real* _row = &thisR;
      if (_min_row == 0)
        {
          _min_row = new Real[thisLen];
          _X_len = thisLen;
          for (int i = 0; i < thisLen; i++)
            _min_row[i] = _row[i];
        }
      else
        {
          for (int i = 0; i < thisLen; i++)
            _min_row[i] = Min(_row[i],_min_row[i]);
        }
    } EndForPencil;

  Real _min = _min_row[0];
  for (int i = 1; i < _X_len; i++)
    _min = Min(_min,_min_row[i]);

  delete [] _min_row;

  return _min;
}


Real
FArrayBox::max (int comp) const
{
  Real *_max_row = 0;
  int _X_len = 0;
  ForAllThisCPencil(Real,domain,comp,1)
    {
      const Real* _row = &thisR;
      if (_max_row== 0)
        {
          _max_row = new Real[thisLen];
          _X_len = thisLen;
          for (int i = 0; i < thisLen; i++)
            _max_row[i] = _row[i];
        }
      else
        {
          for (int i = 0; i < thisLen; i++)
            _max_row[i] = Max(_row[i],_max_row[i]);
        }
    } EndForPencil;

  Real _max = _max_row[0];
  for (int i = 1; i < _X_len; i++)
    _max = Max(_max,_max_row[i]);

  delete [] _max_row;

  return _max;
}


Real
FArrayBox::max (const Box& subbox,
                int        comp) const
{
  Real *_max_row = 0;
  int _X_len = 0;
  ForAllThisCPencil(Real,subbox,comp,1)
    {
      const Real* _row = &thisR;
      if (_max_row == 0)
        {
          _max_row = new Real[thisLen];
          _X_len = thisLen;
          for (int i = 0; i < thisLen; i++)
            _max_row[i] = _row[i];
        }
      else
        {
          for (int i = 0; i < thisLen; i++)
            _max_row[i] = Max(_row[i],_max_row[i]);
        }
    } EndForPencil;

  Real _max = _max_row[0];
  for (int i = 1; i < _X_len; i++)
    _max = Max(_max,_max_row[i]);

  delete [] _max_row;

  return _max;
}


IntVect
FArrayBox::minIndex (int comp) const
{
  IntVect _min_loc(domain.smallEnd());
  Real _min_val = (*this).operator()(_min_loc,comp);
  ForAllThisCBNN(Real,domain,comp,1)
    {
      int nr = nR;
      int nr2 = nr;
      nr = nr2;
      if (thisR < _min_val)
        {
          _min_val = thisR;
          D_EXPR(_min_loc[0] = iR,
                 _min_loc[1] = jR,
                 _min_loc[2] = kR);
        }
    } EndFor;

  return _min_loc;
}


IntVect
FArrayBox::minIndex (const Box& subbox,
                     int        comp) const
{
  IntVect _min_loc(subbox.smallEnd());
  Real _min_val = (*this).operator()(_min_loc,comp);
  ForAllThisCBNN(Real,subbox,comp,1)
    {
      int nr = nR;
      int nr2 = nr;
      nr = nr2;
      if (thisR < _min_val)
        {
          _min_val = thisR;
          D_EXPR(_min_loc[0] = iR,
                 _min_loc[1] = jR,
                 _min_loc[2] = kR);
        }
    } EndFor;

  return _min_loc;
}


IntVect
FArrayBox::maxIndex (int comp) const
{
  IntVect _max_loc(domain.smallEnd());
  Real _max_val = (*this).operator()(_max_loc,comp);
  ForAllThisCBNN(Real,domain,comp,1)
    {
      int nr = nR;
      int nr2 = nr;
      nr = nr2;
      if (thisR > _max_val)
        {
          _max_val = thisR;
          D_EXPR(_max_loc[0] = iR,
                 _max_loc[1] = jR,
                 _max_loc[2] = kR);
        }
    } EndFor;

  return _max_loc;
}


IntVect
FArrayBox::maxIndex (const Box& subbox,
                     int        comp) const
{
  IntVect _max_loc(subbox.smallEnd());
  Real _max_val = (*this).operator()(_max_loc,comp);
  ForAllThisCBNN(Real,subbox,comp,1)
    {
      int nr = nR;
      int nr2 = nr;
      nr = nr2;
      if (thisR > _max_val)
        {
          _max_val = thisR;
          D_EXPR(_max_loc[0] = iR,
                 _max_loc[1] = jR,
                 _max_loc[2] = kR);
        }
    } EndFor;

  return _max_loc;
}



int
FArrayBox::maskLT (BaseFab<int>& mask,
                   Real             val,
                   int           comp) const
{
  mask.resize(domain,1);
  mask.setVal(0);

  int *mptr = mask.dataPtr();
  int cnt = 0;

  ForAllThisCBNN(Real,domain,comp,1)
    {
      int nr = nR;
      int nr2 = nr;
      nr = nr2;
      int ir = iR;
      int jr = jR;
      ir = jr;
      jr = ir;
      int ix = D_TERM(_i, +_j*_b_len[0], +_k*_b_len[0]*_b_len[1]);
      if (thisR < val)
        {
          mptr[ix] = 1;
          cnt++;
        }
    } EndFor;

  return cnt;
}


int
FArrayBox::maskLE (BaseFab<int>& mask,
                   Real             val,
                   int           comp) const
{
  mask.resize(domain,1);
  mask.setVal(0);

  int *mptr = mask.dataPtr();
  int cnt = 0;

  ForAllThisCBNN(Real,domain,comp,1)
    {
      int nr = nR;
      int nr2 = nr;
      nr = nr2;
      int ir = iR;
      int jr = jR;
      ir = jr;
      jr = ir;
      int ix = D_TERM(_i, +_j*_b_len[0], +_k*_b_len[0]*_b_len[1]);
      if (thisR <= val)
        {
          mptr[ix] = 1;
          cnt++;
        }
    } EndFor;

  return cnt;
}


int
FArrayBox::maskEQ (BaseFab<int>& mask,
                   Real             val,
                   int           comp) const
{
  mask.resize(domain,1);
  mask.setVal(0);

  int *mptr = mask.dataPtr();
  int cnt = 0;

  ForAllThisCBNN(Real,domain,comp,1)
    {
      int nr = nR;
      int nr2 = nr;
      nr = nr2;
      int ir = iR;
      int jr = jR;
      ir = jr;
      jr = ir;
      int ix = D_TERM(_i, +_j*_b_len[0], +_k*_b_len[0]*_b_len[1]);
      if (thisR == val)
        {
          mptr[ix] = 1;
          cnt++;
        }
    } EndFor;

  return cnt;
}


int
FArrayBox::maskGT (BaseFab<int>& mask,
                   Real             val,
                   int           comp) const
{
  mask.resize(domain,1);
  mask.setVal(0);

  int *mptr = mask.dataPtr();
  int cnt = 0;

  ForAllThisCBNN(Real,domain,comp,1)
    {
      int nr = nR;
      int nr2 = nr;
      nr = nr2;
      int ir = iR;
      int jr = jR;
      ir = jr;
      jr = ir;
      int ix = D_TERM(_i, +_j*_b_len[0], +_k*_b_len[0]*_b_len[1]);
      if (thisR > val)
        {
          mptr[ix] = 1;
          cnt++;
        }
    } EndFor;

  return cnt;
}


int
FArrayBox::maskGE(BaseFab<int>& mask,
                  Real             val,
                  int           comp) const
{
  mask.resize(domain,1);
  mask.setVal(0);

  int *mptr = mask.dataPtr();
  int cnt = 0;

  ForAllThisCBNN(Real,domain,comp,1)
    {
      int nr = nR;
      int nr2 = nr;
      nr = nr2;
      int ir = iR;
      int jr = jR;
      ir = jr;
      jr = ir;
      int ix = D_TERM(_i, +_j*_b_len[0], +_k*_b_len[0]*_b_len[1]);
      if (thisR >= val)
        {
          mptr[ix] = 1;
          cnt++;
        }
    } EndFor;

  return cnt;
}
//here is where the orderedfab functions end


//here is where the normedfab functions begin


void
FArrayBox::abs ()
{
  ForAllThis(Real)
    {
      thisR = Abs(thisR);
    } EndFor
        }


void
FArrayBox::abs (int comp,
                int numcomp)
{
  ForAllThisNN(Real,comp,numcomp)
    {
      thisR = Abs(thisR);
    } EndFor
        }


void
FArrayBox::abs (const Box& subbox,
                int        comp,
                int        numcomp)
{
  ForAllThisBNN(Real,subbox,comp,numcomp)
    {
      thisR = Abs(thisR);
    } EndFor
        }



Real
FArrayBox::norm (int p,
                 int comp,
                 int numcomp) const
{
  return norm(domain,p,comp,numcomp);
}


//here is where the normedfab functions end

//here is where the arithfab functions begin

FArrayBox&
FArrayBox::plus (Real r)
{
  return operator+=(r);
}


FArrayBox&
FArrayBox::plus (const FArrayBox& x)
{
  return operator+=(x);
}


FArrayBox&
FArrayBox::operator-= (Real r)
{
  return operator+=(-r);
}


FArrayBox&
FArrayBox::minus (const FArrayBox& x)
{
  return operator-=(x);
}



FArrayBox&
FArrayBox::mult (Real r)
{
  return operator*=(r);
}


FArrayBox&
FArrayBox::mult (const FArrayBox& x)
{
  return operator*=(x);
}


FArrayBox&
FArrayBox::divide (Real r)
{
  return operator/=(r);
}


FArrayBox&
FArrayBox::divide (const FArrayBox& x)
{
  return operator/=(x);
}




Real
FArrayBox::sum (int comp,
                int numcomp) const
{
  Real *_sum_row = 0;
  int _sum_len = 0;
  ForAllThisCPencil(Real,domain,comp,numcomp)
    {
      const Real* _row = &thisR;
      if (_sum_row == 0)
        {
          _sum_row = new Real[thisLen];
          _sum_len = thisLen;
          for (int i = 0; i < thisLen; i++)
            _sum_row[i] = _row[i];
        }
      else
        {
          for (int i = 0; i < thisLen; i++)
            _sum_row[i] += _row[i];
        }
    } EndForPencil;

  Real _sum = _sum_row[0];
  for (int i = 1; i < _sum_len; i++)
    _sum += _sum_row[i];
  delete [] _sum_row;

  return _sum;
}


Real
FArrayBox::sum (const Box& subbox,
                int        comp,
                int        numcomp) const
{
  Real *_sum_row = 0;
  int _sum_len = 0;
  ForAllThisCPencil(Real,subbox,comp,numcomp)
    {
      const Real* _row = &thisR;
      if (_sum_row == 0)
        {
          _sum_row = new Real[thisLen];
          _sum_len = thisLen;
          for (int i = 0; i < thisLen; i++)
            _sum_row[i] = _row[i];
        }
      else
        {
          for (int i = 0; i < thisLen; i++)
            {
              _sum_row[i] += _row[i];
            }
        }
    } EndForPencil;

  Real _sum = _sum_row[0];
  for (int i = 1; i < _sum_len; i++)
    _sum += _sum_row[i];
  delete [] _sum_row;

  return _sum;
}


FArrayBox&
FArrayBox::negate ()
{
  ForAllThis(Real)
    {
      thisR = - thisR;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::negate (int comp,
                   int numcomp)
{
  ForAllThisNN(Real,comp,numcomp)
    {
      thisR = - thisR;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::negate (const Box& b,
                   int        comp,
                   int        numcomp)
{
  ForAllThisBNN(Real,b,comp,numcomp)
    {
      thisR = - thisR;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::invert (Real r)
{
  ForAllThis(Real)
    {
      thisR = r/thisR;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::invert (Real   r,
                   int comp,
                   int numcomp)
{
  ForAllThisNN(Real,comp,numcomp)
    {
      thisR = r/thisR;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::invert (Real          r,
                   const Box& b,
                   int        comp,
                   int        numcomp)
{
  ForAllThisBNN(Real,b,comp,numcomp)
    {
      thisR = r/thisR;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::operator += (Real r)
{
  ForAllThis(Real)
    {
      thisR += r;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::plus (Real   r,
                 int comp,
                 int numcomp)
{
  ForAllThisNN(Real,comp,numcomp)
    {
      thisR += r;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::plus (Real          r,
                 const Box& b,
                 int        comp,
                 int        numcomp)
{
  ForAllThisBNN(Real,b,comp,numcomp)
    {
      thisR += r;
    } EndFor
        return *this;
}
 
// added bvs Tue May 18, PDT 1999

FArrayBox&
FArrayBox::plus(const FArrayBox& src, const Real& scale)
{
  ForAllThisBNNXC(Real, src.box()&domain, 0, nvar, src, 0)
    {
      thisR += srcR*scale;
    } EndForTX
        return *this;
}

FArrayBox&
FArrayBox::plus(const FArrayBox& src, const Real& scale,
				int                srccomp,
				int                destcomp,
				int                numcomp)
{
  ForAllThisBNNXC(Real, src.box()&domain, destcomp, numcomp, src, srccomp)
    {
      thisR += srcR*scale;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::operator += (const FArrayBox& x)
{
  ForAllThisXC(Real,x)
    {
      thisR += xR;
    } EndForTX
        return *this;
}

FArrayBox&
FArrayBox::plus (const FArrayBox& src,
                 int                srccomp,
                 int                destcomp,
                 int                numcomp)
{
  ForAllThisBNNXC(Real,domain,destcomp,numcomp,src,srccomp)
    {
      thisR += srcR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::plus (const FArrayBox& src,
                 const Box&         subbox,
                 int                srccomp,
                 int                destcomp,
                 int                numcomp)
{
  ForAllThisBNNXC(Real,subbox,destcomp,numcomp,src,srccomp)
    {
      thisR += srcR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::plus (const FArrayBox& src,
                 const Box&         srcbox,
                 const Box&         destbox,
                 int                srccomp,
                 int                destcomp,
                 int                numcomp)
{
  ForAllThisBNNXCBN(Real,destbox,destcomp,numcomp,src,srcbox,srccomp)
    {
      thisR += srcR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::operator -= (const FArrayBox& x)
{
  ForAllThisXC(Real,x)
    {
      thisR -= xR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::minus (const FArrayBox& src,
                  int                srccomp,
                  int                destcomp,
                  int                numcomp)
{
  ForAllThisBNNXC(Real,domain,destcomp,numcomp,src,srccomp)
    {
      thisR -= srcR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::minus (const FArrayBox& src,
                  const Box&         subbox,
                  int                srccomp,
                  int                destcomp,
                  int                numcomp)
{
  ForAllThisBNNXC(Real,subbox,destcomp,numcomp,src,srccomp)
    {
      thisR -= srcR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::minus (const FArrayBox& src,
                  const Box&         srcbox,
                  const Box&         destbox,
                  int                srccomp,
                  int                destcomp,
                  int                numcomp)
{
  ForAllThisBNNXCBN(Real,destbox,destcomp,numcomp,src,srcbox,srccomp)
    {
      thisR -= srcR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::operator *= (Real r)
{
  ForAllThis(Real)
    {
      thisR *= r;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::mult (Real   r,
                 int comp,
                 int numcomp)
{
  ForAllThisNN(Real,comp,numcomp)
    {
      thisR *= r;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::mult (Real          r,
                 const Box& b,
                 int        comp,
                 int        numcomp)
{
  ForAllThisBNN(Real,b,comp,numcomp)
    {
      thisR *= r;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::operator *= (const FArrayBox &x)
{
  ForAllThisXC(Real,x)
    {
      thisR *= xR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::mult (const FArrayBox& src,
                 int                srccomp,
                 int                destcomp,
                 int                numcomp)
{
  ForAllThisBNNXC(Real,domain,destcomp,numcomp,src,srccomp)
    {
      thisR *= srcR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::mult (const FArrayBox& src,
                 const Box&         subbox,
                 int                srccomp,
                 int                destcomp,
                 int                numcomp)
{
  ForAllThisBNNXC(Real,subbox,destcomp,numcomp,src,srccomp)
    {
      thisR *= srcR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::mult (const FArrayBox& src,
                 const Box&         srcbox,
                 const Box&         destbox,
                 int                srccomp,
                 int                destcomp,
                 int                numcomp)
{
  ForAllThisBNNXCBN(Real,destbox,destcomp,numcomp,src,srcbox,srccomp)
    {
      thisR *= srcR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::operator /= (Real r)
{
  ForAllThis(Real)
    {
      thisR /= r;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::divide (Real   r,
                   int comp,
                   int numcomp)
{
  ForAllThisNN(Real,comp,numcomp)
    {
      thisR /= r;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::divide (Real          r,
                   const Box& b,
                   int        comp,
                   int        numcomp)
{
  ForAllThisBNN(Real,b,comp,numcomp)
    {
      thisR /= r;
    } EndFor
        return *this;
}


FArrayBox&
FArrayBox::operator /= (const FArrayBox &x)
{
  ForAllThisXC(Real,x)
    {
      thisR /= xR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::divide (const FArrayBox& src,
                   int                srccomp,
                   int                destcomp,
                   int                numcomp)
{
  ForAllThisBNNXC(Real,domain,destcomp,numcomp,src,srccomp)
    {
      thisR /= srcR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::divide (const FArrayBox& src,
                   const Box&         subbox,
                   int                srccomp,
                   int                destcomp,
                   int                numcomp)
{
  ForAllThisBNNXC(Real,subbox,destcomp,numcomp,src,srccomp)
    {
      thisR /= srcR;
    } EndForTX
        return *this;
}


FArrayBox&
FArrayBox::divide (const FArrayBox& src,
                   const Box&         srcbox,
                   const Box&         destbox,
                   int                srccomp,
                   int                destcomp,
                   int                numcomp)
{
  ForAllThisBNNXCBN(Real,destbox,destcomp,numcomp,src,srcbox,srccomp)
    {
      thisR /= srcR;
    } EndForTX
        return *this;
}

void FArrayBox::performCopy (const BaseFab<Real>& src,
							 const Box&        srcbox,
							 int               srccomp,
							 const Box&        destbox,
							 int               destcomp,
							 int               numcomp)
{
  //BaseFab<Real>::performCopy(src, srcbox, srccomp, destbox, destcomp, numcomp);
  FArrayBox& dest = *this;
  FORT_PERFORMCOPY(CHF_FRA(dest),
				   CHF_CONST_FRA(src),
				   CHF_BOX(destbox),
				   CHF_BOX(srcbox),
				   CHF_CONST_INT(srccomp),
				   CHF_CONST_INT(destcomp),
				   CHF_CONST_INT(numcomp));
				   
}
