
/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
//  ANAG, LBNL

#ifdef HDF5  // if you don't have HDF5, then this file is useless

#ifndef HDF5_H
#define HDF5_H

#include "LevelData.H"
#include "Vector.H"
#include <hdf5.h>
// hdf5 #defines inline
#undef inline
#include <string>
#include <map>

using std::map;

class HDF5Handle;

// HDF5.H
// ============


/// user-friendly function to write out data on a AMR level
/**
    all data is IN data.
  a_handle:  handle open and ready.
  a_data:    data, what else do you want to know ?
  a_dx:      the grid spacing at this level
  a_dt:      the timestep size that was last completed
  a_time:    the time of this level (might not be the same as other levels)
  a_domain:  the problem domain, represented at this level of refinement
  a_refRatio:the refinement of a_level+1 wrt a_level. for vis systems it
               would probably help if you use 1 for a_level==max_level.
  */
template <class T>
int writeLevel(HDF5Handle& a_handle, 
			   const int& a_level, 
			   const LevelData<T>& a_data,
			   const Real& a_dx, 
			   const Real& a_dt, 
			   const Real& a_time, 
			   const Box& a_domain,
			   const int& a_refRatio);

template <class T>
int readLevel(HDF5Handle& a_handle,  
			  const int& a_level, 
			  LevelData<T>& a_data, 
			  Real& a_dx, 
			  Real& a_dt, 
			  Real& a_time, 
			  Box& a_domain,
			  int& a_refRatio, 
			  int a_comps =-1, 
			  const IntVect& ghost = IntVect::Zero, 
			  bool setGhost = false);


// More basic HDF5 functions.  These can be used at a persons own
// discretion.  Refer to the User's Guide for a description of how these
// functions interact with the convenience functions writeLevel, etc.

/// writes BoxLayout to HDF5 file.
/** 
    Writes BoxLayout to HDF5 file.  Only one BoxLayout per group is permitted, this operation overwrites
  previous entries.
  This operation assumes boxes are cell-centered.\\
  returns: success:    0\\
           HDF5 error: negative error code.\\
 */
int write(HDF5Handle& a_handle, 
		  const BoxLayout& a_layout);

/// writes a BoxLayoutData<T> to an HDF5 file.
/**
   writes a BoxLayoutData<T> to an HDF5 file.\\
  returns: success:    0\\
           HDF5 error: negative error code.\\
 */
template <class T>
int write(HDF5Handle& a_handle, 
		  const BoxLayoutData<T>& a_data, 
		  const std::string& a_name);

/// writes a LevelData<T> to an HDF5 file.
/**
   Writes a LevelData<T> to an HDF5 file.
  the DisjointBoxLayout is not written out with the data, the user is required to
  handle that object seperately. (see the "read" interface below).\\
  returns: success:    0\\
           HDF5 error: negative error code.\\
 */
template <class T>
int write(HDF5Handle& a_handle, 
		  const LevelData<T>& a_data, 
		  const std::string& a_name);

/// reads BoxLayout from location specified by a_handle.
/**  
     Reads BoxLayout from location specified by a_handle.  Only one BoxLayout per group is permitted, this operation overwrites.
  This operation assumes boxes are cell-centered.\\
  returns: success:    0\\
           HDF5 error: negative error code.\\
 */
int read(HDF5Handle& a_handle, 
		 Vector<Box>& boxes);

/// read BoxLayoutData named a_name from location specified by a_handle.
/** 
    Read BoxLayoutData named a_name from location specified by a_handle.  User must supply the correct BoxLayout for this function if redefineData == true.  \\
  returns: success:      0\\
           bad location: 1\\
           HDF5 error:   negative error code.\\
 */
template <class T>
int read(HDF5Handle& a_handle, 
		 BoxLayoutData<T>& a_data, 
		 const std::string& a_name, 
		 const BoxLayout& a_layout, 
		 bool redefineData = true);

/// read LevelData named a_name from location specified by a_handle.
/** 
    Read LevelData named a_name from location specified by a_handle.  User must supply the correct BoxLayout for this function if redefineData == true.  \\
  returns: success:      0\\
           bad location: 1\\
           HDF5 error:   negative error code.\\
 */
template <class T>
int read(HDF5Handle& a_handle, 
		 LevelData<T>& a_data, 
		 const std::string& a_name, 
		 const DisjointBoxLayout& a_layout, 
		 bool redefineData = true);

/// Handle to a particular group in an HDF file.  
/** 
    HDF5Handle is a handle to a particular group in an HDF file.  Upon
    construction, group is defined to be the root.  All data is
    written and read assuming the native representations for data on
    the architecture it is running on.  When a file is opened, these
    settings are checked and an error is flagged when things don't
    match up.  It is the USER'S responsibility to close() this object
    when it is no longer needed.

*/
class HDF5Handle
{
public:
///
/**
   Enumeration of opening modes for HDF5 files.  \\

   CREATE: file is created if it didn't exist, or an existing file of
   the same name is clobbered.\\

   OPEN_RDONLY: existing file is opened in read-only mode.  If the
   file doesn't already exist then open fails and isOpen() returns
   false.\\

   OPEN_RDWR: existing file is opened in read-write mode.  If the file
   doesn't already exist then open fails and isOpen() returns false.\\ 

*/
  enum mode {CREATE, OPEN_RDONLY, OPEN_RDWR};

///    {\bf constructor}

///
/**
  Default constructor.  User must call open() prior to using
  constructed object.

*/
  HDF5Handle();

  /// 
  /** Opens file and sets the current group to the root "/" group. \\

    if mode == CREATE, then file is created if it didn't exist, or an
                       existing file of the same name is clobbered.\\

    if mode == OPEN_*, then existing file is opened, if the file doesn't
                     already exist then open fails and isOpen() returns false.\\
 */
  HDF5Handle(const std::string& a_filename, mode a_mode);

/// {\bf File functions}

  /// 
  /** 
      Opens file and sets the current group of this HDF5Handle to the
      root "/" group.  File that this HDF5Handle previously pointed at is
      NOT closed, that is the users responsibility.\\
   
    if mode == CREATE, then file is created if it didn't exist, or an
                       existing file of the same name is clobbered.\\

    if mode == OPEN_*, then existing file is opened, if the file doesn't
                     already exist then open fails and isOpen() returns false.\\

   returns:\\
   0  on success\\
   negative number if file open failed (return code from HDF5)\\
   1  file does not appear to contain datacheck info, probably not a Chombo file\\
   2  on data bit size differences between code and file.\\
   
   aborts on SpaceDim not matching between code and file\\
   
   */
  int open(const std::string& a_filename, mode a_mode);
   
  /// 
/**
   A NULL or failed constructed HDF5Handle will return false.
*/
  bool isOpen() const;

  ///  
  /**
     Closes the file.  Must be called to close file.  Files are not
     automatically closed.  

*/
  void close();

/// {\bf Group functions}

  /// 
/**
   Sets the current group to be "/levelx" where x=a_level.
*/
  void setGroupToLevel(int a_level);
  
  /// 
  /** 
      Set group to users choice, referenced from file root.
      groupAbsPath will look like a Unix file path:
      "/mySpecialData/group1/" "/" is the root group of the
      file. returns a negative value on failure

*/
  int setGroup(const std::string& groupAbsPath);

///
/**
   Returns name of current group.  groupAbsPath will look like a Unix
   file path: "/mySpecialData/group1/" "/" is the root group of the
   file.  

*/
  const std::string& getGroup() const;

  const hid_t& fileID() const;
  const hid_t& groupID() const;
  static hid_t box_id;
  static hid_t intvect_id;

private:

  HDF5Handle(const HDF5Handle&);
  HDF5Handle& operator=(const HDF5Handle&);

  hid_t         m_fileID;
  hid_t         m_currentGroupID;
  bool          m_isOpen;
  std::string   m_filename; // keep around for debugging
  std::string   m_group;
  int           m_level;
  
  static hid_t  file_access;
  static bool   initialized;
  static void   initialize();
   
};

/// data to be added to HDF5 files.
/**
   HDF5HeaderData is a wrapper for some data maps to be added to HDF5
   files.  instead of an overdose of access functions, the maps are
   made public and they can be manipulated by the user at will.  They
   maintain type safety.

  to add a Real data entry, a user can simply program as follows:
  
  \begin{verbatim}
  Real dx;
  .
  .
  HDF5HeaderData metaData;
  metaData.m_real["dx"] = dx;
  \end{verbatim}
  
  If "dx" already existed, then it is overwritten, otherwise an entry is
  created and added with the new value;

  To search for entries, the user does the following:
  \begin{verbatim}
  
  HDF5HeaderData metaData;
  HDF5Handle currentStep(filename);
  currentStep.setGroupToLevel(0);
  metaData.readFromFile(currentStep);
  if(metaData.m_intvect.find("ghost") != metaData.m_intvect.end())
    ghost = metaData.m_intvect["ghost"];
  else
    ghost = defaultGhostIntVect;
  
  \end{verbatim}

  A user can skip the check for existence if they have reason to "know" the
  data will be there.  It is just good coding practice.

  To erase an entry, you can use:
  \begin{verbatim}
  metaData.m_real.erase("dx");
  \end{verbatim}

  */
class HDF5HeaderData
{
public:

  /// 
  /** 
      Writes this HDF5HeaderData's current attribute list to the
      current group in 'file.'  Returns 0 on success, returns the
      error code from HDF5 on failure.

*/
  int writeToFile(HDF5Handle& file) const;

  /// 
  /** 
      Reads into this HDF5HeaderData's attribute list from file.  Read
      process is add/change, does not remove key-value pairs. Reads
      from current group.  Returns 0 on success, positive number if a
      particular member of group caused an error, negative on general
      error. 

*/
  int readFromFile(HDF5Handle& file);

  ///
  void clear();

  ///
  map<std::string, Real>        m_real;

  ///
  map<std::string, int>         m_int;
  
  ///
  map<std::string, std::string> m_string;
  
  ///
  map<std::string, IntVect>     m_intvect;
  
  ///
  map<std::string, Box>         m_box;

  //users should not need these functions in general

  int writeToLocation(hid_t loc_id) const;
  int readFromLocation(hid_t loc_id);

private:
  static herr_t attributeScan(hid_t loc_id, const char *name, void *opdata);
};

extern "C"{
herr_t HDF5HeaderDataattributeScan(hid_t loc_id, const char *name, void *opdata);
	  }

std::ostream& operator<<(std::ostream& os, const HDF5HeaderData& data);


template <class T>
void dataTypes(Vector<hid_t>& a_types, const T& a_dummySpecializationArg);

template <class T>
void dataSize(const T& item, Vector<int>& a_sizes, 
		 const Box& box, const Interval& comps);

template <class T>
void write(const T& item, Vector<void*>& a_allocatedBuffers, 
	      const Box& box, const Interval& comps);

template <class T>
void read(T& item, Vector<void*>& a_allocatedBuffers, const Box& box, 
	     const Interval& comps);

template <class T>
const char* name(const T& a_dummySpecializationArg);

#include "UserHDF5Specializations.H"


//=============================================================================
//
// end of declarations.
//
//=============================================================================

#include "HDF5Implem.H"

#endif  // HDF5_H

#endif // HDF5
