/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
//  ANAG, LBNL

#ifdef HDF5

#include "HDF5.H"
#include "MayDay.H"
#include <cstdio>
//#include "LayoutIterator.H"
using std::ostream;
using std::cout;
using std::endl;


int  write(HDF5Handle& a_handle, const BoxLayout& a_layout)
{
  assert(a_layout.isClosed());
  herr_t ret;
  hssize_t offset[1];
  hsize_t  flatdims[1], count[1];
  count[0] = 1;
  flatdims[0] = a_layout.size();

  hid_t boxdataspace = H5Screate_simple(1, flatdims, NULL);

  hid_t boxdataset   = H5Dcreate(a_handle.groupID(), "boxes",  a_handle.box_id, 
				 boxdataspace, H5P_DEFAULT);
  if(boxdataset < 0) return boxdataset;
  hid_t memdataspace = H5Screate_simple(1, count, NULL);

  // write boxes in parallel
  for(DataIterator it = a_layout.dataIterator(); it.ok(); ++it)
    {
      offset[0] = a_layout.index(it());
      ret = H5Sselect_hyperslab (boxdataspace, H5S_SELECT_SET, offset, NULL, 
				 count, NULL);
      if(ret < 0) return ret;
      ret = H5Dwrite(boxdataset, a_handle.box_id, memdataspace, boxdataspace,
		     H5P_DEFAULT, &(a_layout[it()]));
      if(ret < 0) return ret;
    }
  
  H5Sclose(boxdataspace);
  H5Sclose(memdataspace);
  H5Dclose(boxdataset);
      
  return 0;
}

int read(HDF5Handle& a_handle, Vector<Box>& boxes)
{
  hid_t  boxdataset = H5Dopen(a_handle.groupID(), "boxes");
  if(boxdataset < 0) return boxdataset;
  hid_t boxdataspace =  H5Dget_space(boxdataset);
  if(boxdataspace < 0) return boxdataspace;
  hsize_t dims[1], maxdims[1];
  H5Sget_simple_extent_dims(boxdataspace, dims, maxdims);
  
  hid_t memdataspace = H5Screate_simple(1, dims, NULL);

  Box* rawboxes = new Box[dims[0]];
  if(rawboxes == NULL)
    MayDay::Error("out of memory in read(HDF5Handle& a_handle, BoxLayout& a_layout)");

  herr_t error = H5Dread(boxdataset, a_handle.box_id, memdataspace, boxdataspace,
			 H5P_DEFAULT, rawboxes);
  if(error < 0) return error;

  boxes.resize(dims[0]);
  for(int index = 0; index < dims[0]; ++index)
    {
      rawboxes[index].computeBoxLen();
      boxes[index] = rawboxes[index];
    }

  delete[] rawboxes;
  H5Dclose(boxdataset);
  H5Sclose(boxdataspace);
  H5Sclose(memdataspace);
  return 0;
}


bool HDF5Handle::initialized = false;
hid_t HDF5Handle::box_id = 0;
hid_t HDF5Handle::intvect_id = 0;
hid_t HDF5Handle::file_access = 0;

void HDF5Handle::initialize()
{
#ifdef MPI
  file_access = H5Pcreate (H5P_FILE_ACCESS);
  H5Pset_mpi(file_access,  Chombo_MPI::comm, MPI_INFO_NULL);
#else
  file_access = H5P_DEFAULT;
#endif

  /*  old composite-of-composite version of box 
  intvect_id = H5Tcreate (H5T_COMPOUND, sizeof(IntVect));
#if CH_SPACEDIM == 2
  H5Tinsert (intvect_id, "intvecti", HOFFSET(IntVect, vect[0]), H5T_NATIVE_INT);
  H5Tinsert (intvect_id, "intvectj", HOFFSET(IntVect, vect[1]), H5T_NATIVE_INT);
#endif
#if CH_SPACEDIM == 3
  H5Tinsert (intvect_id, "intvecti", HOFFSET(IntVect, vect[0]), H5T_NATIVE_INT);
  H5Tinsert (intvect_id, "intvectj", HOFFSET(IntVect, vect[1]), H5T_NATIVE_INT);
  H5Tinsert (intvect_id, "intvectk", HOFFSET(IntVect, vect[2]), H5T_NATIVE_INT);
#endif

  box_id = H5Tcreate (H5T_COMPOUND, sizeof(Box));
  H5Tinsert (box_id, "smallend", HOFFSET(Box, smallend), intvect_id);
  H5Tinsert (box_id, "bigend",   HOFFSET(Box, bigend), intvect_id);
  */

  intvect_id = H5Tcreate (H5T_COMPOUND, sizeof(IntVect));
  box_id = H5Tcreate (H5T_COMPOUND, sizeof(Box));
#if CH_SPACEDIM == 2
  H5Tinsert (intvect_id, "intvecti", HOFFSET(IntVect, vect[0]), H5T_NATIVE_INT);
  H5Tinsert (intvect_id, "intvectj", HOFFSET(IntVect, vect[1]), H5T_NATIVE_INT);
  H5Tinsert (box_id, "lo_i", HOFFSET(Box, smallend.vect[0]), H5T_NATIVE_INT);
  H5Tinsert (box_id, "lo_j", HOFFSET(Box, smallend.vect[1]), H5T_NATIVE_INT);
  H5Tinsert (box_id, "hi_i", HOFFSET(Box, bigend.vect[0]),   H5T_NATIVE_INT);
  H5Tinsert (box_id, "hi_j", HOFFSET(Box, bigend.vect[1]),   H5T_NATIVE_INT);
  
#endif
#if CH_SPACEDIM == 3
  H5Tinsert (intvect_id, "intvecti", HOFFSET(IntVect, vect[0]), H5T_NATIVE_INT);
  H5Tinsert (intvect_id, "intvectj", HOFFSET(IntVect, vect[1]), H5T_NATIVE_INT);
  H5Tinsert (intvect_id, "intvectk", HOFFSET(IntVect, vect[2]), H5T_NATIVE_INT);
  H5Tinsert (box_id, "lo_i", HOFFSET(Box, smallend.vect[0]), H5T_NATIVE_INT);
  H5Tinsert (box_id, "lo_j", HOFFSET(Box, smallend.vect[1]), H5T_NATIVE_INT);
  H5Tinsert (box_id, "lo_k", HOFFSET(Box, smallend.vect[2]), H5T_NATIVE_INT);
  H5Tinsert (box_id, "hi_i", HOFFSET(Box, bigend.vect[0]),   H5T_NATIVE_INT);
  H5Tinsert (box_id, "hi_j", HOFFSET(Box, bigend.vect[1]),   H5T_NATIVE_INT);
  H5Tinsert (box_id, "hi_k", HOFFSET(Box, bigend.vect[2]),   H5T_NATIVE_INT);
#endif


  initialized = true;
}

HDF5Handle::HDF5Handle(): m_isOpen(false)
{
  if(!initialized) initialize();
}

HDF5Handle::HDF5Handle(const std::string& a_filename, mode a_mode)
  : m_isOpen(false), m_level(-1)
{
  open(a_filename, a_mode);
}


int HDF5Handle::open(const std::string& a_filename, mode a_mode)
{
  int ret = 0;
  if(m_isOpen)
    {
      MayDay::Error("Calling 'open' on already open file.  use 'close' on finished files");
    }

  m_filename = a_filename;
  if(!initialized) initialize();
  m_group    = "/";
  switch(a_mode){
  case CREATE:
    m_fileID = H5Fcreate(a_filename.c_str(), H5F_ACC_TRUNC, H5P_DEFAULT, file_access);
    if(m_fileID < 0) return m_fileID;
    break;
  case OPEN_RDONLY:
    m_fileID = H5Fopen(a_filename.c_str(), H5F_ACC_RDONLY, file_access);
    if(m_fileID < 0) return m_fileID;
    break;
  case OPEN_RDWR:
    m_fileID = H5Fopen(a_filename.c_str(), H5F_ACC_RDWR, file_access);
    if(m_fileID < 0) return m_fileID;
    break;
  default:
    MayDay::Error("unrecognized file access mode:  HDF5Handle::open");
  }
  m_currentGroupID = H5Gopen(m_fileID, m_group.c_str());
  if(m_fileID >= 0 && m_currentGroupID >= 0) m_isOpen = true;

  // Write or read dimension checks and accuracy data

  HDF5HeaderData info;
  char buf[100];
  hid_t attr, datatype, group;
  size_t codeprecision, fileprecision;
  switch(a_mode){
  case CREATE:
    group = H5Gcreate(m_fileID, "Chombo_global", 0); 
    info.m_int["SpaceDim"] = SpaceDim;
    info.m_real["testReal"] = 0.0;
    info.writeToLocation(group);
    break;
  default:
    group =  H5Gopen(m_fileID, "Chombo_global");
    if(group < 0)
      {
	MayDay::Warning("This files appears to be missing a 'Chombo_global' section");
      }
    info.readFromLocation(group);
    if(info.m_int["SpaceDim"] == 0)
      {
	sprintf(buf, "file %s appears to lack a SpaceDim definition",a_filename.c_str());
	MayDay::Error(buf);
      }
    if(info.m_int["SpaceDim"] != SpaceDim)
      {
	MayDay::Error("SpaceDim of %s does not match code");
      }
    attr = H5Aopen_name(group, "testReal");
    if(attr < 0) return 1;
    datatype = H5Aget_type(attr);
    fileprecision = H5Tget_precision(datatype);
    codeprecision = H5Tget_precision(H5T_NATIVE_REAL);
    if(fileprecision > codeprecision) ret = 2;
    if(codeprecision > fileprecision)
      {
	sprintf(buf, "code is compiled with Real=%i bits, file %s has Real=%i bits",
		codeprecision, a_filename.c_str(), fileprecision);
	MayDay::Warning(buf);
	ret = 2;
      }
    H5Aclose(attr);
    H5Tclose(datatype);
  }
  H5Gclose(group);


  return ret;
}


bool HDF5Handle::isOpen() const{ return m_isOpen;}

void HDF5Handle::setGroupToLevel(int a_level)
{
  char ch[15];
  sprintf(ch, "level%i", a_level);
  setGroup(ch);
} 
  
int HDF5Handle::setGroup(const std::string& group)
{
  if(!m_isOpen)
    {
      MayDay::Error("cannot access group until file is successfully opened");
    }
  if(m_group == group) return 0;
  
  int ret = 0;

  ret = H5Gclose(m_currentGroupID);
  if(ret < 0)
    { 
      MayDay::Warning(" Error closing old group");
      return ret;
    }

  H5E_auto_t efunc; void* edata; // turn auto error messaging off 
  H5Eget_auto(&efunc, &edata);
  H5Eset_auto(NULL, NULL);
  m_currentGroupID = H5Gopen(m_fileID, group.c_str());
  if(m_currentGroupID < 0)
    {
      H5Eset_auto(efunc, edata); //turn error messaging back on.
      //open failed, go to group creation
      m_currentGroupID = H5Gcreate(m_fileID, group.c_str(), 0);
    }
  if(m_currentGroupID < 0)
    ret = -1;
  
  H5Eset_auto(efunc, edata); //turn error messaging back on.
  m_group = group;
  return ret;
}
void HDF5Handle::close()
{
  if(m_isOpen){
    if(!(m_currentGroupID < 0)) H5Gclose(m_currentGroupID);
    if(!(m_fileID < 0)) H5Fclose(m_fileID);
    m_isOpen = false;
  }
}



const std::string&  HDF5Handle::getGroup() const
{
  return m_group;
}
const hid_t& HDF5Handle::fileID() const {return m_fileID;}
const hid_t& HDF5Handle::groupID() const {return m_currentGroupID;}



//=====================================================================================
  /// writes the current attribute list to the current group in 'file'
int HDF5HeaderData::writeToFile(HDF5Handle& file) const
{
  return writeToLocation(file.groupID());
}

int HDF5HeaderData::writeToLocation(hid_t loc_id) const
{
  H5E_auto_t efunc; void* edata;  
  H5Eget_auto(&efunc, &edata);
        
  herr_t  ret;
  char messg[80];
#define INSERT(Ttype, mapName, H5Ttype)                                   \
  for(map<std::string, Ttype>::const_iterator p = mapName.begin();        \
      p!= mapName.end(); ++p)                                             \
    {                                                                     \
      hid_t aid  = H5Screate(H5S_SCALAR);                                 \
      H5Eset_auto(NULL, NULL);                                            \
      hid_t attr = H5Acreate(loc_id, p->first.c_str(), H5Ttype,           \
			     aid, H5P_DEFAULT);                           \
      if(attr < 0)                                                        \
	{                                                                 \
	  H5Adelete(loc_id, p->first.c_str());                            \
	  attr = H5Acreate(loc_id, p->first.c_str(), H5Ttype,             \
			     aid, H5P_DEFAULT);                           \
	  if(attr < 0)                                                    \
	    {                                                             \
	      sprintf(messg," Problem writing attribute %s",p->first.c_str());  \
	      MayDay::Warning(messg);                                     \
	    }                                                             \
	}                                                                 \
      H5Eset_auto(efunc, edata);                                          \
      Ttype tmp = p->second;                                              \
      ret = H5Awrite(attr, H5Ttype, &tmp);                                \
      if(ret < 0) return ret;                                             \
      H5Sclose(aid);                                                      \
      H5Aclose(attr);                                                     \
    }                                                                     \
    
    INSERT(Real, m_real, H5T_NATIVE_REAL);
    INSERT(int, m_int, H5T_NATIVE_INT);
    INSERT(IntVect, m_intvect, HDF5Handle::intvect_id);
    INSERT(Box, m_box, HDF5Handle::box_id);

    // string is different, of course
 
    for(map<std::string, std::string>::const_iterator p = m_string.begin(); 
	p!= m_string.end(); ++p)                                        
    {
      hid_t s_type = H5Tcopy(H5T_C_S1);
      H5Tset_size(s_type, p->second.length()); //extra requirement for strings
      hid_t aid  = H5Screate(H5S_SCALAR);
      H5Eset_auto(NULL, NULL);             
      hid_t attr = H5Acreate(loc_id, p->first.c_str(), s_type,
			     aid, H5P_DEFAULT); 
      if(attr < 0)          
	{                   
	  H5Adelete(loc_id, p->first.c_str());     
	  attr = H5Acreate(loc_id, p->first.c_str(), s_type, 
			   aid, H5P_DEFAULT);       
	  if(attr < 0)                                     
	    {                                              
	      sprintf(messg," Problem writing attribute %s",p->first.c_str()); 
	      MayDay::Warning(messg);
	    }                   
	}
      H5Eset_auto(efunc, edata);
      char* tmp = (char*)p->second.c_str();
      ret = H5Awrite(attr, s_type, tmp);           
      if(ret < 0) return ret;                                 
      H5Sclose(aid);                                           
      H5Aclose(attr);
      H5Tclose(s_type);
    }
 

    return 0;
}


  /// read process is add/change, does not remove key-value pairs. reads from current group.
int HDF5HeaderData::readFromFile(HDF5Handle& file)
{
  return H5Aiterate(file.groupID(), NULL, HDF5HeaderDataattributeScan , this);
}

int HDF5HeaderData::readFromLocation(hid_t loc_id)
{
   return H5Aiterate(loc_id, NULL, HDF5HeaderDataattributeScan , this);
}

extern "C"{
herr_t HDF5HeaderDataattributeScan(hid_t loc_id, const char *name, void *opdata)
{
  herr_t ret = 0;
  HDF5HeaderData& data = *(static_cast<HDF5HeaderData*>(opdata));

  hid_t attr   = H5Aopen_name(loc_id, name);
  hid_t atype  = H5Aget_type(attr);
  hid_t aclass = H5Tget_class(atype);
  char* buf = NULL;  size_t size = 0;

  switch(aclass){
  case H5T_INTEGER :
    int Ivalue;
    ret  = H5Aread(attr, H5T_NATIVE_INT, &Ivalue);
    if(ret < 0) break;
    data.m_int[name] = Ivalue;
    break;
  case H5T_FLOAT:
    Real Rvalue;
    ret = H5Aread(attr, H5T_NATIVE_REAL, &Rvalue);
    if(ret < 0) break;
    data.m_real[name] = Rvalue;
    break;
  case H5T_STRING:
    size = H5Tget_size(atype);
    buf = new char[size+1];
    ret = H5Aread(attr, atype, buf);
    if(ret < 0) break;
    buf[size] = 0; // for some reason HDF5 is not null terminating strings correctly
    data.m_string[name] = std::string(buf);
    break;
  case H5T_COMPOUND:
    if(strcmp(H5Tget_member_name(atype, 0), "lo_i") == 0)
      {
	Box value;
	ret = H5Aread(attr, HDF5Handle::box_id, &value);
	if(ret < 0) break;
	value.computeBoxLen();
	data.m_box[name] = value;
	break;
      }
    else if(strcmp(H5Tget_member_name(atype, 0), "intvecti") == 0)
      {
	IntVect value;
	ret = H5Aread(attr, HDF5Handle::intvect_id, &value);
	if(ret < 0) break;
	data.m_intvect[name] = value;
	break;
      }
  default:
    // heel if I know what to do about attributes I don't recognize
    MayDay::Warning("HDF5HeaderData::readFromFile encountered unrecognized attribute");
  }
  delete[] buf;
  H5Tclose(atype);
  H5Aclose(attr);
  return ret;
}
	  }
void HDF5HeaderData::clear()
{
  m_real.clear();
  m_int.clear();
  m_string.clear();
  m_intvect.clear();
  m_box.clear();
}

ostream& operator<<(ostream& os, const HDF5HeaderData& data)
{

#define PRINT(Ttype, mapName) \
  for(map<std::string, Ttype>::const_iterator p = data.mapName.begin();        \
      p!= data.mapName.end(); ++p)                   \
  os<<p->first<<" :"<<p->second<<"\n";

  PRINT(Real, m_real);
  PRINT(int, m_int);
  PRINT(std::string, m_string);
  PRINT(IntVect, m_intvect);
  PRINT(Box, m_box);
  return os;
}                        
#endif

