/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
//  ANAG, LBNL

#include "LoadBalance.H"
#include "LayoutIterator.H"
#include "Vector.H"

// First, template specializations for read/write for FArrayBox.

template <>
inline void dataTypes(Vector<hid_t>& a_types, const BaseFab<int>& dummy)
{  
  a_types.resize(1); 
  a_types[0] = H5T_NATIVE_INT;
}

/* since many compilers choke on the proper template specialization syntax
   I am forced to pass in a dummy specialization argument
   */
template <>
inline void dataTypes(Vector<hid_t>& a_types, const FArrayBox& dummy)
{  a_types.resize(1); a_types[0] = H5T_NATIVE_REAL;}


template <>
inline void dataSize(const BaseFab<int>& item, Vector<int>& a_sizes, 
	      const Box& box, const Interval& comps)
{
  a_sizes[0] = box.numPts() * comps.size();
}

template <>
inline void dataSize(const FArrayBox& item, Vector<int>& a_sizes, 
	      const Box& box, const Interval& comps)
{
  a_sizes[0] = box.numPts() * comps.size();
}

template <>
inline const char* name(const FArrayBox& a_dummySpecializationArg)
{
  static const char* name = "FArrayBox";
  return name;
}

template <>
inline const char* name(const BaseFab<int>& a_dummySpecializationArg)
{
  static const char* name = "BaseFab<int>";
  return name;
}

//
// now, generic, non-binary portable version of template functions
// for people who just want ot rely on linearIn/linearOut

template <class T>
inline void dataTypes(Vector<hid_t>& a_types, const T& dummy)
{  a_types.resize(1); a_types[0] = H5T_NATIVE_CHAR;}

template <class T>
inline void dataSize(const T& item, Vector<int>& a_sizes, 
	      const Box& box, const Interval& comps)
{
  a_sizes[0] = item.size(box, comps);
}
  

template <class T>
inline void write(const T& item, Vector<void*>& a_allocatedBuffers, 
	   const Box& box, const Interval& comps)
{
  item.linearOut(a_allocatedBuffers[0], box, comps);
}

template <class T>
inline void read(T& item, Vector<void*>& a_allocatedBuffers, 
		 const Box& box, const Interval& comps)
{
  item.linearIn(a_allocatedBuffers[0], box, comps);
}

template <class T>
inline const char* name(const T& a_dummySpecializationArg)
{
  static const char* name = "unknown";
  return name;
}

template <class T>
void getOffsets(Vector<Vector<int> >& offsets, const BoxLayoutData<T>& a_data,  
		int types, const Interval& comps)
{
  if(!T::preAllocatable())
    {
      MayDay::Error("write not yet implemented for T that isn't preAllocatable");
    }

  const BoxLayout& layout = a_data.boxLayout();

  T dummy;

  Vector<int> thisSize(types);
  offsets.resize(layout.size() + 1, Vector<int>(types));
  offsets[0] = Vector<int>(types, 0);
  unsigned int index = 1;
  for(LayoutIterator it(layout.layoutIterator()); it.ok(); ++it)
    {
      dataSize(dummy, thisSize, layout[it()], comps);
      for(unsigned int i=0; i<thisSize.size(); ++i) 
	{
	  offsets[index][i] = offsets[index-1][i] + thisSize[i];
	}
      ++index;
    }
}

//==================================================================
//
// Now, linear IO routines for a BoxLayoutData of T
//

template <class T>
int write(HDF5Handle& a_handle, const BoxLayoutData<T>& a_data, const std::string& a_name)
{
  int ret = 0;

  Interval comps = a_data.interval(); // later, we can make comps an argument.....
  T dummy;
  Vector<hid_t> types;
  dataTypes(types, dummy);

  Vector<Vector<int> > offsets;
  getOffsets(offsets, a_data, types.size(), comps);
  
  // create datasets collectively.
  hsize_t flatdims[1];
  char dataname[100];
  Vector<hid_t> dataspace(types.size());
  Vector<hid_t> dataset(types.size());

  herr_t err;
  hsize_t count[1];
  hssize_t offset[1];
  Vector<int> bufferCapacity(types.size(), 500);
  Vector<void*> buffers(types.size(), NULL);

  for(unsigned int i=0; i<types.size(); ++i) 
    {
      flatdims[0] = offsets[offsets.size()-1][i];
      sprintf(dataname, "%s:datatype=%i",a_name.c_str(), i);
      dataspace[i]      = H5Screate_simple(1, flatdims, NULL);
      dataset[i]        = H5Dcreate(a_handle.groupID(), dataname,  
				    types[i],
				    dataspace[i], H5P_DEFAULT);
    }

  // write BoxLayoutData attributes into Dataset[0]
  HDF5HeaderData info;
  info.m_int["comps"] = comps.size();
  info.m_string["objectType"] = name(dummy);
  std::string group = a_handle.getGroup();
  a_handle.setGroup(group+"/"+a_name+"_attributes");
  info.writeToFile(a_handle);
  a_handle.setGroup(group);

  // collective operations finished, now perform parallel writes
  // to specified hyperslabs.

  Vector<size_t> type_size(types.size());
  for(unsigned int i=0; i<types.size(); ++i) 
    {
      type_size[i] = H5Tget_size(types[i]);
      buffers[i] = malloc(bufferCapacity[i]);
      if(buffers[i] == NULL) MayDay::Error("memory error in buffer allocation write");
    }

  Vector<int> thisSize(types.size());
  for(DataIterator it = a_data.dataIterator(); it.ok(); ++it)
    {
      const T& data = a_data[it()];
      unsigned int index = a_data.boxLayout().index(it());
      const Box& box = a_data.box(it());
      for(unsigned int i=0; i<types.size(); ++i) 
	{
	  int size = (offsets[index+1][i] - offsets[index][i]) * type_size[i];
	  // cout << "size:"<<size<<" i:"<<i<<" bufferCapacity[i]:"<<bufferCapacity[i]<<"\n";
	  assert(size > 0);
	  if(size > bufferCapacity[i]) // grow buffer if necessary.....
	    {
	      free(buffers[i]);
	      bufferCapacity[i] = Max(2*bufferCapacity[i], size);
	      buffers[i] = malloc(bufferCapacity[i]);
	      if(buffers[i] == NULL) MayDay::Error("memory error in buffer allocation write");
	      // cout << "GROW :"<<" i:"<<i<<" bufferCapacity[i]:"<<bufferCapacity[i]<<"\n";
	    } 
	}
      write(data, buffers, box, comps);
      for(unsigned int i=0; i<types.size(); ++i) 
	{
	  offset[0] = offsets[index][i];
	  count[0] = offsets[index+1][i] - offset[0];
	  assert(count[0] > 0);
	  err =  H5Sselect_hyperslab(dataspace[i], H5S_SELECT_SET, 
				     offset, NULL, 
				     count, NULL);
	  hid_t memdataspace = H5Screate_simple(1, count, NULL);
	  err = H5Dwrite(dataset[i], types[i], memdataspace, dataspace[i],
			 H5P_DEFAULT, buffers[i]);
	  H5Sclose(memdataspace);
	  if(err < 0) { ret = err; goto cleanup;}
	}
    }

  // OK, clean up data structures

 cleanup:
  for(unsigned int i=0; i<types.size(); ++i) 
    {
      free(buffers[i]);
      H5Sclose(dataspace[i]);
      H5Dclose(dataset[i]);
    }
  return ret;

}

template <class T>
int write(HDF5Handle& a_handle, const LevelData<T>& a_data, const std::string& a_name)
{
  HDF5HeaderData info;
  info.m_intvect["ghost"] = a_data.ghostVect();
  std::string group = a_handle.getGroup();
  a_handle.setGroup(group+"/"+a_name+"_attributes");
  info.writeToFile(a_handle);
  a_handle.setGroup(group);
  return write(a_handle, (const BoxLayoutData<T>&)a_data, a_name);
}

template <class T>
int read(HDF5Handle& a_handle, LevelData<T>& a_data, const std::string& a_name, 
	 const DisjointBoxLayout& a_layout, bool a_redefineData)
{
  if(a_redefineData)
    {
      HDF5HeaderData info;
      std::string group = a_handle.getGroup();
      if(a_handle.setGroup(group+"/"+a_name+"_attributes"))
	{
	  std::string message = "error opening "+a_handle.getGroup()+"/"+a_name ;
	  MayDay::Warning(message.c_str());
	  return 1;
	}
      info.readFromFile(a_handle);
      a_handle.setGroup(group);
      int ncomp =  info.m_int["comps"];
      IntVect ghost = info.m_intvect["ghost"];
      a_data.define(a_layout, ncomp, ghost);
    }
  return read(a_handle, (BoxLayoutData<T>&)a_data, a_name, a_layout, false);
  
}
template <class T>
int read(HDF5Handle& a_handle, BoxLayoutData<T>& a_data, const std::string& a_name, 
	 const BoxLayout& a_layout, bool a_redefineData)
{
  int ret = 0; // return value;

  herr_t err;

  char dataname[100];
  hsize_t count[1];
  hssize_t offset[1];
  Vector<Vector<int> > offsets;

  T dummy;
  Vector<hid_t> types;
  dataTypes(types, dummy);
  Vector<hid_t> dataspace(types.size());
  Vector<hid_t> dataset(types.size());

  Vector<int> bufferCapacity(types.size(), 500);
  Vector<void*> buffers(types.size(), NULL);
  
  for(unsigned int i=0; i<types.size(); ++i) 
    {
      sprintf(dataname, "%s:datatype=%i",a_name.c_str(), i);
      dataset[i]        = H5Dopen(a_handle.groupID(), dataname);
      if(dataset[i] < 0) {MayDay::Warning("dataset open failure"); return dataset[i];}
      dataspace[i]      = H5Dget_space(dataset[i]);
      if(dataspace[i] < 0) {MayDay::Warning("dataspace open failure"); return dataspace[i];}
    }
  
  HDF5HeaderData info;
  std::string group = a_handle.getGroup();
  if(a_handle.setGroup(a_handle.getGroup()+"/"+a_name+"_attributes"))
    {
      std::string message = "error opening "+a_handle.getGroup()+"/"+a_name ;
      MayDay::Warning(message.c_str());
      return 1;
    }

  info.readFromFile(a_handle);
  a_handle.setGroup(group);
  int ncomps = info.m_int["comps"];
  if(ncomps <= 0){
    MayDay::Warning("ncomps <= 0 in read");
    return ncomps;
  }
  
  if(a_redefineData){
    a_data.define(a_layout, ncomps);
  }

  Interval comps(0, ncomps-1);

  getOffsets(offsets, a_data, types.size(), comps);
  
  Vector<size_t> type_size(types.size());
  for(unsigned int i=0; i<types.size(); ++i) 
    {
      type_size[i] = H5Tget_size(types[i]);
      buffers[i] = malloc(bufferCapacity[i]);
      if(buffers[i] == NULL) MayDay::Error("memory error in buffer allocation read");
    }
  
  Vector<int> thisSize(types.size());
  for(DataIterator it = a_data.dataIterator(); it.ok(); ++it)
    {
      T& data = a_data[it()];
      unsigned int index = a_data.boxLayout().index(it());
      const Box& box = a_data.box(it());
  
      for(unsigned int i=0; i<types.size(); ++i) 
	{
	  offset[0] = offsets[index][i];
	  count[0] = offsets[index+1][i] - offset[0];
	  assert(count[0] > 0);
	  int size = count[0] * type_size[i];
	  if(size > bufferCapacity[i])
	    {
	      free(buffers[i]);
	      bufferCapacity[i] = Max(2*bufferCapacity[i], size);
	      buffers[i] = malloc(bufferCapacity[i]);
	      if(buffers[i] == NULL) MayDay::Error("memory error in buffer allocation read");
	    } 

	  err =  H5Sselect_hyperslab(dataspace[i], H5S_SELECT_SET, 
				     offset, NULL, 
				     count, NULL);
	  hid_t memdataspace = H5Screate_simple(1, count, NULL);
	  err = H5Dread(dataset[i], types[i], memdataspace, dataspace[i],
			 H5P_DEFAULT, buffers[i]);
	  H5Sclose(memdataspace);
	  if(err < 0) { ret = err; goto cleanup;}
	}
      read(data, buffers,  box, comps);
    }
  
 cleanup:
  for(unsigned int i=0; i<types.size(); ++i) 
    {
      free(buffers[i]);
      H5Sclose(dataspace[i]);
      H5Dclose(dataset[i]);
    }
  return ret;
}


template <class T>
int writeLevel(HDF5Handle& a_handle, 
               const int&  a_level, 
	       const LevelData<T>& a_data,
	       const Real& a_dx, 
               const Real& a_dt, 
	       const Real& a_time, 
               const Box&  a_domain,
	       const int&  a_refRatio)
{
  int error;
  char levelName[10];
  std::string currentGroup = a_handle.getGroup();
  sprintf(levelName, "/level_%i",a_level);
  error = a_handle.setGroup(currentGroup + levelName);
  if(error != 0) return 1;

  HDF5HeaderData meta;
  meta.m_real["dx"] = a_dx;
  meta.m_real["dt"] = a_dt;
  meta.m_real["time"] = a_time;
  meta.m_box["prob_domain"] = a_domain;
  meta.m_int["ref_ratio"] = a_refRatio;

  error = meta.writeToFile(a_handle);
  if(error != 0) return 2;

  error = write(a_handle, a_data.boxLayout());
  if(error != 0) return 3;

  error = write(a_handle, a_data, "data");
  if(error != 0) return 4;

  a_handle.setGroup(currentGroup);

  return 0;
}


template <class T>
int readLevel(HDF5Handle&   a_handle,  
	      const int&    a_level, 
	      LevelData<T>& a_data, 
	      Real& a_dx, 
	      Real& a_dt, 
	      Real& a_time, 
	      Box&  a_domain,
	      int&  a_refRatio,
              int   a_comps,
	      const IntVect& ghost,
	      bool  setGhost)
{
  int error;
  char levelName[10];
  std::string currentGroup = a_handle.getGroup();
  sprintf(levelName, "/level_%i",a_level);
  error = a_handle.setGroup(currentGroup + levelName);
  if(error != 0) return 1;

  HDF5HeaderData meta;
  error = meta.readFromFile(a_handle);
  if(error != 0) return 2;
  a_dx       = meta.m_real["dx"];
  a_dt       = meta.m_real["dt"];
  a_time     = meta.m_real["time"];
  a_domain   = meta.m_box["prob_domain"];
  a_refRatio = meta.m_int["ref_ratio"];

  Vector<Box> boxes;
  error = read(a_handle, boxes);
  Vector<int> procIDs;
  LoadBalance(procIDs, boxes);
  
  DisjointBoxLayout layout(boxes, procIDs);

  layout.close();
  if(error != 0) return 3;

  if(!setGhost)
    error = read(a_handle, a_data, "data", layout);
  else
    {
      if (a_comps <= 0)
      {
        MayDay::Error ("readLevel: input comps <= 0");
      }
      a_data.define(layout, a_comps, ghost);
      error = read(a_handle, a_data, "data", layout, false);
    }
  if(error != 0) return 4;

  a_handle.setGroup(currentGroup);

  return 0;
}
