/*
** This software is copyright (C) by the Lawrence Berkeley National
** Laboratory.  Permission is granted to reproduce this software for
** non-commercial purposes provided that this notice is left intact.
**  
** It is acknowledged that the U.S. Government has rights to this
** software under Contract DE-AC03-765F00098 between the U.S.  Department
** of Energy and the University of California.
**  
** This software is provided as a professional and academic contribution
** for joint exchange. Thus it is experimental, is provided ``as is'',
** with no warranties of any kind whatsoever, no support, no promise of
** updates, or printed documentation. By using this software, you
** acknowledge that the Lawrence Berkeley National Laboratory and Regents
** of the University of California shall have no liability with respect
** to the infringement of other copyrights by any part of this software.
**  
*/

#ifndef CH_INTVECT_H
#define CH_INTVECT_H

//
// $Id: IntVect.H,v 1.1.1.1 2000/09/05 23:55:56 graves Exp $
//

#include <cstddef>
#include <cstdlib>
#include <cstring>
#include <iostream>


class HDF5Handle;

#include "SPACE.H"
#include "Misc.H"
#include <cassert>

//
//@Man:
//@Memo: An Integer Vector in SpaceDim-Dimensional Space
/*@Doc:

  The class IntVect is an implementation of an integer vector in a
  SpaceDim-dimensional space.  It represents a point in a discrete space.
  IntVect values are accessed using the operator[] function, as for a normal
  C++ array.  In addition, the basic arithmetic operators have been overloaded
  to implement scaling and translation operations.
*/

class IntVect
{
public:

/// {\bf Constructors and Accessors}

///
/**
  Construct an IntVect whose components are uninitialized.

*/
  IntVect (){;}

///
  /**
     Construct an IntVect given the specific values for its
     coordinates.  D\_DECL is a macro that sets the constructor to
     take CH\_SPACEDIM arguments.  

*/
  IntVect (D_DECL(int i, int j, int k));

///
/**
  Construct an IntVect setting the coordinates to the corresponding
  values in the integer array a.  

*/
  explicit IntVect (const int* a);

  ///
/**
  The copy constructor.

*/
  IntVect (const IntVect& rhs);

  ///
/**
   The assignment operator.

*/
  IntVect& operator= (const IntVect& rhs);

  ///
/**
   Returns a modifiable lvalue reference to the i'th coordinate of the
   IntVect.

*/
  inline
  int& operator[] (int i);

  ///
/**
   Returns the i'th coordinate of the IntVect.

*/
  inline
  int operator[] (int i) const;

  ///
/**
   Set i'th coordinate of IntVect to val.

*/
  IntVect& setVal (int i,
                   int val);

///
/**
   Returns a const pointer to an array of coordinates of the IntVect.
   Useful for arguments to FORTRAN calls.

*/
  const int* getVect () const;

/// {\bf Comparison Operators}

///
/**
  Returns true if this IntVect is equivalent to argument IntVect.  All
  comparisons between analogous components must be satisfied.  

*/
  bool operator== (const IntVect& p) const;

///
  /**
     Returns true if this IntVect is different from argument IntVect.
     All comparisons between analogous components must be satisfied.  

  */
  bool operator!= (const IntVect& p) const;

///
  /**
     Returns true if this IntVect is less than argument IntVect.  All
     comparisons between analogous components must be satisfied.  Note
     that, since the comparison is componentwise, it is possible for
     an IntVect to be neither greater than, less than, nor equal to
     another.  

  */
  bool operator< (const IntVect& p) const;

///
  /**
     Returns true if this IntVect is less than or equal to argument
     IntVect.  All comparisons between analogous components must be
     satisfied.  Note that, since the comparison is componentwise, it
     is possible for an IntVect to be neither greater than or equal
     to, less than or equal to, nor equal to another.  

  */
  bool operator<= (const IntVect& p) const;

///
  /**
     Returns true if this IntVect is greater than argument IntVect.
     All comparisons between analogous components must be satisfied.
     Note that, since the comparison is componentwise, it is possible
     for an IntVect to be neither greater than, less than, nor equal
     to another.  

  */
  bool operator> (const IntVect& p) const;

///
  /**
     Returns true if this IntVect is greater than or equal to argument
     IntVect.  All comparisons between analogous components must be
     satisfied.  Note that, since the comparison is componentwise, it
     is possible for an IntVect to be neither greater than or equal
     to, less than or equal to, nor equal to another.  

  */

  bool operator>= (const IntVect& p) const;

///
  /**
     Returns true if this IntVect is lexically less than the argument.
     An IntVect MUST BE either lexically less than, lexically greater
     than, or equal to another IntVect.\\

    iv1 is lexically less than iv2 if: \\

    in 2-D: \\
    (iv1[0] < iv2[0]) || ((iv1[0] == iv2[0]) && (iv1[1] < iv2[1])); \\

    in 3-D: \\
  (iv1[0] < iv2[0]) || (iv1[0]==iv2[0] && ((iv1[1] < iv2[1] || ((iv1[1] == iv2[1]) && (iv1[2] < iv2[2]))))); \\ 

  */
  bool lexLT (const IntVect& s) const;

///
  /**
     Returns true if this IntVect is lexically greater than the
     argument.  An IntVect MUST BE either lexically less than,
     lexically greater than, or equal to another IntVect.\\

    iv1 is lexically less than iv2 if: \\

    in 2-D: \\
    (iv1[0] > iv2[0]) || ((iv1[0] == iv2[0]) && (iv1[1] > iv2[1])); \\

    int 3-D: \\
  (iv1[0] > iv2[0]) || (iv1[0]==iv2[0] && ((iv1[1] > iv2[1] || ((iv1[1] == iv2[1]) && (iv1[2] > iv2[2]))))); \\ 

  */
  bool lexGT (const IntVect& s) const;

/// {\bf Arithmetic Operators}

  ///
/**
   Unary plus -- for completeness.

*/
  IntVect operator+ () const;

  ///
/**
   Unary Minus -- negates all components of this IntVect.

*/
  IntVect operator- () const;

/// addition

  ///
/**
  Modifies this IntVect by addition of a scalar to each component.

*/
  IntVect& operator+= (int s);

  ///
/**
   Modifies this IntVect by component-wise addition with argument.

*/
  IntVect& operator+= (const IntVect& p);

  ///
/**
   Returns component-wise sum of this IntVect and argument.

*/
  IntVect operator+ (const IntVect& p) const;

  ///
  /**
     Return an IntVect that is this IntVect with a scalar added to
     each component.

  */
  IntVect operator+ (int s) const;

  ///
/**
   Returns an IntVect that is an IntVect p with a scalar s added to
   each component.

*/
  friend inline IntVect operator+ (int            s,
                                   const IntVect& p);

/// subtraction

  ///
  /**
     Modifies this IntVect by subtraction of a scalar from each
     component.

  */
  IntVect& operator-= (int s);

  ///
/**
   Modifies this IntVect by component-wise subtraction with argument.

*/
  IntVect& operator-= (const IntVect& p);

///
/**
   Returns component-wise difference of this IntVect and argument.

*/
  IntVect operator- (const IntVect& p) const;

  ///
/**
   Returns an IntVect that is this IntVect with a scalar s subtracted
   from each component.

*/
  IntVect operator- (int s) const;
 
  ///
/**
   Returns -p + s.

*/
  friend inline IntVect operator- (int            s,
                                   const IntVect& p);

/// multiplication

  ///
/**
   Modifies this IntVect by multiplication of a scalar to each
   component.  

*/
  IntVect& operator*= (int s);

  ///
  /**
     Modifies this IntVect by component-wise multiplication with argument.

  */
  IntVect& operator*= (const IntVect& p);

  ///
  /**
     Returns component-wise product of this IntVect and argument.

  */
  IntVect operator* (const IntVect& p) const;

  ///
/**
  Returns an IntVect that is this IntVect with each component
  multiplied by a scalar.

*/
  IntVect operator* (int s) const;

  ///
/**
   Returns and IntVect that is an IntVect p with each component
   multiplied by a scalar s.

*/
  friend inline IntVect operator* (int            s,
                                   const IntVect& p);

/// division

  ///
/**
   Modifies this IntVect by division by a scalar of each component.

*/
  IntVect& operator/= (int s);

  ///
/**
  Modifies this IntVect by component-wise division with IntVect
  argument.

*/
  IntVect& operator/= (const IntVect& p);

///
/**
  Returns component-wise division of this IntVect by IntVect argument.

*/
  IntVect operator/ (const IntVect& p) const;

  ///
/**
  Returns component-wise division of this IntVect by scalar s.

*/
  IntVect operator/ (int s) const;

/// other

  ///
/**
   Modifies this IntVect by taking component-wise min with IntVect
   argument.

*/
  IntVect& min (const IntVect& p);

///
/**
  Returns the IntVect that is the component-wise minimum of two
  argument IntVects.  

*/
  friend inline IntVect min (const IntVect& p1,
                             const IntVect& p2);

  ///
/**
   Modifies this IntVect by taking component-wise max with //IntVect
   argument.

*/
  IntVect& max (const IntVect& p);

///
  /**
     Returns the IntVect that is the component-wise minimum of two
     argument IntVects.  

  */
  friend inline IntVect max (const IntVect& p1,
                             const IntVect& p2);

  ///
/**
  Modifies this IntVect by multiplying each component by a scalar.

*/
  IntVect& scale (int s);

///
  /**
     Returns an IntVect obtained by multiplying each of the components
     of the given IntVect by a scalar.  

  */
  friend inline IntVect scale (const IntVect& p,
                               int            s);

///
  /**
     Modifies IntVect by reflecting it in the plane defined by the
     index ref\_ix and with normal in the direction of idir.
     Directions are zero-based.  

  */
  IntVect& reflect (int ref_ix,
                    int idir);

///
  /**
     Returns an IntVect that is the reflection of the given IntVect in
     the plane which passes through ref\_ix and normal to the
     coordinate direction idir.  

  */
  friend inline IntVect reflect(const IntVect& a,
                                int            ref_ix,
                                int            idir);
  
///
/**
   Modifies this IntVect by adding s to component in given coordinate
   direction.

*/
  IntVect& shift (int coord,
                  int s);

  ///
/**
   Modifies this IntVect by component-wise addition with IntVect
   argument.

*/
  IntVect& shift (const IntVect& iv);

  ///
/**
   Modifies this IntVect by adding a scalar s to each component.

*/
  IntVect& diagShift (int s);

///
/**
   Returns IntVect obtained by adding a scalar to each of the
   components of the given IntVect.  

*/
  friend inline IntVect diagShift (const IntVect& p,
                                   int            s);

  ///
/**
   Modify IntVect by component-wise integer projection.

*/
  IntVect& coarsen (const IntVect& p);

  ///
/**
   Modify IntVect by component-wise integer projection.

*/
  IntVect& coarsen (int p);

  ///
/**
   Returns an IntVect that is the component-wise integer projection of
   p by s.  

*/
  friend inline IntVect coarsen (const IntVect& p,
                                 int            s);

///
  /**
     Returns an IntVect which is the component-wise integer projection
     of IntVect p1 by IntVect p2.  

  */
  friend inline IntVect coarsen (const IntVect& p1,
                                 const IntVect& p2);


/// {\bf I/O Functions}

  ///
/**
   Print an IntVect to the ostream.

*/
  void printOn (std::ostream& os) const;

  ///
/**
   Print an IntVect to the ostream a bit more verbosely.

*/
  void dumpOn (std::ostream& os) const;

  ///
/**
   Print the IntVect to given output stream in ASCII.

*/
  friend std::ostream& operator<< (std::ostream&       os,
                              const IntVect& iv);

  ///
/**
   Read next IntVect from given input stream.

*/
  friend std::istream& operator>> (std::istream& os,
                              IntVect& iv);

/// {\bf IntVect Constants}

///
/**
   Returns a basis vector in the given coordinate direction; eg.
   IntVect3d BASISV(1) == (0,1,0).  Note that the coordinate
   directions are zero based.  

*/
  friend inline IntVect BASISV (int dir);

///
/**
   This is an IntVect all of whose components are equal to zero.

*/
  static const IntVect Zero;

///
/**
   This is an IntVect all of whose components are equal to one.

*/
  static const IntVect Unit;

// TheZeroVector is out of favor.  use IntVect::Zero.
  static const IntVect& TheZeroVector () {return Zero;}

// TheUnitoVector is out of favor.  use IntVect::Unit.
  static const IntVect& TheUnitVector () {return Unit;}

protected:
  //
  // Box is a friend of ours.
  //
  friend class Box;

  friend class HDF5Handle;

  int vect[SpaceDim];

  static const size_t IntVectSize;
};

//
// Inlines.
//

// try uninitialized IntVect null construction for now.....

// inline
// IntVect::IntVect ()
// {
//     D_EXPR(vect[0] = 0, vect[1] = 0, vect[2] = 0);
// }

inline
IntVect::IntVect (D_DECL(int i, int j, int k))
{
  D_EXPR(vect[0] = i, vect[1] = j, vect[2] = k);
}

inline
IntVect::IntVect (const int *a)
{
  D_EXPR(vect[0] = a[0], vect[1] = a[1], vect[2] = a[2]);
}

inline
IntVect::IntVect (const IntVect &iv)
{
  //D_EXPR(vect[0]=iv.vect[0], vect[1]=iv.vect[1], vect[2]=iv.vect[2]);
  memcpy(vect, iv.vect, IntVectSize);
}

inline
IntVect&
IntVect::operator= (const IntVect &iv)
{
  D_EXPR(vect[0]=iv.vect[0], vect[1]=iv.vect[1], vect[2]=iv.vect[2]);
  return *this;
}

inline
int&
IntVect::operator[] (int i)
{
  assert(i>=0 && i < SpaceDim);
  return vect[i];
}

inline
int
IntVect::operator[] (int i) const
{
  assert(i>=0 && i < SpaceDim);
  return vect[i];
}

inline
IntVect&
IntVect::setVal (int i,
                 int val)
{
  assert(i >=0 && i < SpaceDim);
  vect[i] = val;
  return *this;
}

inline
const int*
IntVect::getVect () const
{
  return vect;
}

inline
bool
IntVect::operator== (const IntVect& p) const
{
  return D_TERM(vect[0] == p[0], && vect[1] == p[1], && vect[2] == p[2]);
}

inline
bool
IntVect::operator!= (const IntVect& p) const
{
  return D_TERM(vect[0] != p[0], || vect[1] != p[1], || vect[2] != p[2]);
}

inline
bool
IntVect::operator< (const IntVect& p) const
{
  return D_TERM(vect[0] < p[0], && vect[1] < p[1], && vect[2] < p[2]);
}

inline
bool
IntVect::operator<= (const IntVect& p) const
{
  return D_TERM(vect[0] <= p[0], && vect[1] <= p[1], && vect[2] <= p[2]);
}

inline
bool
IntVect::operator> (const IntVect& p) const
{
  return D_TERM(vect[0] > p[0], && vect[1] > p[1], && vect[2] > p[2]);
}

inline
bool
IntVect::operator>= (const IntVect& p) const
{
  return D_TERM(vect[0] >= p[0], && vect[1] >= p[1], && vect[2] >= p[2]);
}

inline
bool
IntVect::lexLT (const IntVect &s) const
{
#define LLT0 (vect[0] < s[0])
#define LLT1 ((vect[0] == s[0]) && (vect[1] < s[1]))
#define LLT2 ((vect[1] == s[1]) && (vect[2] < s[2]))
#if   CH_SPACEDIM == 1
  return LLT0;
#elif CH_SPACEDIM == 2
  return LLT0 || LLT1;
#elif CH_SPACEDIM == 3
  return LLT0 || (vect[0]==s[0] && ((vect[1] < s[1] || LLT2)));
#endif
#undef LLT0
#undef LLT1
#undef LLT2
}

inline
bool
IntVect::lexGT (const IntVect& s) const
{
#define LGT0 (vect[0] > s[0])
#define LGT1 ((vect[0] == s[0]) && (vect[1] > s[1]))
#define LGT2 ((vect[1] == s[1]) && (vect[2] > s[2]))
#if   CH_SPACEDIM == 1
  return LGT0;
#elif CH_SPACEDIM == 2
  return LGT0 || LGT1;
#elif CH_SPACEDIM == 3
  return LGT0 || (vect[0] == s[0] && ((vect[1] > s[1] || LGT2)));
#endif
#undef LGT0
#undef LGT1
#undef LGT2
}

inline
IntVect
IntVect::operator+ () const
{
  return IntVect(*this);
}

inline
IntVect
IntVect::operator- () const
{
  return IntVect(D_DECL(-vect[0], -vect[1], -vect[2] ));
}

inline
IntVect&
IntVect::operator+= (int s)
{
  D_EXPR(vect[0] += s, vect[1] += s, vect[2] += s);
  return *this;
}

inline
IntVect&
IntVect::operator+= (const IntVect& p)
{
  D_EXPR(vect[0] += p[0], vect[1] += p[1], vect[2] += p[2]);
  return *this;
}

inline
IntVect&
IntVect::operator*= (int s)
{
  D_EXPR(vect[0] *= s, vect[1] *= s, vect[2] *= s);
  return *this;
}

inline
IntVect&
IntVect::operator*= (const IntVect &p)
{
  D_EXPR(vect[0] *= p[0], vect[1] *= p[1], vect[2] *= p[2]);
  return *this;
}

inline
IntVect&
IntVect::operator/= (int s)
{
  D_EXPR(vect[0] /= s, vect[1] /= s, vect[2] /= s);
  return *this;
}

inline
IntVect&
IntVect::operator/= (const IntVect& p)
{
  D_EXPR(vect[0] /= p[0], vect[1] /= p[1], vect[2] /= p[2]);
  return *this;
}

inline
IntVect&
IntVect::operator-= (int s)
{
  D_EXPR(vect[0] -= s, vect[1] -= s, vect[2] -= s);
  return *this;
}

inline
IntVect&
IntVect::operator-= (const IntVect& p)
{
  D_EXPR(vect[0] -= p[0], vect[1] -= p[1], vect[2] -= p[2]);
  return *this;
}

inline
IntVect
IntVect::operator+ (const IntVect& p) const
{
  return IntVect(D_DECL(vect[0] + p[0], vect[1] + p[1], vect[2] + p[2]));
}

inline
IntVect
IntVect::operator+ (int s) const
{
  return IntVect(D_DECL(vect[0] + s, vect[1] + s, vect[2] + s));
}

inline
IntVect
IntVect::operator- (const IntVect& p) const
{
  return IntVect(D_DECL(vect[0] - p[0], vect[1] - p[1], vect[2] - p[2]));
}

inline
IntVect
IntVect::operator- (int s) const
{
  return IntVect(D_DECL(vect[0] - s, vect[1] - s, vect[2] - s));
}

inline
IntVect
IntVect::operator* (const IntVect& p) const
{
  return IntVect(D_DECL(vect[0] * p[0], vect[1] * p[1], vect[2] * p[2]));
}

inline
IntVect
IntVect::operator* (int s) const
{
  return IntVect(D_DECL(vect[0] * s, vect[1] * s, vect[2] * s));
}

inline
IntVect
IntVect::operator/ (const IntVect& p) const
{
  return IntVect(D_DECL(vect[0] / p[0], vect[1] / p[1], vect[2] / p[2]));
}

inline
IntVect
IntVect::operator/ (int s) const
{
  return IntVect(D_DECL(vect[0] / s, vect[1] / s, vect[2] / s));
}

inline
IntVect&
IntVect::min (const IntVect& p)
{
  D_EXPR(vect[0] = Min(vect[0], p.vect[0]),
         vect[1] = Min(vect[1], p.vect[1]),
         vect[2] = Min(vect[2], p.vect[2]));
  return *this;
}

inline
IntVect&
IntVect::max (const IntVect& p)
{
  D_EXPR(vect[0] = Max(vect[0], p.vect[0]),
         vect[1] = Max(vect[1], p.vect[1]),
         vect[2] = Max(vect[2], p.vect[2]));
  return *this;
}

inline
IntVect&
IntVect::scale (int s)
{
  D_EXPR(vect[0] *= s, vect[1] *= s, vect[2] *= s);
  return *this;
}

inline
IntVect&
IntVect::reflect (int ref_ix,
                  int idir)
{
  assert(idir >= 0 && idir < SpaceDim);
  vect[idir] = -vect[idir] + 2*ref_ix;
  return *this;
}

inline
IntVect&
IntVect::shift (int coord,
                int s)
{
  assert(coord >= 0 && coord < SpaceDim);
  vect[coord] += s;
  return *this;
}

inline
IntVect&
IntVect::shift (const IntVect& iv)
{
  *this += iv;
  return *this;
}

inline
IntVect&
IntVect::diagShift (int s)
{
  D_EXPR(vect[0] += s, vect[1] += s, vect[2] += s);
  return *this;
}

inline
IntVect
operator+ (int            s,
           const IntVect& p)
{
  return IntVect(D_DECL(p[0] + s, p[1] + s, p[2] + s));
}

inline
IntVect
operator- (int            s,
           const IntVect& p)
{
  return IntVect(D_DECL(s - p[0], s - p[1], s - p[2]));
}

inline
IntVect
operator* (int            s,
           const IntVect& p)
{
  return IntVect(D_DECL(s * p[0], s * p[1], s * p[2]));
}

inline
IntVect
scale (const IntVect& p,
       int            s)
{
  return IntVect(D_DECL(s * p[0], s * p[1], s * p[2]));
}

inline
IntVect
diagShift (const IntVect &p, int s)
{
  return IntVect(D_DECL(p[0] + s, p[1] + s, p[2] + s));
}

inline
IntVect
min (const IntVect& p1,
     const IntVect& p2)
{
  IntVect p(p1);
  return p.min(p2);
}

inline
IntVect
max (const IntVect& p1,
     const IntVect& p2)
{
  IntVect p(p1);
  return p.max(p2);
}

inline
IntVect
BASISV (int dir)
{
  assert(dir >= 0 && dir < SpaceDim);
  IntVect tmp = IntVect::TheZeroVector() ;
  tmp.vect[dir] = 1;
  return tmp;
}

inline
IntVect
reflect (const IntVect& a,
         int            ref_ix,
         int            idir)
{
  assert(idir >= 0 && idir < SpaceDim);
  IntVect b(a);
  b.vect[idir] = -b.vect[idir] + 2*ref_ix;
  return b;
}

inline
IntVect
coarsen (const IntVect& p,
         int            s)
{
  assert(s > 0);
  return IntVect(
                 D_DECL((p.vect[0]<0) ? -abs(p.vect[0]+1)/s-1 : p.vect[0]/s ,
                        (p.vect[1]<0) ? -abs(p.vect[1]+1)/s-1 : p.vect[1]/s ,
                        (p.vect[2]<0) ? -abs(p.vect[2]+1)/s-1 : p.vect[2]/s ));
}

inline
IntVect
coarsen (const IntVect& p1,
         const IntVect& p2)
{
  assert(p2 > IntVect::TheZeroVector());
  return IntVect(
                 D_DECL(
                        (p1.vect[0]<0)?-abs(p1.vect[0]+1)/p2.vect[0]-1:p1.vect[0]/p2.vect[0],
                        (p1.vect[1]<0)?-abs(p1.vect[1]+1)/p2.vect[1]-1:p1.vect[1]/p2.vect[1],
                        (p1.vect[2]<0)?-abs(p1.vect[2]+1)/p2.vect[2]-1:p1.vect[2]/p2.vect[2])
                 );
}

inline
IntVect&
IntVect::coarsen (int s)
{
  assert(s > 0);
  for (int i = 0; i < SpaceDim; ++i)
    vect[i] = ((vect[i]<0) ? -abs(vect[i]+1)/s-1 : vect[i]/s);
  return *this;
}

inline
IntVect&
IntVect::coarsen (const IntVect& p)
{
  assert(p > IntVect::TheZeroVector());
  for (int i = 0; i <SpaceDim; ++i)
    {
      const int s = p.vect[i];
      vect[i] = ((vect[i]<0) ? -abs(vect[i]+1)/s-1 : vect[i]/s);
    }
  return *this;
}

#endif /*CH_INTVECT_H*/

