// -*- Mode: C++; c-file-style: "gnu"; Modified: "Thu 20 Jan 2000 14:04:09 "; -*- 
// file: IntVectSet.H

/* 
     _______              __
    / ___/ /  ___  __ _  / /  ___
   / /__/ _ \/ _ \/  ' \/ _ \/ _ \
   \___/_//_/\___/_/_/_/_.__/\___/ 
*/

//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//

#ifndef _INTVECTSET_H_
#define _INTVECTSET_H_
#include <iostream>
#include "Box.H"
#include "IntVect.H"
#include "TreeIntVectSet.H"
#include "DenseIntVectSet.H"

/// An irregular domain on an integer lattice
/**
   IntVectSet represents an irregular region in an integer lattice
   SpaceDim-dimensional index space as an arbitrary collection of
   IntVects.  A full set calculus is defined.  Any IntVect or
   cell-centered Box can be fully represented as an IntVectSet.  There
   is an iterator that provides access to the contents of an
   IntVectSet.  IntVectSets are implicitly cell-centered.
   Intersection, union, and complement operations are defined for
   pairs of IntVectSets (and, by extension, an IntVectSet and an
   IntVect or an IntVectSet and a Box).  The minimum Box of an
   IntVectSet is defined as the smallest Box that contains every
   IntVect in the IntVectSet.

   The IntVects in an IntVectSet do not have a canonical ordering.
*/
class IntVectSet
{
public:
  friend
  class IVSIterator;
  /// {\bf Constructors, Destructor, and defines}

  ~IntVectSet();
  /// 
  /** default constructor: 
      defines an empty IntVectSet. */
  IntVectSet();
  void
  define();

  /// 
  /** copy constructor:
      defines this to be a copy of ige_in. */
  IntVectSet(const IntVectSet& ige_in);
  void
  define(const IntVectSet& ige_in);

  /// 
  /** IntVect constructor:
      defines this to be an IntVectSet with just one IntVect. */
  explicit
  IntVectSet(const IntVect& iv_in);
  void define(const IntVect& iv_in);

  /// 
  /** Box constructor:
      defines this to be an IntVectSet with all the cells in a Box. */
  explicit
  IntVectSet(const Box& b);
  void
  define(const Box& b);


  /// {\bf union operators}

  ///
  /** 
      Returns the union of two IntVectSets. \\
      {\bf Arguments:} \\
      ivs1, ivs2 (not modified): \\
      {\bf Returns:} \\
      The union of the two IntVectSets ivs1 and ivs2.\\
  */
  friend
  IntVectSet 
  operator|(const IntVectSet& ivs1, const IntVectSet& ivs2);

  ///
  /** 
      Returns the union of an IntVectSet and an IntVect. \\
      {\bf Arguments:} \\
      ivs (not modified) \\
      iv (not modified): \\
      {\bf Returns:} \\
      The union of the IntVectSet ivs and the IntVect iv.\\
  */
  friend
  IntVectSet 
  operator|(const IntVectSet& ivs, const IntVect& iv);

  ///
  /** 
      Returns the union of an IntVectSet and an IntVect. \\
      {\bf Arguments:} \\
      ivs (not modified) \\
      iv (not modified): \\
      {\bf Returns:} \\
      The union of the IntVectSet ivs and the IntVect iv.\\
  */
  friend
  IntVectSet 
  operator|(const IntVect& iv, const IntVectSet& ivs);

  ///
  /** 
      Returns the union of an IntVectSet and a Box. \\
      {\bf Arguments:} \\
      ivs (not midified): \\
      b (not modified): \\
      {\bf Returns:} \\
      The union of the IntVectSet ivs and the Box b.\\
  */
  friend
  IntVectSet 
  operator|(const IntVectSet& ivs, const Box& b);

  ///
  /** 
      Returns the union of an IntVectSet and a Box. \\
      {\bf Arguments:} \\
      ivs (not midified): \\
      b (not modified): \\
      {\bf Returns:} \\
      The union of the IntVectSet ivs and the Box b.\\
  */
  friend
  IntVectSet 
  operator|(const Box& b, const IntVectSet& ivs);


  ///
  /** 
      Sets this IntVectSet to its union with another IntVectSet. \\
      {\bf Arguments:} \\
      ivs (not modified): \\
      {\bf Returns:} \\
      The union of the two IntVectSets.\\
      {\bf This:} \\
      ---The object is modified---
  */
  IntVectSet&
  operator|=(const IntVectSet& ivs);


  ///
  /** 
      Sets this IntVectSet to its union with an IntVect. \\
      {\bf Arguments:} \\
      iv (not modified): \\
      {\bf Returns:} \\
      The union of this IntVectSet and the IntVect.\\
      {\bf This:} \\
      ---The object is modified---
  */
  IntVectSet&
  operator|=(const IntVect& iv);


  ///
  /** 
      Sets this IntVectSet to its union with a Box. \\
      {\bf Arguments:} \\
      box (not modified): \\
      {\bf Returns:} \\
      The union of this IntVect and the Box.\\
      {\bf This:} \\
      ---The object is modified---
  */
  IntVectSet&
  operator|=(const Box& b);



  /// {\bf complement operators}

  /// 
  /** 
      Return the complement of the argument IntVectSet within this IntVectSet.
   */
  IntVectSet
  operator-(const IntVectSet& ivs) const;

  /// 
  /** 
      Return the complement of the argument Box within this IntVectSet.
   */
  IntVectSet
  operator-(const Box& b) const;

  /// 
  IntVectSet 
  /** 
      Return the complement of the argument IntVect within this IntVectSet.
   */
  operator-(const IntVect& iv) const;

  /// 
  /**
     Modifies this IntVectSet to be the complement of the argument
     IntVectSet within this IntVectSet.  \\
      {\bf This:} \\
      ---This object is modified.---
  */
  IntVectSet&
  operator-=(const IntVectSet& ivs);

  /// 
  /**
     Modifies this IntVectSet to be the complement of the argument
     Box within this IntVectSet.  \\
      {\bf This:} \\
      ---This object is modified.---
  */
  IntVectSet&
  operator-=(const Box& b);

  /// 
  /**
     Modifies this IntVectSet to be the complement of the argument
     IntVect within this IntVectSet.\\
      {\bf This:} \\
      ---This object is modified.---  
  */
  IntVectSet&
  operator-=(const IntVect& iv);


  /// {\bf Intersection operators}

  /// 
  /** 
      Returns the intersection of two IntVectSets.   The result may be empty.\\
      {\bf Arguments:} \\
      ivs1, ivs2 (not modified): \\
      {\bf Returns:} \\
      The intersection of the two IntVectSets ivs1 and ivs2.\\
  */
  friend 
  IntVectSet
  operator&(const IntVectSet& ivs1, const IntVectSet& ivs2);

  /// 
  /** 
      Returns the intersection of an IntVectSet and a Box. \\
      {\bf Arguments:} \\
      ivs (not modified): \\
      {\bf Returns:} \\
      The intersection of the IntVectSet and the Box.\\
  */
  friend
  IntVectSet 
  operator&(const IntVectSet& ivs, const Box& b);

  /// 
  /** 
      Returns the intersection of an IntVectSet and a Box. \\
      {\bf Arguments:} \\
      ivs (not modified): \\
      {\bf Returns:} \\
      The intersection of the IntVectSet and the Box.\\
  */
  friend
  IntVectSet 
  operator&(const Box& b, const IntVectSet& ivs);

  /// 
  /** 
      Sets this IntVectSet to its intersection with another IntVectSet. \\
      {\bf Arguments:} \\
      ivs (not modified): \\
      {\bf Returns:} \\
      The intersection of the two IntVectSets.\\
      {\bf This:} \\
      ---This object is modified.---
  */
  IntVectSet&
  operator&=(const IntVectSet& ivs);


  /// 
  /** 
      Modifies this IntVectSet to be its intersection with a Box. \\
      {\bf Arguments:} \\
      box (not modified): \\
      {\bf Returns:} \\
      The intersection of the this IntVectSet and the Box.\\
      {\bf This:} \\
      ---This object is modified.---
  */
  IntVectSet&
  operator&=(const Box& b);


  /// {\bf modification functions}

  ///
  /** 
      Returns the argument IntVectSet grown it by the specified number
      of cells in all directions, including diagonal.  \\
      {\bf Arguments:} \\
      ivs (not modified): \\
      igrow (not midified): number of cells to grow.\\
      {\bf Returns:} \\
      The argument IntVectSet grown by igrow in all directions.\\
*/
  friend
  IntVectSet
  grow(const IntVectSet& ivs, int igrow);

  ///
  /** 
      Modifies this IntVectSet by growing it by the specified number
      of cells in all directions, including diagonal.  It is an error
      if igrow < 0. \\
      {\bf Arguments:} \\
      igrow (not modified): number of cells to grow.\\
      {\bf Returns:} \\
      This IntVectSet grown by igrow in all directions.\\ 
      {\bf This:} \\
      ---This object is modified.---
*/
  IntVectSet&
  grow(int igrow);

  ///
  /** 
      Modifies this IntVectSet by growing it by the specified number
      of cells in the specified coordinate direction.  Directions are
      zero-based.\\
      {\bf Arguments:} \\
      idir (not modified): zero-based coordinate direction to grow in.\\
      igrow (not modified): number of cells to grow.\\
      {\bf Returns:} \\
      This IntVectSet grown by igrow the direction idir.\\ 
      {\bf This:} \\
      ---This object is modified.---
*/
  IntVectSet&
  grow(int idir, int igrow);

  /// 
  /**
     Modifies this IntVectSet by refining it by a specified factor
     iref.  It is an error if iref <= 0.  Definition of refinement:
     For each IntVect iv in the original IntVectSet, the refined
     IntVectSet will contain the Box refine(Box(iv,iv),iref).\\
      {\bf This:} \\
      ---This object is modified.---

   */
  IntVectSet&
  refine(int iref = 2);

  /// 
  /**
     Returns the IntVectSet that is the argument IntVectSet refined by
     a specified factor iref.  It is an error if iref <= 0.
     Definition of refinement: For each IntVect iv in the original
     IntVectSet, the refined IntVectSet will contain the Box
     refine(Box(iv,iv),iref).

   */
  friend
  IntVectSet
  refine(const IntVectSet& ivs, int iref = 2);


  /// 
  /**
     Modifies this IntVectSet by coarsening it by a specified factor
     iref.  It is an error if iref <= 0.  Definition of coarsening:
     For each IntVect iv in the original IntVectSet, the refined
     IntVectSet will contain the IntVect coarsen(iv,iref).\\
      {\bf This:} \\
      ---This object is modified.---

   */
  IntVectSet&
  coarsen(int iref = 2);

  /// 
  /**
     Returns the IntVectSet that is the argument IntVectSet coarsened
     by a specified factor iref.  It is an error if iref <= 0.
     Definition of coarsening: For each IntVect iv in the original
     IntVectSet, the refined IntVectSet will contain the IntVect
     coarsen(iv,iref).

   */
  friend
  IntVectSet
  coarsen(const IntVectSet& ivs, int iref = 2);

  ///
  /** 
      Increment all the IntVects in this IntVectSet by the given iv.\\
      --this IS changed--
   */
  void
  shift(const IntVect& iv);

  ///
  /**
	 remove all IntVects that are not at least 'radius' from the the
	 edge of the IntVectSet in any direction.  'radius' must be positive
	 IntVects that border the box 'domain' are spared from this trimming
  */
  void
  nestingRegion(int radius, const Box& domain);

  ///
  /** sets this IntVectSet to be empty */
  void
  makeEmpty();

  ///
  /** Chop the IntVectSet  at the chop_pnt in the dir direction \\
      Returns one IntVectSet and  modifies the object IntVectSet. \\
      The union of the two is the original IntVectSet. \\
      The modified IntVectSet is the low end, the returned IntvectSet \\
      is the high end. */
  IntVectSet
  chop(int dir, int chop_pnt);


  /// {\bf accessor and inquiry functions}

  ///
  /** Returns number of IntVects in this IntVectSet. */
  int
  numPts() const;

  ///
  /** Returns minimum enclosing box of this IntVectSet. */
  const Box&
  minBox() const;

  ///
  /** Returns true if no cells are in this IntVectSet. */
  bool
  isEmpty() const;

  ///
  /** Does this IntVectSet contain the argument IntVect? */
  bool
  contains(const IntVect& iv) const;

  ///
  /** is every IntVect in this 'box' contained in the IntvectSet.
	  note: this is not performed in a crude point-wise fashion
	  and is faster than a user doing it pointwise themselves */
  bool
  contains(const Box& box) const;

  ///
  /**
     Returns Vector<Box> representation of this IntVectSet.
  */
  Vector<Box>
  boxes() const;

  ///
  /**
     Write Vector<Box> representation to an output stream.
  */
  void
  printBoxes(std::ostream& a_ostrm) const;

  friend
  std::ostream& operator<<(std::ostream& os, const IntVectSet& ivs);

private:

  void convert() const; // turn dense rep into Tree rep.  very costly. 
                        // it is 'logically' const, but does modify data structures;
  bool m_isdense;
  TreeIntVectSet m_ivs;
  DenseIntVectSet m_dense;

};


///Iterator for an IntVectSet
/**
   IVSIterator iterates over every point (IntVect)
   in an IntVectSet.  It has exactly the same
   syntax and sematic as BoxIterator.  Typical usage:\\

\begin{verbatim}
   IntVectSet ivs;
   ...
   IVSIterator ivsit (ivs); 
   for (ivsit.begin(); ivsit.ok(); ++ivsit)  
   {  
     IntVect iv = ivsit(); 
     (do operations involving iv) 
   } 
\end{verbatim}
*/
class IVSIterator
{
public:
  ///
  IVSIterator():m_isdense(true) {;}
  
  ///
  IVSIterator(const IntVectSet& ivs);

  ///
  void define(const IntVectSet& ivs);

  ///
  const IntVect& operator()() const ;

  ///
  bool ok() const;

  ///
  void operator++();

  ///
  void begin();

  ///
  void reset();

  ///
  void end();
private:
  bool m_isdense;
  DenseIntVectSetIterator m_dense;
  TreeIntVectSetIterator  m_tree;
};


inline const IntVect& IVSIterator::operator()() const
{
  if(m_isdense) return m_dense();
  return m_tree();
}

inline bool IVSIterator::ok() const
{
  if(m_isdense) return m_dense.ok();
  return m_tree.ok();
}

inline void  IVSIterator::operator++()
{
  if(m_isdense) ++m_dense;
  else          ++m_tree;
}
inline void IVSIterator::reset()
{
  begin();
}

inline void IVSIterator::begin()
{
  if(m_isdense) m_dense.begin();
  else          m_tree.begin();
}

inline void IVSIterator::end()
{
  if(m_isdense) m_dense.end();
  else          m_tree.end();
}


inline IntVectSet::IntVectSet(): m_isdense(true) {;}

inline void IntVectSet::define(const IntVectSet& ige_in)
{
  *this = ige_in;
}

inline IntVectSet::IntVectSet(const IntVectSet& ige_in)
{
  *this = ige_in;
}

#endif
