/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/


// IntVectSet.H
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
// ============

#include "IntVectSet.H"

IntVectSet::~IntVectSet(){;}

void IntVectSet::define()
{
  m_ivs.clear();
  m_dense = DenseIntVectSet();
  m_isdense = true;
}


IntVectSet::IntVectSet(const IntVect& iv_in)
{
  define(iv_in);
}
  
void IntVectSet::define(const IntVect& iv_in)
{
  define(Box(iv_in, iv_in));
}

IntVectSet::IntVectSet(const Box& b)
{
  define(b);
}
 
void IntVectSet::define(const Box& b)
{
  m_ivs.clear();
  m_isdense = true;
  m_dense = DenseIntVectSet(b);
}

IntVectSet operator|(const IntVectSet& ivs1, const IntVectSet& ivs2)
{
  IntVectSet rtn = ivs1;
  rtn |= ivs2;
  return rtn;
}

IntVectSet operator|(const IntVectSet& ivs, const IntVect& iv)
{
  IntVectSet rtn = ivs;
  rtn |= iv;
  return rtn;
}

IntVectSet operator|(const IntVect& iv, const IntVectSet& ivs)
{
  IntVectSet rtn = ivs;
  rtn |= iv;
  return rtn;
}

IntVectSet operator|(const IntVectSet& ivs, const Box& b)
{
  IntVectSet rtn = ivs;
  rtn |= b;
  return rtn;
}

IntVectSet operator|(const Box& b, const IntVectSet& ivs)
{
  IntVectSet rtn = ivs;
  rtn |= b;
  return rtn;
}

IntVectSet& IntVectSet::operator|=(const IntVect& ivs)
{
  if(m_isdense) convert();
  m_ivs |= ivs;
  return *this;
}

IntVectSet& IntVectSet::operator|=(const Box& b)
{
  if(m_isdense) convert();
  m_ivs |= b;
  return *this;
}


IntVectSet& IntVectSet::operator|=(const IntVectSet& ivs)
{
  if(m_isdense) convert();
  if(ivs.m_isdense) ivs.convert();
  m_ivs |= ivs.m_ivs;
  return *this;
}

IntVectSet IntVectSet::operator-(const IntVectSet& ivs) const
{
  IntVectSet rtn = *this;
  rtn -= ivs;
  return rtn;
}

IntVectSet IntVectSet::operator-(const IntVect& iv) const
{
  IntVectSet rtn = *this;
  rtn -= iv;
  return rtn;
}

IntVectSet IntVectSet::operator-(const Box& b) const
{
  IntVectSet rtn = *this;
  rtn -= b;
  return rtn;
}

IntVectSet& IntVectSet::operator-=(const IntVectSet& ivs)
{
  if(!(minBox().intersects(ivs.minBox()))) return *this;
  if(m_isdense)
	{
	  if(ivs.m_isdense) m_dense-=ivs.m_dense;
	  else
		{
		  for(TreeIntVectSetIterator it(ivs.m_ivs); it.ok(); ++it) m_dense -= it();
		}
	}
  else
	{
	  if(ivs.m_isdense)
		{
		  for(DenseIntVectSetIterator it(ivs.m_dense);it.ok(); ++it) m_ivs -= it();
		}
	  else
		m_ivs -= ivs.m_ivs;
	}
  
  return *this;
}

IntVectSet& IntVectSet::operator-=(const IntVect& iv)
{
  if(m_isdense) m_dense -= iv;
  else          m_ivs   -= iv;
  return *this;
}

IntVectSet& IntVectSet::operator-=(const Box& b)
{
  if(m_isdense) m_dense -= b;
  else          m_ivs   -= b;
  return *this;
}


IntVectSet operator&(const IntVectSet& ivs1, const IntVectSet& ivs2)
{
  IntVectSet rtn = ivs1;
  rtn &= ivs2;
  return rtn;
}



IntVectSet operator&(const IntVectSet& ivs, const Box& b)
{
  IntVectSet rtn = ivs;
  rtn &= b;
  return rtn;
}

IntVectSet operator&(const Box& b, const IntVectSet& ivs)
{
  IntVectSet rtn = ivs;
  rtn &= b;
  return rtn;
}

IntVectSet& IntVectSet::operator&=(const Box& b)
{
  if(m_isdense) m_dense &= b;
  else          m_ivs   &= b;
  return *this;
}

IntVectSet& IntVectSet::operator&=(const IntVectSet& ivs)
{
  if(!(minBox().intersects(ivs.minBox())))
	{
	  m_ivs.clear();
	  m_isdense = true;
	  m_dense = DenseIntVectSet();
	  return *this;
	}
  if(m_isdense)
	{
	  if(ivs.m_isdense) m_dense&=ivs.m_dense;
	  else
		{
		  convert();
		  m_ivs &= ivs.m_ivs;
		}
	}
  else
	{
	  if(ivs.m_isdense) ivs.convert();
		m_ivs &= ivs.m_ivs;
	}
  
  return *this;
}

IntVectSet grow(const IntVectSet& ivs, int igrow)
{
  IntVectSet rtn(ivs);
  rtn.grow(igrow);
  return rtn;
}

IntVectSet& IntVectSet::grow(int igrow)
{
  if(m_isdense) m_dense.grow(igrow);
  else          m_ivs.grow(igrow);
  return *this;
}

void IntVectSet::nestingRegion(int radius, const Box& domain)
{
  if(m_isdense) m_dense.nestingRegion(radius, domain);
  else          m_ivs.nestingRegion(radius, domain);
}

IntVectSet& IntVectSet::grow(int idir, int igrow)
{
  assert(idir >= 0);
  assert(idir < SpaceDim);
  if(m_isdense) m_dense.grow(idir, igrow);
  else          m_ivs.grow(idir, igrow);
  return *this;
}

IntVectSet refine(const IntVectSet& ivs, int iref)
{
  IntVectSet rtn(ivs);
  rtn.refine(iref);
  return rtn;
}

IntVectSet& IntVectSet::refine(int iref)
{
  if(m_isdense) m_dense.refine(iref);
  else          m_ivs.refine(iref);
  return *this;
}

IntVectSet coarsen(const IntVectSet& ivs, int iref)
{
  IntVectSet rtn(ivs);
  rtn.coarsen(iref);
  return rtn;
}

IntVectSet& IntVectSet::coarsen(int iref)
{
  if(m_isdense) m_dense.coarsen(iref);
  else          m_ivs.coarsen(iref);
  return *this;
}

void IntVectSet::shift(const IntVect& iv)
{
  if(m_isdense) m_dense.shift(iv);
  else          m_ivs.shift(iv);
}

void IntVectSet::makeEmpty()
{
  if(m_isdense) m_dense = DenseIntVectSet();
  else          m_ivs.clear();
}

IntVectSet IntVectSet::chop(int dir, int chop_pnt)
{
  if(m_isdense) convert();
  IntVectSet rtn;
  rtn.m_isdense = false;
  rtn.m_ivs = m_ivs.chop(dir, chop_pnt);
  return rtn;
}

const Box& IntVectSet::minBox() const
{
  if(m_isdense) return m_dense.box();
  return m_ivs.minBox();
}

bool IntVectSet::isEmpty() const
{
  if(m_isdense) return m_dense.isEmpty();
  return m_ivs.isEmpty();
}

int IntVectSet::numPts() const
{
  if(m_isdense) return m_dense.numPts();
  return m_ivs.numPts();
}

bool IntVectSet::contains(const IntVect& iv) const
{
  if(m_isdense) return m_dense[iv];
  return m_ivs.contains(iv);
}

bool IntVectSet::contains(const Box& box) const
{
  if(m_isdense) return m_dense.contains(box);
  return m_ivs.contains(box);
}

Vector<Box> IntVectSet::boxes() const
{
  if(m_isdense) convert();
  return m_ivs.createBoxes();
}

void IntVectSet::printBoxes(std::ostream& a_ostrm) const
{
  Vector<Box> b = boxes();
  for(int i=0; i<b.size(); ++i) a_ostrm<<b[i]<<std::endl;
}

std::ostream& operator<<(std::ostream& os, const IntVectSet& ivs)
{
  ivs.printBoxes(os);
  return os;
}

void IntVectSet::convert() const
{
  if(m_dense.isEmpty()) { } // do nothing
  else if(m_dense.isFull())
	{
	  ((TreeIntVectSet&)m_ivs).define(m_dense.box());
	}
  else
	{
	  DenseIntVectSetIterator it(m_dense);
	  for(it.begin(); it.ok(); ++it)
		{
		  ((TreeIntVectSet&)m_ivs) |= it();
		}
	  m_ivs.compact();
	}
  (bool&)m_isdense = false;
}


//====================================================================

IVSIterator::IVSIterator(const IntVectSet& ivs)
{
  define(ivs);
}

void IVSIterator::define(const IntVectSet& ivs)
{
  if(ivs.m_isdense)
	{
	  m_isdense = true;
	  m_dense.define(ivs.m_dense);
	}
  else
	{
	  m_isdense = false;
	  m_tree.define(ivs.m_ivs);
	}
}
