// -*- Mode: C++; c-file-style: "gnu"; Modified: "Thu 27 Jan 2000 13:16:15 by dave"; -*-
// file: IntVectSet_old.cpp

//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
// DTGraves


#include <cstdlib>

#include "IntVectSet_old.H"
#include "MayDay.H"

//
// Constructors and 'define's (grouped by arglist)
//

IntVectSet_old::IntVectSet_old()
{
    tree_ptr = NULL;
    define();
}

void
IntVectSet_old::define()
{
    if(tree_ptr != NULL) delete tree_ptr;
    tree_ptr = new Tree();
}


IntVectSet_old::IntVectSet_old(const IntVect& iv_in)
{
    tree_ptr = NULL;
    define( iv_in );
}

void
IntVectSet_old::define(const IntVect& iv_in)
{
    if(tree_ptr != NULL) delete tree_ptr;
    tree_ptr = new Tree();
    (*this) |= iv_in;
}


IntVectSet_old::IntVectSet_old(const Box&  box_in)
{
    tree_ptr = NULL;
    define( box_in );
}

void
IntVectSet_old::define(const Box&  box_in)
{
    if(tree_ptr != NULL) delete tree_ptr;
    tree_ptr = new Tree();
    (*this) |= box_in;
}


IntVectSet_old::IntVectSet_old(const IntVectSet_old& ige_in)
{
    tree_ptr = NULL;
    define( ige_in );
}

void
IntVectSet_old::define(const IntVectSet_old& ige_in)
{
    if(tree_ptr != NULL) delete tree_ptr;
    tree_ptr = new Tree();
    (*this) = ige_in;
}


//
// Destructor
//
    
IntVectSet_old::~IntVectSet_old()
{
    if(tree_ptr != NULL) delete tree_ptr;
}


//
// Operators
//

IntVectSet_old& 
IntVectSet_old::operator=(const IntVectSet_old& ige_in)
{
    if(&ige_in != this)
    {
        Box invbox;
        if(tree_ptr != NULL) delete tree_ptr;
        tree_ptr = new Tree();
        minbox = invbox;
        (*this) |= ige_in;
    }
    return *this;
}

IntVectSet_old &
IntVectSet_old::operator|=(const IntVectSet_old& ige_in)
{
    if (!ige_in.isEmpty())
    {
        if (!isEmpty())
        {
            minbox = minbox.minBox(ige_in.minbox);
        }
        else
        {
            minbox = ige_in.minbox;
        }

        // Now add boxes of ige_in to tree.

        LeafIterator lt_in(ige_in.tree_ptr,Leaf::Regular);

        for (lt_in.begin(); lt_in.ok(); lt_in.next())
        {
            Box newbox = lt_in.getDomain();
            tree_ptr->addBox(newbox, Leaf::Regular);
        }
    }
    //if ige_in is empty.  nothing to add.
    return *this;
}


IntVectSet_old& 
IntVectSet_old::operator|=(const IntVect& iv_in)
{
    Box addbox(iv_in, iv_in);
    (*this) |= addbox;
    return *this;
}


IntVectSet_old& 
IntVectSet_old::operator|=(const Box& box_in)
{
    if(box_in.isEmpty()) return *this;

    if (!isEmpty())
    {
        minbox.minBox(box_in);
    }
    else
    {
        minbox = box_in;
    }
    tree_ptr->addBox(box_in, Leaf::Regular);
    return *this;
}


IntVectSet_old 
operator|(const IntVectSet_old& ivs, const Box& box_in) 
{
    IntVectSet_old outivs = ivs;
    outivs |= box_in;
    return outivs;
}


IntVectSet_old 
operator|(const Box& box_in, const IntVectSet_old& ivs) 
{
    IntVectSet_old outivs = ivs;
    outivs |= box_in;
    return outivs;
}


IntVectSet_old 
operator|(const IntVectSet_old& ivs, const IntVect& iv_in) 
{
    IntVectSet_old outivs = ivs;
    Box addbox(iv_in, iv_in);
    outivs |= addbox;
    return outivs;
}


IntVectSet_old 
operator|(const IntVect& iv_in, const IntVectSet_old& ivs) 
{
    IntVectSet_old outivs = ivs;
    Box addbox(iv_in, iv_in);
    outivs |= addbox;
    return outivs;
}


IntVectSet_old 
operator|(const IntVectSet_old& ivs1,
          const IntVectSet_old& ivs2)
{
    IntVectSet_old outivs = ivs1;
    outivs |= ivs2;
    return outivs;
}


IntVectSet_old& 
IntVectSet_old::operator&=(const IntVectSet_old& ige_in)
{
    if((!isEmpty()) && !(ige_in.isEmpty()))
    {
        // Neither IVS is empty. Compute intersection by using:
        // A & B = A - (A-B).
        IntVectSet_old ivtemp = *this - ige_in;  // A - B
        *this -= ivtemp;                     // A - (A-B)
    }
    else
    {
        //one of the two IVS's is empty.  
        //intersection is therefore null
        makeEmpty();
    }
    return *this;
}


IntVectSet_old& 
IntVectSet_old::operator&=(const IntVect& iv_in)
{
    Box addbox(iv_in, iv_in);
    (*this) &= addbox;
    return *this;
}


IntVectSet_old& 
IntVectSet_old::operator&=(const Box& box_in)
{
    if(box_in.isEmpty())
    {
        makeEmpty();
        return *this;
    }
    if(!isEmpty())
    {
        Vector<Box> vect_inter;
        Box addbox = box_in;
        LeafIterator lt_this(tree_ptr,Leaf::Regular);
        for(lt_this.begin(); lt_this.ok(); lt_this.next())
        {
            Box thisbox = lt_this.getDomain();
            thisbox &= addbox;
            if (!thisbox.isEmpty())
            {
                vect_inter.push_back(thisbox);
            }
        }
        //create new tree and add intersecting boxes
      
        Tree* new_tree_ptr = new Tree();
        for(int ivect = 0; ivect < vect_inter.size(); ivect++)
        {
            Box hoombox= vect_inter[ivect];
            new_tree_ptr->addBox(hoombox, Leaf::Regular);
        }
        if(tree_ptr != NULL) delete tree_ptr;
        tree_ptr = new_tree_ptr;
    }
    getNewMinBox();
    return *this;
}


IntVectSet_old 
operator&(const IntVectSet_old& ivs, const Box& box_in) 
{
    IntVectSet_old outivs = ivs;
    outivs &= box_in;
    return outivs;
}


IntVectSet_old 
operator&(const Box& box_in, const IntVectSet_old& ivs) 
{
    IntVectSet_old outivs = ivs;
    outivs &= box_in;
    return outivs;
}


IntVectSet_old 
operator&(const IntVectSet_old& ivs, const IntVect& iv_in) 
{
    IntVectSet_old outivs = ivs;
    Box addbox(iv_in, iv_in);
    outivs &= addbox;
    return outivs;
}


IntVectSet_old
operator&(const IntVectSet_old& ivs1, const IntVectSet_old& ivs2)
{
    IntVectSet_old outivs = ivs1;
    outivs &= ivs2;
    return outivs;
}


IntVectSet_old& 
IntVectSet_old::operator-=(const Box& box_in)
{ 
    // Make temporaries to calculate minbox later.
    if (box_in.isEmpty()) return *this;

    if (!isEmpty())
    {
        Box delbox = box_in;

        // Can only delete boxes inside bounding box.
        delbox &= minbox;

        // delbox &= tree_ptr->getDomain().
        if (!delbox.isEmpty())
        {
            tree_ptr->deleteBox(delbox,Leaf::Regular);

            // Recalculate minimum box, all that.
            if (!isEmpty())
            {
                getNewMinBox();
            }
            else
            {
                makeEmpty();
            }
        }

    }
    return *this;
}


IntVectSet_old& 
IntVectSet_old::operator-=(const IntVectSet_old& ige_in)
{
    if (ige_in.isEmpty()) return *this ;

    LeafIterator lt_in(ige_in.tree_ptr,Leaf::Regular);
    for(lt_in.begin(); lt_in.ok(); lt_in.next())
    {
        const Box& addbox = lt_in.getDomain();
        (*this) -= addbox;
    }
    return *this;
}


IntVectSet_old& 
IntVectSet_old::operator-=(const IntVect& iv_in)
{
    Box addbox(iv_in, iv_in);
    (*this) -= addbox;
    return *this;
}



IntVectSet_old 
IntVectSet_old::operator-(const IntVect& iv_in) const
{
    IntVectSet_old outivs = *this;
    Box addbox(iv_in, iv_in);
    outivs -= addbox;
    return outivs;
}


IntVectSet_old 
IntVectSet_old::operator-(const Box& box_in) const
{
    IntVectSet_old outivs = *this;
    outivs -= box_in;
    return outivs;
}


IntVectSet_old 
IntVectSet_old::operator-(const IntVectSet_old& ivs_in) const
{
    IntVectSet_old outivs = *this;
    outivs -= ivs_in;
    return outivs;
}


// Read a text representation of an IntVectSet_old.  Either "(" or "<" is accepted
// as the marker of the start of a set.
istream &
operator>> ( istream&    is,
             IntVectSet_old& ivs )
{
    //[WARNING: this doesn`t check for any errors.]

    ivs.define(); //reinitialize IVS to empty

    // skip whitespace until the start of a set, and determine what the matching
    // end-of-set character should be

    char startch ,endch ;
    is >> startch ;
    switch( startch ) {
    case '(':
        endch = ')' ;  break ;
    case '<' :
        endch = '>' ;  break ;
    default:
        char err[] = "read error: expecting start of IntVectSet_old, got ' '" ;
        //put the erroneous char into the error msg
        err[strlen(err)-2] = startch ;   
        MayDay::Error( err ) ;
    }

    // read IntVects until the end-of-set marker is seen after an IntVect.
    char c ;
    is >> c ;
    while( c != endch ) {
        is.putback( c ) ;

        // read one IntVect and if the read works, put the IntVect into the set.
        //[NOTE: any bad characters in the input will be handled by IntVect::>>.]
        IntVect iv;
        is >> iv ;  //read an IntVect
        ivs |= iv ; //put it in the IntVectSet_old
        is >> c ;   //next non-whitespace input
    }        
    return( is ) ;
}


ostream &
operator<< ( ostream&          os,
             const IntVectSet_old& ivs )
{
    os << "( " ;
    IVSIterator_old i( ivs ) ;
    for( i.begin() ; i.ok() ; i.next() ){
      os << i() << " ";
    }
    os << ")" << endl ;

    return( os ) ;
}
        


//
// functions to manipulate IntVectSet_olds
//

IntVectSet_old
IntVectSet_old::chop(int idir, int chop_pnt)
{
    assert(idir >= 0);
    assert(idir < SpaceDim);
    Vector<Box> hiboxes;
    LeafIterator lt_this(tree_ptr,Leaf::Regular);
    for (lt_this.begin(); lt_this.ok(); lt_this.next())
    {
        Box thisbox = lt_this.getDomain();
        if (thisbox.bigEnd(idir) >= chop_pnt)
        {
            if (thisbox.smallEnd(idir) < chop_pnt)
            {
                Box hibox = thisbox.chop(idir, chop_pnt);
                if (!hibox.isEmpty()) 
                  {
                    hiboxes.push_back(hibox);
                  }
            }
            else
            {
                hiboxes.push_back(thisbox);
            }
        }
    }
    IntVectSet_old retivs;
    for(int ibox = 0; ibox < hiboxes.size(); ibox++)
    {
        retivs |= hiboxes[ibox];
        *this -= hiboxes[ibox];
    }
    return retivs;
}


IntVectSet_old&
IntVectSet_old::coarsen(int iref) 
{
    assert(iref > 0);

    // Save old boxes.
    Vector<Box> oldboxes;
    LeafIterator lt_cur(tree_ptr,Leaf::Regular);

    for (lt_cur.begin(); lt_cur.ok(); lt_cur.next())
    {
        Box thisbox = lt_cur.getDomain();
        oldboxes.push_back(thisbox);
    }

    minbox.coarsen(iref);
    if(tree_ptr != NULL) delete tree_ptr;
    tree_ptr = new Tree();

    for (int ivec = 0; ivec < oldboxes.size(); ivec++)
    {
        Box coarbox = oldboxes[ivec];
        coarbox.coarsen(iref);
        tree_ptr->addBox(coarbox, Leaf::Regular);
    }
    return *this;
}


IntVectSet_old
coarsen(const IntVectSet_old& ivsin, int iref)
{
    assert(iref > 0);
    IntVectSet_old coar_ivs;
    coar_ivs.minbox = coarsen(ivsin.minbox, iref);
    coar_ivs.tree_ptr = new Tree();

    LeafIterator lt_cur(ivsin.tree_ptr,Leaf::Regular);
    for (lt_cur.begin(); lt_cur.ok(); lt_cur.next())
    {
        Box coarbox = lt_cur.getDomain();
        coarbox.coarsen(iref);
        coar_ivs.tree_ptr->addBox(coarbox, Leaf::Regular);
    }
    assert((coar_ivs.tree_ptr)->numChildren ==
           (coar_ivs.tree_ptr)->countChildren());
    return coar_ivs;
}


bool
IntVectSet_old::contains(const IntVect& iv) const
{
    bool retvalue = false;
    if(!minbox.isEmpty() && minbox.contains(iv))
    {
        if (tree_ptr->isLeaf(iv))
        {
            retvalue = true;
        }
    }
    return retvalue;
}


IntVectSet_old&
IntVectSet_old::grow(int igrow )
{
  assert (igrow >= 0);

  IntVectSet_old return_ivs;
  return_ivs.minbox = minbox;
  return_ivs.minbox.grow(igrow);
  
  if(return_ivs.tree_ptr != NULL) 
    delete return_ivs.tree_ptr;
  return_ivs.tree_ptr = new Tree();
  
  LeafIterator lt_cur(tree_ptr,Leaf::Regular);
  for (lt_cur.begin(); lt_cur.ok(); lt_cur.next())
    {
      Box addbox = lt_cur.getDomain();
      addbox.grow(igrow);
      return_ivs.tree_ptr->addBox(addbox,Leaf::Regular);
    }
  
  *this =  return_ivs;
  return *this;
}


IntVectSet_old&
IntVectSet_old::grow(int idir, int igrow )
{
  assert (idir >= 0);
  assert (idir < SpaceDim);
  assert (igrow >= 0);

    IntVectSet_old return_ivs;
    
    LeafIterator lt_cur(tree_ptr,Leaf::Regular);
    for (lt_cur.begin(); lt_cur.ok(); lt_cur.next())
    {
        Box addbox = lt_cur.getDomain();
        addbox.grow(idir, igrow);
        return_ivs |= addbox;
    }

    *this =  return_ivs;
    return *this;
}


IntVectSet_old
grow(const IntVectSet_old& ivs, int igrow )
{
  assert (igrow >= 0);

    IntVectSet_old return_ivs;

    LeafIterator lt_cur(ivs.tree_ptr,Leaf::Regular);
    for (lt_cur.begin(); lt_cur.ok(); lt_cur.next())
    {
        Box addbox = lt_cur.getDomain();
        addbox.grow(igrow);
        return_ivs |= addbox;
    }

    return (return_ivs);
}


void
IntVectSet_old::makeEmpty()
{
    Box invbox;
    if(tree_ptr != NULL) delete tree_ptr;
    tree_ptr = new Tree();
    minbox = invbox;
}


IntVectSet_old&
IntVectSet_old::refine(int iref)
{
    assert(iref > 0);

    // Save old boxes.
    Vector<Box> oldboxes;
    LeafIterator lt_cur(tree_ptr,Leaf::Regular);

    for (lt_cur.begin(); lt_cur.ok(); lt_cur.next())
    {
        Box thisbox = lt_cur.getDomain();
        oldboxes.push_back(thisbox);
    }

    minbox.refine(iref);
    if(tree_ptr != NULL) delete tree_ptr;
    tree_ptr = new Tree();

    for (int ivec = 0; ivec < oldboxes.size(); ivec++)
    {
        Box finebox = oldboxes[ivec];
        finebox.refine(iref);
        tree_ptr->addBox(finebox, Leaf::Regular);
    }
    return *this;
    
}


IntVectSet_old
refine(const IntVectSet_old& ivsin, int iref) 
{
    assert(iref > 0);
    IntVectSet_old fine_ivs;
    fine_ivs.minbox = refine(ivsin.minbox,iref);

    LeafIterator lt_cur(ivsin.tree_ptr,Leaf::Regular);
    for (lt_cur.begin(); lt_cur.ok(); lt_cur.next())
    {
        Box thisbox = lt_cur.getDomain();
        Box finebox = refine(thisbox,iref);
        fine_ivs.tree_ptr->addBox(finebox, Leaf::Regular);
    }
    return fine_ivs;
    
}


void 
IntVectSet_old::shift(IntVect iv)
{
    if (!isEmpty())
    {
        minbox += iv;
        LeafIterator lt(tree_ptr,Leaf::Regular);
        Vector<Box> bSave;

        for (lt.begin(); lt.ok(); lt.next())
        {
            bSave.push_back(lt.getDomain());
        }
        
        delete tree_ptr;
        tree_ptr = new Tree();

        for (int i = 0;i < bSave.size();i++)
        {
            Box b = bSave[i];
            b += iv;
            tree_ptr -> addBox(b,Leaf::Regular);
        }
    }
}



//
// Access functions
//

bool
IntVectSet_old::isEmpty() const
{
    LeafIterator lt_this(tree_ptr,Leaf::Regular);
    lt_this.begin();
    return(!(lt_this.ok()));
}



int 
IntVectSet_old::numPts() const
{
    int iout = 0;
    LeafIterator lt_in(tree_ptr,Leaf::Regular);
    for(lt_in.begin(); lt_in.ok(); lt_in.next())
    {
        Box addbox = lt_in.getDomain();
        BoxIterator bit(addbox);
        for (bit.begin(); bit.ok(); ++bit) 
        {
            iout++;
        }
    }
    return iout;
}




//
// Protected functions

void
IntVectSet_old::getNewMinBox() 
{
    // Calculate the minimum box for this and reset member data
    Box new_minbox;
    LeafIterator lt_this(tree_ptr,Leaf::Regular);
    for(lt_this.begin(); lt_this.ok(); lt_this.next())
    {
        Box thisbox = lt_this.getDomain();
        if (!new_minbox.isEmpty() && !thisbox.isEmpty())
        {
            new_minbox.minBox(thisbox);
        }
        else if (!thisbox.isEmpty())
        {
            new_minbox = thisbox;
        }
        else
        {
            cerr << "IntVectSet_old::getNewMinBox: LeafIterator returned bad box" 
                 << endl;
            abort();
        }
    }
    minbox = new_minbox;
}


Vector<Box>
IntVectSet_old::boxes() const
{
  Vector<Box> return_boxes;
  LeafIterator lt_in(tree_ptr, Leaf::Regular);
  for (lt_in.begin(); lt_in.ok(); lt_in.next())
    {
      Box b = lt_in.getDomain();
      return_boxes.push_back(b);
    }
  return (return_boxes);
}


void
IntVectSet_old::printBoxes(ostream& a_ostrm) const
{
  LeafIterator lt_in(tree_ptr,Leaf::Regular);
  for (lt_in.begin(); lt_in.ok(); lt_in.next())
    {
      Box b = lt_in.getDomain();
      a_ostrm << b << endl;
    }
}


ostream&
IntVectSet_old::writeBoxesPlotfile(ostream& a_ostrm) const
{
#if (CH_SPACEDIM == 2)  
  LeafIterator lt_in(tree_ptr,Leaf::Regular);
  for (lt_in.begin(); lt_in.ok(); lt_in.next())
    {
      Box b = lt_in.getDomain();
      a_ostrm << b.smallEnd(0) << "  " << b.smallEnd(1) << endl;
      a_ostrm << b.smallEnd(0) << "  " << b.bigEnd(1) + 1. << endl;
      a_ostrm << b.bigEnd(0) + 1. << "  " << b.bigEnd(1) + 1. << endl;
      a_ostrm << b.bigEnd(0) + 1. << "  " << b.smallEnd(1) << endl;
      a_ostrm << b.smallEnd(0) << "  " << b.smallEnd(1) << endl;
      a_ostrm << b.bigEnd(0) + 1. << "  " << b.bigEnd(1) + 1. << endl;
      a_ostrm << endl;
      a_ostrm << b.smallEnd(0) << "  " << b.bigEnd(1) + 1. << endl;
      a_ostrm << b.bigEnd(0) + 1. << "  " << b.smallEnd(1) << endl;
      a_ostrm << endl;
    }
#else
  cerr << "IntVect::writeBoxesPlotFile: this is only defined for CH_SPACEDIM == 2" << endl;
#endif
  return a_ostrm;
}


ostream&
IntVectSet_old::plotASCII(ostream& a_ostrm) const
{
#if (CH_SPACEDIM == 2)
  const int tic_spacing = 5;
  const int tic_remainder = 0;

  for (int row = minbox.bigEnd(1); 
       row >= minbox.smallEnd(1); 
       --row)
    {
      for (int column = minbox.smallEnd(0); 
           column <= minbox.bigEnd(0); 
           ++column)
        {
          const IntVect iv(column,row);
          if (contains(iv))
            {
              if ( (column % tic_spacing == tic_remainder) || 
                   (row    % tic_spacing == tic_remainder) )
                {
                  a_ostrm << "%";
                }
              else
                {
                  a_ostrm << "x";
                }
            }
          else
            {
              if ( (column % tic_spacing == tic_remainder) && 
                   (row    % tic_spacing == tic_remainder) )
                {
                  a_ostrm << "+";
                }
              else if (column % tic_spacing == tic_remainder)
                {
                  a_ostrm << "|";
                }
              else if (row % tic_spacing == tic_remainder)
                {
                  a_ostrm << "-";
                }
              else
                {
                  a_ostrm << ".";
                }
            }
        }
      a_ostrm << endl;
    }
#else
  a_ostrm << "IntVectSet_old::plotASCII(...) is only defined for 2D" << endl;
#endif  
  return a_ostrm;
}
