/*  _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
/*

** This software is copyright (C) by the Lawrence Berkeley National
** Laboratory.  Permission is granted to reproduce this software for
** non-commercial purposes provided that this notice is left intact.
**  
** It is acknowledged that the U.S. Government has rights to this
** software under Contract DE-AC03-765F00098 between the U.S.  Department
** of Energy and the University of California.
**  
** This software is provided as a professional and academic contribution
** for joint exchange. Thus it is experimental, is provided ``as is'',
** with no warranties of any kind whatsoever, no support, no promise of
** updates, or printed documentation. By using this software, you
** acknowledge that the Lawrence Berkeley National Laboratory and Regents
** of the University of California shall have no liability with respect
** to the infringement of other copyrights by any part of this software.
*/

#ifndef LAYOUTDATA_H
#define LAYOUTDATA_H

#include "BoxLayout.H"
#include "DataIterator.H"

//class ArrayViewData;

///  Data that maintains a one-to-one mapping to the boxes in a BoxLayout
/**

  A collection of Box-oriented objects.  The arrangement
  of the T objects is given by the underlying BoxLayout object.
  LayoutData does not attempt to prevent users from manipulating
  the template class T to become out of synch with the
  boxes in the BoxLayout.  Caveat emptor.
  
  Non-local (off-processor access) to a LayoutData index is an error.
  you can assure that access to the data stored in a LayoutData is
  local to your processor by using the DataIterator object for indexing.
  (In fact, you are pretty much forced to by the 

  Data in a LayoutData CANNOT be communicated to other
  processors using the API presented here.  

  class T must provide the following methods:
  \begin{verbatim}
  {
    T()
  }
  \end{verbatim}

  This class is useful for processor-local data that needs to be indexable
  along with a BoxLayoutData.  Auxillary data objects, etc.

*/
template<class T> class LayoutData
{
public:

  friend class ArrayViewData;

  ///
  LayoutData();
	
  ///
  /** constructor.  allocates a T object for every box in the BoxLayout dp
    using the T null constructor.  dp must be closed. */
  LayoutData(const BoxLayout& dp);

  ///
  virtual ~LayoutData();

  ///
  /** constructor.  allocates a T object for every box in the BoxLayout dp
    using the T null constructor.  dp must be closed.  previously stored
    T objects are removed */
  virtual void define(const BoxLayout& dp);

  // this approach, of using the iterator itself for indexing is
  // my own brilliance.  This way you don't need all the
  // crappy [Const][Dependent] Iterators.  It also forces all
  // data access to be done through the DataIterator, which will
  // know how to talk to KeLP (or other parallel architectures,
  // like multithreading, etc).

  ///
  DataIterator dataIterator() const;

  ///
  const T& operator[](const DataIndex& index) const;

  ///
  T& operator[](const DataIndex& index);

  ///
  const Box& box(const DataIndex& index) const;


  // not really a protection mechanism, since a user can always
  // perform a shallow copy of the BoxLayout, and be free to manipulate
  // the data in the object.  If you wish to have an actual copy 
  // of the BoxLayout, then you need to invoke the clone() method.

  ///
  const BoxLayout& boxLayout() const
    {
      return m_boxLayout;
    }


protected:

  BoxLayout m_boxLayout;

  // this used to be std::vector<T>, and I could let vector handle
  // destruction for me, but vector.resize() absolutely demands that
  // I provide a copy constructor for T.  People get uncomfortable when
  // I make them provide me with copy constructors.
  std::vector<T*> m_vector;

  // thinking about making this class self-documenting to a greater degree
  // and having the component names also be kept in the class and carried
  // around through various transformations.....
  //  vector<string> m_componentNames;

private:

  // disallow copy constructors and assignment operators
  // to avoid very hard-to-find performance problems 
  LayoutData<T>& operator= (const LayoutData<T>& rhs);
  LayoutData(const LayoutData& rhs);

 // handy usage function, tied to the implementation of this class
  // as a vector<T*>.  Assumes that m_comps and m_boxLayout have already
  // been initialized correctly.  Sets the correct size for the data
  // vector, deletes the extra T objects if m_vector is to shorten, and
  // performs either construction or define on the remaining T objects.

  void allocate();

};


//======================================================================
//======================================================================




//========= inline functions ===========================

// not literally a .H file, but just an experiment in
// having useable, readable headers, while having 
// the dependeny system work properly.  Since LayoutData.H
// now depends on LayoutDataI.H, then changing either
// should cause all code that includes LayoutData.H to be
// recompiled.  This way people can just read the interface
// in this file.  Implementation is put in the *I.H file.

#include "LayoutDataI.H"

#endif // LAYOUTDATA_H
