/*  _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
/*

** This software is copyright (C) by the Lawrence Berkeley National
** Laboratory.  Permission is granted to reproduce this software for
** non-commercial purposes provided that this notice is left intact.
**  
** It is acknowledged that the U.S. Government has rights to this
** software under Contract DE-AC03-765F00098 between the U.S.  Department
** of Energy and the University of California.
**  
** This software is provided as a professional and academic contribution
** for joint exchange. Thus it is experimental, is provided ``as is'',
** with no warranties of any kind whatsoever, no support, no promise of
** updates, or printed documentation. By using this software, you
** acknowledge that the Lawrence Berkeley National Laboratory and Regents
** of the University of California shall have no liability with respect
** to the infringement of other copyrights by any part of this software.
*/

#ifndef LEVELARRAY_H
#define LEVELARRAY_H

#include "IntVect.H"
#include "BoxLayoutData.H"
#include "DisjointBoxLayout.H"
#include "Copier.H"
#include "SPMD.H"


///Data over a disjoint union of rectangles

/**
 LevelData is-a BoxLayoutData.  It is built from a DisjointBoxLayout
 instead though, thus making the data in a DisjointBoxLayout uniquely
 defined for all spatial indices defined by the underlying
 DisjointBoxLayout.  It carries within it an inherent concept of
 ghost cells (which the user can choose to be empty).

  Since LevelData is-a BoxLayoutData, all the methods
  required of the template class T for BoxLayoutData are also
  required for LevelData. LevelData does not have any extra requirements.
*/

template<class T> class LevelData : public BoxLayoutData<T>
{
public:

  ///
  LevelData();
	
  ///
  LevelData(const DisjointBoxLayout& dp, int comps,
			const IntVect& ghost = IntVect::TheZeroVector(),
			const DataFactory<T>& a_factory = DataFactory<T>());

  virtual ~LevelData();
  ///

  ///
  virtual void define(const DisjointBoxLayout& dp, int comps, 
					  const IntVect& ghost = IntVect::TheZeroVector(),
					  const DataFactory<T>& a_factory = DataFactory<T>());

  ///
  /** 
    copy definer.  this LevelData thrown away and da's LevelData copied
    */
  virtual void define(const LevelData<T>& da, 
					  const DataFactory<T>& a_factory = DataFactory<T>());

  ///
  /**
    Copy definer.  'this' LevelData thrown away and da's LevelData copied.
    This LevelData is now defined for the range [0, comps.size()] and filled
    with the data in da from [comps.begin(), comps.end()]
    */
  virtual void define(const LevelData<T>& da, const Interval& comps,
					  const DataFactory<T>& a_factory = DataFactory<T>());

  ///
  virtual void copyTo(const Interval& srcComps, 
		      BoxLayoutData<T>& dest,
		      const Interval& destComps) const;

  ///
  virtual void copyTo(const Interval& srcComps,
		      BoxLayoutData<T>& dest,
		      const Interval& destComps,
		      const Copier& copier) const;
  ///
  /** same as copyTo that takes a BoxLayoutData, except that it fills the
    ghost cells of 'dest' with data from 'this' also. */
  virtual void copyTo(const Interval& srcComps, 
		      LevelData<T>& dest,
		      const Interval& destComps) const;

  ///
  virtual void copyTo(const Interval& srcComps,
		      LevelData<T>& dest,
		      const Interval& destComps,
		      const Copier& copier) const;
  ///
  virtual void exchange(const Interval& comps);

  ///
  virtual void exchange(const Interval& comps,
			const Copier& copier);

  ///
  const IntVect& ghostVect() const { return m_ghost;}

  /**
    These functions will invoke error messages when invoked.
    C++ will ensure that constructors are not called for the
    base class by a user, but a 'define' function has no such protection,
    hence the need to prevent such usage.  A runtime error is not
    a perfect solution...(strong construction gets around this  *sigh*).
    classes that derive from LevelData will have to turn it's valid
    defines into runtime errors also and make it's own defines.  Thus
    taking over the job of the compiler.
    */

  ///
  virtual void define(const BoxLayout& dp, int comps,
					  const DataFactory<T>& factory);

  ///
  virtual void define(const BoxLayoutData<T>& da,
					  const DataFactory<T>& factory = DataFactory<T>());

  ///
  virtual void define(const BoxLayoutData<T>& da, const Interval& comps,
					  const DataFactory<T>& factory = DataFactory<T>());

  virtual void define(const BoxLayout& deadFunction);
  ///
  const DisjointBoxLayout& getBoxes() const
  {
    return m_disjointBoxLayout;
  }

  ///
  const DisjointBoxLayout& disjointBoxLayout() const
  {
    return m_disjointBoxLayout;
  }

protected:

  void makeItSo(const Interval&     a_srcComps, 
		const LevelData<T>& a_src,
		BoxLayoutData<T>&   a_dest,
		const Interval&     a_destComps,
		const Copier&       a_copier) const;

  DisjointBoxLayout m_disjointBoxLayout;

  IntVect   m_ghost;

  //========================================================================
  //
  // data structures used by makeItSo when we have some
  // data that needs to be moved (ie. there are entries
  // in the 'FROM' or 'TO' CopyIterators)
  //
  void completePendingSends() const;

  void allocateBuffers(const LevelData<T>& a_src, 
		       const Interval& a_srcComps,
		       const BoxLayoutData<T>& a_dest,
		       const Interval& a_destComps,
		       const Copier&   a_copier) const;

  void writeSendDataFromMeIntoBuffers(const LevelData<T>& a_src, 
				      const Interval&     a_srcComps) const;

  void postSendsFromMe() const ;

  void postReceivesToMe() const ;

  void unpackReceivesToMe(BoxLayoutData<T>& a_dest, 
			  const Interval&   a_destComps) const ;
  mutable void*  m_sendbuffer; // pointer member OK here, since LevelData<T> has no copy
  mutable size_t m_sendcapacity;
  mutable void*  m_recbuffer; // pointer member OK here, since LevelData<T> has no copy
  mutable size_t m_reccapacity;

#ifdef MPI

  struct bufEntry
  {
    void* bufPtr; // pointer into contiguous m_buffer
    size_t size;
    const MotionItem* item;
    unsigned int procID;
    bool operator < (const bufEntry& rhs) const
      { 
	if(procID == rhs.procID) 
	  {
	    const Box& left = item->region;
	    const Box& right= rhs.item->region;
	    if(left.smallEnd() == right.smallEnd())
	      {
		return left.bigEnd().lexLT(right.bigEnd());
	      }
	    else
	      {
		return item->region < rhs.item->region;
	      }
	  }
	//else
	return procID < rhs.procID;
      }
  };


  mutable std::vector<bufEntry> m_fromMe;
  mutable std::vector<bufEntry> m_toMe;

  mutable MPI_Request  *m_sendRequests, *m_receiveRequests;
  mutable MPI_Status   *m_sendStatus,   *m_receiveStatus;
  mutable int numSends, numReceives;

#endif

};


// ====== inlined function definitions ================

// As with the BoxLayoutData implementation file.  This file
// just gives the interface and the inline implmentations
// are given in LevelDataI.H

#include "LevelDataI.H"



#endif
