// -*- Mode: C++; Modified: "Fri Dec 17 17:19:49 1999 by dbs"; -*- 
// file: LoadBalance.H

// Purpose:
//  LoadBalance() is a global function to compute an assignment
//  of boxes to processors for an AMR mesh hierarchy.  The assignment
//  is made so as to balance the computation and communication
//  workload on each processor (i.e. make it as even as possible).
//
//  ********************************************************
//  CAVEAT: this version ignores the communication workload.
//          It balances ONLY the compute workload.
//  ********************************************************
//
// Usage:
//  The meshes in the AMR hierarchy are represented using
//   a Vector of Vectors of Boxes.
//
//  The computational workload is a real number for each box in the
//   hierarchy, also represented as a Vector of Vectors.
//
//  The communication workload is a real number for each connection
//   between pairs of boxes, which specifies the cost of communication
//   between the two boxes if they are on different processors.  No
//   allowance is made for varying the communication cost depending on the
//   distance between processors (ie. all processors are equally far apart)
//   and it assumes that the effective cost of communicating between two
//   boxes on the same processor is zero.  The communication workload is
//   represented as a graph.
//
//  ********************************************************
//  CAVEAT: the communication workload argument is not
//           present in this version of the function.
//  ********************************************************
//
//  The resulting assignment is an integer for each box in the hierarchy,
//   which gives the processor number (starting at zero) on which each box
//   should reside.  Again, represented as a Vector of Vectors.
//
//  The other output argument is the efficiency ratio, which is a measure
//   of how close to perfect the final load balance is.  The \var{effRatio}
//   output variable is defined as the smallest load on any processor divided
//   by the largest load.  A perfect load balance has efficiency == 1.
//
//  ********************************************************
//  CAVEAT: in this version, each level in the AMR hierarchy
//          is balanced independently, so the efficiency ratio
//          that is returned is the worst over all levels.
//  ********************************************************
//
//  It is important to note that it is the sum of the computation cost 
//   and communication cost that is balanced, so the values in the two
//   variables must be in the same units.  It doesn't matter what the
//   units are.
//
//  ********************************************************
//  CAVEAT: the communication workload argument is not
//           present in this version of the function.
//  ********************************************************
// 
// Interface Specification:
//  int LoadBalance() arguments are:
//       Vector<Vector<int>>& procAssignments  //output: processor number 
//                                             //   for each box
//       Real&                effRatio         //output: efficiency ratio
// const Vector<Vector<Box>>& Grids            //input: meshes to balance
// const Vector<Vector<long>>&ComputeLoads     //input: computational cost
//                                             //   of each box
//###not used###
//### const [...something...]&CommunicateLoads //input: communication cost
//                                             //   between each pair of 
//                                             //   neighboring boxes
// const Vector<int>&         RefRatios        //input: refinement ratio
//                                             //   for each level
//
// Returns: integer status_code
//    =0  means succesful completion
//  exceptions: (output variables are not defined)
//  -1011 input vectors (\var{Grids} and \var{ComputeLoads}) are not
//        the same size
//  -1012 one of the vector elements of the input vectors (\var{Grids} 
//        and \var{ComputeLoads}) are not the same size
//  -1013 input vectors (\var{Grids} and \var{RefRatios}) are not
//        the same size
//
// Modification History
//  19Nov99 <dbs> initial design and coding
//


#ifndef _LOADBALANCE_H_
#define _LOADBALANCE_H_

#include "SPACE.H"
#include "REAL.H"
#include "Box.H"
#include "Vector.H"
#include "BoxLayout.H"
#include "SPMD.H"

int
LoadBalance( Vector<Vector<int> >& procAssignments  //output: processor number  
//                                                  //   for each box           
            ,Real&                 effRatio         //output: ratio of min load
//                                                  //   to max load
     ,const Vector<Vector<Box> >&  Grids            //input: meshes to balance  
     ,const Vector<Vector<long> >& ComputeLoads     //input: computational cost 
//                                                  //   of each box            
//## ,const [...something...]&     CommunicateLoads //input: communication cost 
//                                                  //   between each pair of   
//                                                  //   neighboring boxes      
     ,const Vector<int>&           RefRatios        //input: refinement ratio   
//                                                  //   for each level        
			 ,int nProc = numProc()
             ) ;


int
LoadBalance( Vector<BoxLayout>&    Grids            //in-out: input grids to balance
//                                                  //   and output proc. numbers
            ,Real&                 effRatio         //output: ratio of min load
//                                                  //   to max load
     ,const Vector<Vector<long> >& ComputeLoads     //input: computational cost 
//                                                  //   of each box            
//## ,const [...something...]&     CommunicateLoads //input: communication cost 
//                                                  //   between each pair of   
//                                                  //   neighboring boxes      
     ,const Vector<int>&           RefRatios        //input: refinement ratio   
//                                                  //   for each level         
			 ,int nProc = numProc()
             ) ;
///
/** convenience function to load balance a Vector of Boxes based 
 on load=box.numPts() */
int LoadBalance(Vector<int>& procAssignments, const Vector<Box>& boxes);

#endif
