// -*- Mode: C++; c-file-style: "gnu"; Modified: "Thu 27 Jan 2000 12:52:06 by dave"; -*- 
// file: MeshRefine.H

// MeshRefine version 2.3

// NOTE: The version 2.3 API is not compatible with the version 1.x or 2.0 API.

// See MeshRefine.cpp for complete documentation.


#if ! defined(__MESH_REFINE_HEADER__)

// Include files:

#include <climits>

#include "Vector.H"
#include "Box.H"
#include "IntVectSet.H"
#include "REAL.H"

// Constants:

// Minimum acceptable ratio of tagged cells to total cells for 
// the Berger-Rigoutsos algorithm in \func{makeBoxes}.
// Used as default for \var{FillRatio} optional argument.
#if ! defined(_BR_MIN_BOX_FILL_RATIO_)
#define _BR_MIN_BOX_FILL_RATIO_ ( 0.75 )
#endif

///
/**
   Splits domain into a vector of disjoint boxes with
   maximum size maxsize.
   blocking factor is default to one.
   If you make minimum size > 1, then domain must
   be coarsenable and refineable by blockfactor
   (refine(coarsen(domain,blockfactor), minsize) == domain)
   or an error is thrown
 */
extern int domainSplit(const Box& domain, Vector<Box>& vbox, int maxsize, int blockfactor=1);

/// MeshRefine with a single level of tags
/**
 There are two versions of meshRefine: 
  one takes a single IntVectSet of tags defined on the BaseLevel
  mesh and uses that set of tags for every level to be refined;
  the other takes a Vector<IntVectSet> of tags defined on all the
   mesh levels to be refined and uses those. \\ \\
Arguments: \\ \\
 newmeshes--    output: refined meshes 
                    [0:TopLevel+1] on exit \\ \\
 tags     --    in_out: tagged cells on BaseLevel \\ \\
 BaseLevel--    input: index of base mesh level \\ \\
 TopLevel --    input: index of finest mesh \\ \\
 Domains  --    input: problem domain boundary
                    [BaseLevel:TopLevel] \\ \\
 OldMeshes--    input: existing meshes
                    [0:TopLevel] \\ \\
 RefRatios--    input: range[2:\infty] # of refined cells (in each
                    dimension) at finer level for
                    each cell to be refined
                    [BaseLevel:TopLevel] \\ \\
 FillRatio--    input,optional: range(0:1) minimum
                    acceptable ratio of tagged
                    cells in a box (default=0.75) \\ \\
 BlockFactor--  input,optional: range[1:MaxSize]
                    mesh size granularity
                    and alignment constraint
                    (default = 1) \\ \\
 BufferSize--   input,optional: range[1:\infty] min number
                    of level=L mesh cells between 
                    edge of a level=L+1 mesh and a
                    level=L-1 mesh
                    (default = 1) \\ \\
 MaxSize   --   input,optional: range[1:\infty ] 
                    largest
                    number of grid points allowed
                    in any dimension
                    (default = \infnty) \\ \\

{\bf Long Description:} \\

  Create new meshes based on tagged cells on a range of levels of a mesh
  hierarchy.  Each level of tagged cells is used to generate a new mesh at
  the next finer level.  The finest level in the output mesh will be one
  level higher than the top of the range of levels given as input.  As a 
  special case, use the same tags (appropriately refined) for all levels.

 Usage:

  Call this function after computing error estimates and tagging cells.  To
  add a new mesh level, set TopLevel to the index of the finest level
  in the existing mesh and define tags on the finest level.  To keep the
  existing number of mesh levels, set TopLevel to one less than the
  index of the finest level and don't define any tags on the finest level.
  If a single IntVectSet of tags is passed (instead of a
  Vector<IntVectSet>) then the same tags (properly refined) will be used
  for all the new meshes up to level TopLevel+1.  In any case, the
  meshes at levels BaseLevel and below are not modified.  The output
  argument newmeshes will be reallocated to the necessary size before
  being used.  When this function returns, the elements of the
  newmeshes vector corresponding to the unchanged levels will be
  filled in with copies of the levels from the old mesh vector.  The
  variable tags is modified in an undefined way, so its contents
  should not be relied upon.  The variable BlockFactor specifies a
  multiple of the refinement ratio.  Every grid box will have an integral
  multiple of RefRatio*BlockFactor cells in each dimension and
  also lower index values that are integral multiples.  As a side effect,
  the minimum box size will be RefRatio*BlockFactor


 Arguments:

  newmeshes                type: Vector<Vector<Box>>
      (output, index[0:TopLevel+1]) vector containing the new mesh hierarchy

  tags                     type: Vector<IntVectSet>
      (input/output, index [0:TopLevel]) tagged cells to be covered by the
      new meshes.  For any level L, the new mesh at level L+1 will try to cover
      the tagged cells at level L.  NOTE: This variable is modified in an
      undefined manner so its values should not be used on exit
 --or--
  tags                     type: IntVectSet
      (input/output) tagged cells on the BaseLevel mesh.  These tags
      will be refined and reused for the other mesh levels.

  BaseLevel                type: int
      (input, range[0:TopLevel]) index of the finest mesh level that
      will not be refined

  TopLevel                 type: int
      (input, range[BaseLevel:OldMeshes.size()-1]) index of the
      finest mesh level that will be refined (i.e. the finest mesh level for
      which tags are defined); the new mesh vector newmeshes will have
      an element at index [TopLevel+1]

  Domains                  type: Vector<Box>
      (input, index[0:TopLevel]) boundaries of the physical domain,
      one mesh box at each level.  Must be consistent with RefRatios
      (i.e. Domains[levelL+1] == Domains[levelL] * RefRatios[levelL])

  OldMeshes                type: Vector<Vector<Box>>
      (input, index[0:TopLevel]) the existing mesh hierarchy vector.
      The BaseLevel boxes should be aligned on BufferSize-sized
      internal boundaries.

  RefRatios                type: Vector<int>&
      (input, index[0:TopLevel], range[2^i,i>0])
      the number of cells (in each direction) in level L+1 for each cell in
      level L.  Must be an integral power of 2.

  Optional Arguments:

  FillRatio                type: Real
      (input, optional, range[0:1], default=0.75) the minimum acceptable ratio
      of filled cells in a box in the Berger-Rigoutsos algorithm in
      makeBoxes

  BlockFactor              type: int
      (input, optional, range[1:), default=1)
      multiple of RefRatios that determines the minimum size of a box
      and what index values a box corner may have (i.e. what alignment is has).

  BufferSize               type: int
      (input, optional, range[1:), default=1)
      minimum number of level L mesh cells between the boundary of a level L+1
      mesh and the nearest level L-1 mesh cell.  This is used to guarantee 
      that stencils can be computed across coarse-fine boundaries.

  MaxBoxSize               type: int
      (input, optional, range[2*BufferSize:), default=\infinity)
      largest number of grid points in any dimension that will be generated
      in an individual grid box.  Must be at least twice the BufferSize.
      
 Returns:

  An int giving an exception number is returned.  The first two arguments
  (tags, newmeshes) are modified.

 Exceptions:

  Exception numbers are divided into 3 groups: \\
       0 if no exceptions occurred; \\
      <0 if an unrecoverable exception occurred; \\
      >0 if a recoverable exception occurred. \\
  Unrecoverable exceptions are futher divided into: \\
      <-1000 if some input data (argument or global variable) is invalid \\
      >-1000 if an unpredicted exception occurred \\
  The exception values that this function returns are: \\
     -11 if an exception occurred in makePNDs \\
     -12 if an exception occurred in makeBoxes \\
   -1001 if one of the input argument vectors is not the right size \\
   -1002 if an argument has an invalid value \\
   -1003 if tags has no tagged cells on level TopLevel \\
   -1101 if a validation exception occurred in makePNDs \\
   -1102 if a validation exception occurred in makeBoxes \\

  Expensive validations are done only when debugging is enabled
   (i.e. the DEBUG make variable is "TRUE").
   

 Usage Notes:

  All the input vectors should be defined with max index > TopLevel.
  They should have values for indices [BaseLevel:TopLevel].
  (except for OldMeshes, which must be defined for all indices).  The
  new mesh vector newmeshes will be redefined up to index
  TopLevel+1.  RefRatios should be defined such that
  RefRatios[L] is the value to use to refine the level L mesh to
  produce the level L+1 mesh.  The tags vector is modified in an
  undefined manner.  The output variable newmeshes may not be
  completely defined if an exception occurs.  If you do not know what the
  optional arguments do, it is probably safe to ignore them.
  The BlockFactor can be used to force a minimum box size.

*/
int
meshRefine( Vector<Vector<Box> >&   newmeshes    //output: refined meshes
//                                               //      [0:TopLevel+1] on exit
           ,IntVectSet&             tags         //in_out: tagged cells on BaseLevel
           ,const int               BaseLevel    //input: index of base mesh level
           ,const int               TopLevel     //input: index of finest mesh
           ,const Vector<Box>&      Domains      //input: problem domain boundary
//                                               //      [BaseLevel:TopLevel]
           ,const Vector<Vector<Box> >& OldMeshes//input: existing meshes
//                                               //      [0:TopLevel]
           ,const Vector<int>&      RefRatios    //input: # of refined cells (in each
//                                               //      dimension) at finer level for
//                                               //      each cell to be refined
//                                               //      [BaseLevel:TopLevel]
           ,const Real              FillRatio = _BR_MIN_BOX_FILL_RATIO_
//                                               //input,optional: range(0:1) minimum
//                                               //      acceptable ratio of tagged
//                                               //      cells in a box (default=0.75)
           ,const int               BlockFactor = 1
//                                               //input,optional: range[1:MaxSize]
//                                               //      mesh size granularity
//                                               //      and alignment constraint
//                                               //      (default = 1)
           ,const int               BufferSize  = 1
//                                               //input,optional: range[1:\infty] min number
//                                               //      of level=L mesh cells between 
//                                               //      edge of a level=L+1 mesh and a
//                                               //      level=L-1 mesh
//                                               //      (default = 1)
           ,const int               MaxSize = INT_MAX
//                                               //input,optional: range[1:\infty] 
//                                               //      number of grid points allowed
//                                               //      in any dimension
//                                               //      (default = \infnty)
          ) ;


/// MeshRefine with a hierarchy of tags
/**
 There are two versions of meshRefine: 
  one takes a single IntVectSet of tags defined on the BaseLevel
  mesh and uses that set of tags for every level to be refined;
  the other takes a Vector<IntVectSet> of tags defined on all the
   mesh levels to be refined and uses those. \\ \\
Arguments: \\ \\
 newmeshes--    output: refined meshes 
                   [0:TopLevel+1] on exit \\ \\
 tags     --    in_out: tagged cells on BaseLevel \\ \\
 BaseLevel--    input: index of base mesh level \\\\
 TopLevel --    input: index of finest mesh \\\\
 Domains  --    input: problem domain boundary
                   [BaseLevel:TopLevel] \\\\
 OldMeshes--    input: existing meshes
                   [0:TopLevel] \\\\
 RefRatios--    input: # of refined cells (in each
                   dimension) at finer level for
                   each cell to be refined
                   [BaseLevel:TopLevel] \\\\
 FillRatio--    input,optional: range(0:1) minimum
                   acceptable ratio of tagged
                   cells in a box (default=0.75) 
 BlockFactor--  input,optional: range[1:MaxSize]
                   mesh size granularity
                   and alignment constraint
                   (default = 1) \\\\
 BufferSize--   input,optional: range[1:\infty] min number
                   of level=L mesh cells between 
                   edge of a level=L+1 mesh and a
                   level=L-1 mesh
                   (default = 1) \\\\
 MaxSize   --   input,optional: range[1:\infty] largest
                   number of grid points allowed
                   in any dimension
                   (default = \infnty) \\\\

{\bf Long Description:} \\

  Create new meshes based on tagged cells on a range of levels of a mesh
  hierarchy.  Each level of tagged cells is used to generate a new mesh at
  the next finer level.  The finest level in the output mesh will be one
  level higher than the top of the range of levels given as input.  As a 
  special case, use the same tags (appropriately refined) for all levels.

 Usage:

  Call this function after computing error estimates and tagging cells.  To
  add a new mesh level, set TopLevel to the index of the finest level
  in the existing mesh and define tags on the finest level.  To keep the
  existing number of mesh levels, set TopLevel to one less than the
  index of the finest level and don't define any tags on the finest level.
  If a single IntVectSet of tags is passed (instead of a
  Vector<IntVectSet>) then the same tags (properly refined) will be used
  for all the new meshes up to level TopLevel+1.  In any case, the
  meshes at levels BaseLevel and below are not modified.  The output
  argument newmeshes will be reallocated to the necessary size before
  being used.  When this function returns, the elements of the
  newmeshes vector corresponding to the unchanged levels will be
  filled in with copies of the levels from the old mesh vector.  The
  variable tags is modified in an undefined way, so its contents
  should not be relied upon.  The variable BlockFactor specifies a
  multiple of the refinement ratio.  Every grid box will have an integral
  multiple of RefRatio*BlockFactor cells in each dimension and
  also lower index values that are integral multiples.  As a side effect,
  the minimum box size will be RefRatio*BlockFactor


 Arguments:

  newmeshes                type: Vector<Vector<Box>>
      (output, index[0:TopLevel+1]) vector containing the new mesh hierarchy

  tags                     type: Vector<IntVectSet>
      (input/output, index [0:TopLevel]) tagged cells to be covered by the
      new meshes.  For any level L, the new mesh at level L+1 will try to cover
      the tagged cells at level L.  NOTE: This variable is modified in an
      undefined manner so its values should not be used on exit
 --or--
  tags                     type: IntVectSet
      (input/output) tagged cells on the BaseLevel mesh.  These tags
      will be refined and reused for the other mesh levels.

  BaseLevel                type: int
      (input, range[0:TopLevel]) index of the finest mesh level that
      will not be refined

  TopLevel                 type: int
      (input, range[BaseLevel:OldMeshes.size()-1]) index of the
      finest mesh level that will be refined (i.e. the finest mesh level for
      which tags are defined); the new mesh vector newmeshes will have
      an element at index [TopLevel+1]

  Domains                  type: Vector<Box>
      (input, index[0:TopLevel]) boundaries of the physical domain,
      one mesh box at each level.  Must be consistent with RefRatios
      (i.e. Domains[levelL+1] == Domains[levelL] * RefRatios[levelL])

  OldMeshes                type: Vector<Vector<Box>>
      (input, index[0:TopLevel]) the existing mesh hierarchy vector.
      The BaseLevel boxes should be aligned on BufferSize-sized
      internal boundaries.

  RefRatios                type: Vector<int>&
      (input, index[0:TopLevel], range[2^i,i>0])
      the number of cells (in each direction) in level L+1 for each cell in
      level L.  Must be an integral power of 2.

  Optional Arguments:

  FillRatio                type: Real
      (input, optional, range[0:1], default=0.75) the minimum acceptable ratio
      of filled cells in a box in the Berger-Rigoutsos algorithm in
      makeBoxes

  BlockFactor              type: int
      (input, optional, range[1:), default=1)
      multiple of RefRatios that determines the minimum size of a box
      and what index values a box corner may have (i.e. what alignment is has).

  BufferSize               type: int
      (input, optional, range[1:), default=1)
      minimum number of level L mesh cells between the boundary of a level L+1
      mesh and the nearest level L-1 mesh cell.  This is used to guarantee 
      that stencils can be computed across coarse-fine boundaries.

  MaxBoxSize               type: int
      (input, optional, range[2*BufferSize:), default=\infinity)
      largest number of grid points in any dimension that will be generated
      in an individual grid box.  Must be at least twice the BufferSize.
      
 Returns:

  An int giving an exception number is returned.  The first two arguments
  (tags, newmeshes) are modified.

 Exceptions:

  Exception numbers are divided into 3 groups: \\
       0 if no exceptions occurred; \\
      <0 if an unrecoverable exception occurred; \\
      >0 if a recoverable exception occurred. \\
  Unrecoverable exceptions are futher divided into: \\
      <-1000 if some input data (argument or global variable) is invalid \\
      >-1000 if an unpredicted exception occurred \\
  The exception values that this function returns are: \\
     -11 if an exception occurred in makePNDs \\
     -12 if an exception occurred in makeBoxes \\
   -1001 if one of the input argument vectors is not the right size \\
   -1002 if an argument has an invalid value \\
   -1003 if tags has no tagged cells on level TopLevel \\
   -1101 if a validation exception occurred in makePNDs \\
   -1102 if a validation exception occurred in makeBoxes \\

  Expensive validations are done only when debugging is enabled
   (i.e. the DEBUG make variable is "TRUE").
   

 Usage Notes:

  All the input vectors should be defined with max index > TopLevel.
  They should have values for indices [BaseLevel:TopLevel].
  (except for OldMeshes, which must be defined for all indices).  The
  new mesh vector newmeshes will be redefined up to index
  TopLevel+1.  RefRatios should be defined such that
  RefRatios[L] is the value to use to refine the level L mesh to
  produce the level L+1 mesh.  The tags vector is modified in an
  undefined manner.  The output variable newmeshes may not be
  completely defined if an exception occurs.  If you do not know what the
  optional arguments do, it is probably safe to ignore them.
  The BlockFactor can be used to force a minimum box size.

*/
int
meshRefine( Vector<Vector<Box> >&   newmeshes    //output: refined meshes
//                                               //      [0:TopLevel+1] on exit
           ,Vector<IntVectSet>&     tags         //in_out: tagged cells
//                                               //      [BaseLevel:TopLevel]
           ,const int               BaseLevel    //input: index of base mesh level
           ,const int               TopLevel     //input: index of finest mesh
           ,const Vector<Box>&      Domains      //input: problem domain boundary
//                                               //      [BaseLevel:TopLevel]
           ,const Vector<Vector<Box> >& OldMeshes//input: existing meshes
//                                               //      [0:TopLevel]
           ,const Vector<int>&      RefRatios    //input: # of refined cells (in each
//                                               //      dimension) at finer level for
//                                               //      each cell to be refined
//                                               //      [BaseLevel:TopLevel]
           ,const Real              FillRatio = _BR_MIN_BOX_FILL_RATIO_
//                                               //input,optional: range(0:1) minimum
//                                               //      acceptable ratio of tagged
//                                               //      cells in a box (default=0.75)
           ,const int               BlockFactor = 1
//                                               //input,optional: range[1:MaxSize]
//                                               //      mesh size granularity
//                                               //      and alignment constraint
//                                               //      (default = 1)
           ,const int               BufferSize  = 1
//                                               //input,optional: range[1:\infty] min number
//                                               //      of level=L mesh cells between 
//                                               //      edge of a level=L+1 mesh and a
//                                               //      level=L-1 mesh
//                                               //      (default = 1)
           ,const int               MaxSize = INT_MAX
//                                               //input,optional: range[1:\infty] largest
//                                               //      number of grid points allowed
//                                               //      in any dimension
//                                               //      (default = \infnty)
          ) ;

int
makePNDs( Vector<IntVectSet>& pnds            //output: proper nesting domains at each level
         ,const int           BaseLevel       //input: index of highest AMR level not to be refined
         ,const int           TopLevel        //input: index of highest AMR level in output
         ,const Vector<Box>&  Domains         //input: (same as in \func{meshRefine})
         ,const Vector<Box>&  BaseMesh        //input: boxes at mesh level \var{BaseLevel}
         ,const Vector<int>&  RefRatios       //input: (same as in \func{meshRefine})
         ,const int           BufferSize ) ;  //input: (same as in \func{meshRefine})

int
makeBoxes( Vector<Box>&      mesh        //output: refined boxes at each new level
          ,const IntVectSet& Tags        //input: set of tagged cells to cover
          ,const IntVectSet& Pnd         //input: proper nesting domain in which 
//                                       //       the mesh boxes must live
          ,const Box&        Domain      //input: boundary of physical domain
          ,const int         MaxSize     //input: largest number of cells in any dimension
//                                       //       of any box
          ,Real              FillRatio = _BR_MIN_BOX_FILL_RATIO_
//                                       //input,optional: (same as in \func{meshRefine})
         ) ;



int
makePNDs( Vector<IntVectSet>& pnds            //output: proper nesting domains at each level
         ,const int           BaseLevel       //input: index of highest AMR level not to be refined
         ,const int           TopLevel        //input: index of highest AMR level in output
         ,const Vector<Box>&  Domains         //input: (same as in \func{meshRefine})
         ,const IntVectSet&   BaseMesh        //input: boxes at mesh level \var{BaseLevel}
         ,const Vector<int>&  RefRatios       //input: (same as in \func{meshRefine})
         ,const int           BufferSize ) ;  //input: (same as in \func{meshRefine})


#define __MESH_REFINE_HEADER__
#endif
