// -*- Mode: C++; c-file-style: "gnu"; Modified: "Thu 27 Jan 2000 13:10:53 by dave"; -*-
#include <fstream>
#include <iostream>


// file: MeshRefine.C

/*
   _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/

//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//

// Function:
// ---------
//  int meshRefine()
//
// Short Description:
// ------------------
//  Create new meshes based on tagged cells on a range of levels of a mesh
//  hierarchy.  Each level of tagged cells is used to generate a new mesh at
//  the next finer level.  The finest level in the output mesh will be one
//  level higher than the top of the range of levels given as input.  As a 
//  special case, use the same tags (appropriately refined) for all levels.
//
// Usage:
// ------
//  Call this function after computing error estimates and tagging cells.  To
//  add a new mesh level, set \var{TopLevel} to the index of the finest level
//  in the existing mesh and define tags on the finest level.  To keep the
//  existing number of mesh levels, set \var{TopLevel} to one less than the
//  index of the finest level and don't define any tags on the finest level.
//  If a single IntVectSet of tags is passed (instead of a
//  Vector<IntVectSet>) then the same tags (properly refined) will be used
//  for all the new meshes up to level \var{TopLevel}+1.  In any case, the
//  meshes at levels \var{BaseLevel} and below are not modified.  The output
//  argument \var{newmeshes} will be reallocated to the necessary size before
//  being used.  When this function returns, the elements of the
//  \var{newmeshes} vector corresponding to the unchanged levels will be
//  filled in with copies of the levels from the old mesh vector.  The
//  variable \var{tags} is modified in an undefined way, so its contents
//  should not be relied upon.  The variable \var{BlockFactor} specifies a
//  minimum grid size.  Every grid box will have an integral
//  multiple of \var{RefRatio}*\var{ceiling(BlockFactor/RefRatio)} cells in each dimension and
//  also lower index values that are integral multiples.  
//
//
// Arguments:
// ----------
//  \var{newmeshes}                type: Vector<Vector<Box>>
//      (output, index[0:\var{TopLevel}+1]) vector containing the new mesh hierarchy
//
//  \var{tags}                     type: Vector<IntVectSet>
//      (input/output, index [0:\var{TopLevel}]) tagged cells to be covered by the
//      new meshes.  For any level L, the new mesh at level L+1 will try to cover
//      the tagged cells at level L.  NOTE: This variable is modified in an
//      undefined manner so its values should not be used on exit
// --or--
//  \var{tags}                     type: IntVectSet
//      (input/output) tagged cells on the \var{BaseLevel} mesh.  These tags
//      will be refined and reused for the other mesh levels.
//
//  \var{BaseLevel}                type: int
//      (input, range[0:\var{TopLevel}]) index of the finest mesh level that
//      will not be refined
//
//  \var{TopLevel}                 type: int
//      (input, range[\var{BaseLevel}:\var{OldMeshes.size()-1]) index of the
//      finest mesh level that will be refined (i.e. the finest mesh level for
//      which tags are defined); the new mesh vector \var{newmeshes} will have
//      an element at index [\var{TopLevel}+1].
//
//  \var{Domains}                  type: Vector<Box>
//      (input, index[0:\var{TopLevel}]) boundaries of the physical domain,
//      one mesh box at each level.  Must be consistent with \var{RefRatios}
//      (i.e. Domains[levelL+1] == Domains[levelL] * RefRatios[levelL])
//
//  \var{OldMeshes}                type: Vector<Vector<Box>>
//      (input, index[0:\var{TopLevel}]) the existing mesh hierarchy vector.
//      The \var{BaseLevel} boxes should be aligned on \var{BufferSize}-sized
//      internal boundaries.
//
//  \var{RefRatios}                type: Vector<int>&
//      (input, index[0:\var{TopLevel}], range[2^i,i>0])
//      the number of cells (in each direction) in level L+1 for each cell in
//      level L.  Must be an integral power of 2.
//
//  Optional Arguments:
//
//  \var{FillRatio}                type: Real
//      (input, optional, range[0:1], default=0.75) the minimum acceptable ratio
//      of filled cells in a box in the Berger-Rigoutsos algorithm in
//      \func{makeBoxes}
//
//  \var{BlockFactor}              type: int
//      (input, optional, range[1:), default=1)
//      factor  that determines the minimum size of a box
//      and what index values a box corner may have (i.e. what alignment is has).
//
//  \var{BufferSize}               type: int
//      (input, optional, range[1:), default=1)
//      minimum number of level L mesh cells between the boundary of a level L+1
//      mesh and the nearest level L-1 mesh cell.  This is used to guarantee 
//      that stencils can be computed across coarse-fine boundaries.
//
//  \var{MaxBoxSize}               type: int
//      (input, optional, range[2*BufferSize:), default=\infinity)
//      largest number of grid points in any dimension that will be generated
//      in an individual grid box.  Must be at least twice the BufferSize.
//      
// Returns:
// --------
//  An int giving an exception number is returned.  The first two arguments
//  (\var{tags}, \var{newmeshes}) are modified.
//
// Exceptions:
// -----------
//  Exception numbers are divided into 3 groups:
//       0 if no exceptions occurred;
//      <0 if an unrecoverable exception occurred;
//      >0 if a recoverable exception occurred.
//  Recoverable exceptions are further divided into:
//      >+1000 if some input data has an unexpected, but not fatal, value
//      <+1000 if a correctable exception occurred during execution
//  Unrecoverable exceptions are futher divided into:
//      <-1000 if some input data (argument or global variable) is invalid
//      >-1000 if an unpredicted exception occurred during execution
//  The exception values that this function returns are:
//   +1001 if TopLevel is negative; \var{newmeshes} unchanged
//     -11 if an exception occurred in \func{makePNDs}
//     -12 if an exception occurred in \func{makeBoxes}
//   -1001 if one of the input argument vectors is not the right size
//   -1002 if an argument has an invalid value
//   -1003 if \var{tags} has no tagged cells on level \var{TopLevel}
//   -1101 if a validation exception occurred in \func{makePNDs}
//   -1102 if a validation exception occurred in \func{makeBoxes}
//
//  Expensive validations are done only when debugging is enabled
//   (i.e. the DEBUG make variable is "TRUE").
//   
//
// Usage Notes:
// ------------
//  All the input vectors should be defined with max index > \var{TopLevel}.
//  They should have values for indices [\var{BaseLevel}:\var{TopLevel}].
//  (except for \var{OldMeshes}, which must be defined for all indices).  The
//  new mesh vector \var{newmeshes} will be redefined up to index
//  \var{TopLevel}+1.  \var{RefRatios} should be defined such that
//  \var{RefRatios}[L] is the value to use to refine the level L mesh to
//  produce the level L+1 mesh.  The \var{tags} vector is modified in an
//  undefined manner.  The output variable \var{newmeshes} may not be
//  completely defined if an exception occurs.  If you do not know what the
//  optional arguments do, it is probably safe to ignore them.
//  The BlockFactor can be used to force a minimum box size.
//
// Long Description:
// -----------------
//  MeshRefine replaces the meshes at levels \var{BaseLevel}+1 to
//  \var{TopLevel} and creates a new mesh at level \var{TopLevel}+1.  Levels
//  \var{BaseLevel} and below in the new mesh hierarchy are the same as in
//  the old one.  The new meshes are generated so that the tagged cells at
//  level L are refined and included in the new mesh at level L+1 (except
//  for tagged cells that aren`t properly nested in the \var{BaseLevel}
//  mesh).  If TopLevel<0, no new meshes are created and a warning code
//  is returned.
//
// Numerical Algorithm:
// --------------------
//  The tagged cells at level L are used to generate a new (refined) mesh at
//  level L+1.  The \var{BaseLevel} mesh is unchanged.  The new meshes must
//  satisfy the proper nesting requirements, i.e. each new mesh box must be
//  contained in the proper nesting domain (PND) for the next lower level.  The
//  PNDs of all levels are derived from the \var{BaseLevel} mesh (see
//  \func{makePNDs}).  The basic idea is that the proper nesting domain of a
//  level L mesh contains the interior cells of the level L-1 PND, except for
//  faces that are on the physical boundary.  The base level PND is defined as
//  the interiors of the base level mesh boxes, except for boundary faces
//  (again) and box face cells that are surrounded by cells of other boxes.
//
//  In the following graphic, the PND for two mesh boxes with BufferSize=1 is
//  shown.  
//
//        ###################################################
//        #     |   |   |   |   |   |                       #
//        #     |   | o | o | o |   |                       #
//        #     |   |   |   |   |   |                       #
//        #     +---+---+---+---+---%---+---+---+---+---+---#
//        #     |   |   |   |   |   %   |   |   |   |   |   #
//        #     |   | o | o | o |   %   |   |   |   |   |   #
//        #     |   |   |   |   |   %   |   |   |   |   |   #
//        #     +---+---+---+---+---%---+---+---+---+---+---#
//        #     |   |   |   |   |   %   |   |   |   |   |   #
//        #     |   | o | o | o | o % o | o | o | o | o | o #
//        #     |   |   |   |   |   %   |   |   |   |   |   #
//        #     +---+---+---+---+---%---+---+---+---+---+---#
//        #     |   |   |   |   |   %   |   |   |   |   |   #
//        #     |   |   |   |   |   %   | o | o | o | o | o #
//        #     |   |   |   |   |   %   |   |   |   |   |   #
//        #     +---+---+---+---+---%---+---+---+---+---+---#
//        #                         |   |   |   |   |   |   #
//        #                         |   | o | o | o | o | o #
//        #                         |   |   |   |   |   |   #
//        ###################################################
//
//  where:
//   #    are physical boundaries
//   | +  are cell boundaries of the two mesh boxes
//   %    are box boundaries shared between boxes
//   o    are cells in the proper nesting domain
//
//  The tagged cells outside the proper nesting domains are discarded by
//  intersecting the set of tagged cells at each level with the PND at that

//  level.  This ensures that all cells in the refined meshes are properly
//  nested.  The actual mesh generation starts at the top of the existing mesh
//  (\var{TopLevel}) and works down.  At each level a set of mesh boxes that
//  includes all the tagged cells at that level is generated (using
//  \func{makeBoxes}).  The goal is to minimize the number of boxes and the
//  amount of overlap and maximize the ratio of tagged to untagged cells in
//  each box.  The boxes are refined to the next higher level and saved as the
//  new mesh (in \var{newmeshes}).  The unrefined boxes are coarsened and
//  projected down to the next lower level where a one cell buffer is added
//  around all the boxes (the opposite of taking the interior cells that is
//  done in building the PND) and added to the tagged cells at the coarser
//  level.  This ensures that the next coarser level will have meshes that will
//  properly nest the meshes just generated.  This is repeated until the
//  BaseLevel+1 mesh is replaced.
//
//
// Implementation Method:
// ----------------------
//  There are 5 basic steps to perform.  Two functions (\func{makePNDs} and
//  \func{makeBoxes}) are called to handle the most complex steps and the code
//  in this function handles the rest.
//  Start
//   1) Compute the proper nesting domains (PNDs).  A \var{Vector<IntVectSet>}
//      is created and passed to \func{makePNDs}, along with the grid and
//      boundary data.  Only the part of the Vector that is active
//      [\var{BaseLevel:TopLevel}] is modified.  The other elements of the
//      vector are not touched.
//   2) Removing (clipping) the tag cells outside the PNDs is trivial:
//      intersect the two \var{IntVectSets} (PND and tag cells) at each level.
//      [Note: The PND data is not needed after this step.]
//  Loop for L from \var{TopLevel} downto \var{BaseLevel}
//   3) \func{makeBoxes} is called to find a set of boxes that cover the tag
//      cells at this level.  The output is stored in a temp \var{BoxArray}
//   4) The temp \var{BoxArray} is refined and the result is stored in the
//      output mesh vector at level L+1.
//   5) The temp \var{BoxArray} is expanded by the buffer cells in each direction
//      (clipping at the physical boundaries) and then coarsened and unioned
//      with the tags at level L-1.  This step does not have to be done for
//      the base level.  The resulting tags are used to generate the mesh
//      at the next coarser level.
//  EndLoop
//
// Implementation Notes:
// ---------------------
//  The output vector \var{newmeshes} is redefined to the necessary size
//  without necessarily reusing the existing memory space.
//
// Modification history:
//  dbs = David Serafini <dbs@anag.lbl.gov>
//  dm  = Dave Modiano
//  v0.1  dbs  Mar98  Initial design and coding
//  v0.2  dbs  Jun98  Fixed a bug in makeBoxes. 
//                    Changed some variable names to be more meaningfull.
//                    Changed interface to \func{findMaxInflectionPoint}.
//                    Renamed \func{print_pnd} to \func{print_ivs}.
//  v0.3  dbs  Jun98  Changed \func{splitTags} to use \func{IntVectSet::chop}.
//  v1.1  dbs  Jul98  Changed arglists for \func{meshRefine} and \func{makeBoxes}
//                    to allow \var{FillRatio} to be an argument instead of a constant.
//  v2.0  dbs  Nov98  Added consistency check on \var{Domains} and \var{RefRatios}.  
//                    Added \var{BlockFactors}, \var{MaxBoxSize} and \var{BufferSizes} arguments.
//                    Rearranged argument list (WARNING: not backward compatible!!).
//                    Changed some of the input parameters to scalars from vectors.
//  v2.1  dbs  Oct99  Changed BoxArray types to Vector<Box> to be consistent with
//       /dm          Chombo version 0.99.  Changed BL_SPACEDIM macro to SpaceDim constant.
//                    Changed error handling to use new Chombo error interfaces.
//  v2.2  dbs  Oct99  Bugfixes to handle boxes with negative indices.
//                    Changed the meshRefine() version that takes a scalar tags IVS
//                    so it refines the tags before passing them to the vector meshRefine.
//                    Added alignment check in \func{makePNDs}.
//  v2.3  dbs  Nov99  Changed the order of optional arguments in the API.
//  v2.5  dbs  Nov99  Allow TopLevel=-1 to return with no new grids.
//  v2.6  dbs  Dec99  Explicitly handle TopLevel<0 as a recoverable exception.
//
///////////////////////////////////////////////////////////////////////////////


// Include files:

#include "MeshRefine.H"
#include "MayDay.H"

// Prototypes of internal routines (not for external use)

static int
splitBox( Vector<Box> & boxes ,const int boxindex
         ,const int dimension ,const int maxboxsize ) ;

static Vector<int>
makeTrace( const IntVectSet& Ivs ,int Dir ) ;

static int
findSplit( const Vector<int>& Trace ) ;

static int
findMaxInflectionPoint( const Vector<int>& Trace ,int& MaxVal ) ;

static int
splitTags( const IntVectSet& Tags
          ,const int split_dir ,const int split_indx
          ,IntVectSet& tags_lo ,IntVectSet& tags_hi ) ;

void 
breakBoxes(Vector<Box>& a_vboxin,  const int& a_maxSize, const int& a_idir);
/*
unused
static int
minval( const Vector<int> & V );
*/
static int
maxloc( const int* V ,const int L ) ;


// Code:

//recursive function to enforce max size of boxes in a given direction
void 
breakBoxes(Vector<Box>& a_vboxin,  const int& a_maxSize, const int& a_idir)
{
  int nboxes = a_vboxin.size();
  //need to use STL vector for bools.
  using std::vector;
  vector<bool> splitDec(nboxes);
  bool anyBoxesToSplit = false;
  //find out which boxes need to be chopped and in what direction
  for(int ibox = 0; ibox < nboxes; ibox++)
    {
      if( a_vboxin[ibox].size(a_idir ) > a_maxSize ) 
        {
          splitDec[ibox] = true;
          anyBoxesToSplit = true;
        }
      else
        {
          splitDec[ibox] = false;
        }
    }
  //if there are no boxes to split, just return
  //otherwise, split all the boxes that need to be 
  //split ONCE and then call function recursively
  // and set the return vector to the temporary after
  // the recursion
  if(anyBoxesToSplit)
    {
      Vector<Box> vboxtemp;
      for(int ibox = 0; ibox < nboxes; ibox++)
        {
          Box boxone = a_vboxin[ibox];
          if(splitDec[ibox])
            {
              int len = (boxone.smallEnd(a_idir) + 
                         boxone.bigEnd(a_idir))/2;
              Box boxtwo = boxone.chop(a_idir, len);
              vboxtemp.push_back(boxone);
              vboxtemp.push_back(boxtwo);
            }
          else
            {
              vboxtemp.push_back(boxone);
            }
        }
      breakBoxes(vboxtemp, a_maxSize, a_idir);
      a_vboxin = vboxtemp;
    }
  return;
}
int domainSplit(const Box& domain, Vector<Box>& vbox, int maxsize, int iblock)
{
  assert(!domain.isEmpty());
  assert(maxsize > 0);
  int iref = 1;
  Box coardom = coarsen(domain, iref);
  Vector<Box> domains(2);
  domains[0] = coardom;
  domains[1] = domain;
  Vector<Box> oldGrid0(1, coardom);
  Vector<Vector<Box> > oldMeshes(1);
  oldMeshes[0] = oldGrid0;
  if(maxsize/iblock/iref < 1)
    MayDay::Error("DomainSplit: maxsize and blocking factor incompatible");
  Vector<Vector<Box> > newMeshes;
  Vector<int> refratio(1,iref);
  IntVectSet tags(coardom);
  // make irreg_boxes using berger-rigoutsos
  Real fillrat = 0.7;
  int buffersize = 1;
  int block_factor= iblock;
  Box testdom = coarsen(domain, iref*block_factor);
  testdom.refine(iref*block_factor);
  if(domain != testdom)
    MayDay::Error("DomainSplit:domain and Blocking Factor incompatible");
  int eekflag = meshRefine(newMeshes, tags, 0, 0, domains, 
                           oldMeshes, refratio,
                           fillrat, block_factor, 
                           buffersize,  maxsize);
  if(eekflag != 0)
    MayDay::Error("domainsplit::Problem in meshrefine");

  assert(newMeshes.size() == 2);
  vbox = newMeshes[1];
  return 0;
}
//
// This version of meshRefine takes a single IntVectSet of tags
//
int
meshRefine( Vector<Vector<Box> >&   newmeshes    //output: refined meshes
//                                               //      [0:TopLevel+1] on exit
           ,IntVectSet&             tags         //in_out: tagged cells on BaseLevel
           ,const int               BaseLevel    //input: index of base mesh level
           ,const int               TopLevel     //input: index of finest mesh
           ,const Vector<Box>&      Domains      //input: problem domain boundary
//                                               //      [BaseLevel:TopLevel]
           ,const Vector<Vector<Box> >& OldMeshes//input: existing meshes
//                                               //      [0:TopLevel]
           ,const Vector<int>&      RefRatios    //input: # of refined cells (in each
//                                               //      dimension) at finer level for
//                                                       each cell to be refined
//                                               //      [BaseLevel:TopLevel]
           ,const Real              FillRatio    //input,optional: range(0:1) minimum
//                                               //      acceptable ratio of tagged
//                                               //      cells in a box (default=0.75)
           ,const int               BlockFactor  //input,optional: mesh size granularity
//                                               //      and alignment constraint
//                                               //      (default = 1)
           ,const int               BufferSize   //input,optional: min number of level=L
//                                               //      mesh cells between edge of a 
//                                               //      level=L+1 box and a level=L-1
//                                               //      box  (default = 1)
           ,const int               MaxBoxSize   //input,optional: largest number of grid
//                                               //      points allowed in any dimension
//                                               //      of an individual box
//                                               //      (default = \infnty)
          )
{
    //
    // Handle special cases
    //
    if( TopLevel < 0 )                              return(  1001 ) ;
    if( BaseLevel > (TopLevel+1) || BaseLevel < 0 ) return( -1002 ) ;

    //
    // Create a vector of tags from the single set of tags that is given
    Vector<IntVectSet> tags_vector( TopLevel+1 ,tags );

    // Refine the levels above the base level
    for( int i = 1 ; i <= TopLevel ; i++ )
      {
         tags_vector[i] = refine(tags_vector[i-1], RefRatios[i-1] ) ;
      }

    // run the version of meshRefine that takes a vector of tags
    return meshRefine( newmeshes ,tags_vector 
		      ,BaseLevel ,TopLevel ,Domains ,OldMeshes ,RefRatios
		      ,FillRatio ,BlockFactor ,BufferSize ,MaxBoxSize ) ;
}




//
// This version of meshRefine takes a Vector of tags
//
int
meshRefine( Vector<Vector<Box> >&   newmeshes    //output: refined meshes
//                                               //      [0:TopLevel+1] on exit
           ,Vector<IntVectSet>&     tags         //in_out: tagged cells
//                                               //      [BaseLevel:TopLevel]
           ,const int               BaseLevel    //input: index of base mesh level
           ,const int               TopLevel     //input: index of finest mesh
           ,const Vector<Box>&      a_Domains      //input: problem domain boundary
//                                               //      [BaseLevel:TopLevel]
           ,const Vector<Vector<Box> >& a_OldMeshes//input: existing meshes
//                                               //      [0:TopLevel]
           ,const Vector<int>&      RefRatios    //input: # of refined cells (in each
//                                               //      dimension) at finer level for
//                                                       each cell to be refined
//                                               //      [BaseLevel:TopLevel]
           ,const Real              FillRatio    //input,optional: range(0:1) minimum
//                                               //      acceptable ratio of tagged
//                                               //      cells in a box (default=0.75)
           ,const int               a_BlockFactor  //input,optional: mesh size granularity
//                                               //      and alignment constraint
//                                               //      (default = 1)
           ,const int               BufferSize   //input,optional: min number of level=L
//                                               //      mesh cells between edge of a 
//                                               //      level=L+1 box and a level=L-1
//                                               //      box  (default = 1)
           ,const int               MaxBoxSize   //input,optional: largest number of grid
//                                               //      points allowed in any dimension
//                                               //      of an individual box
//                                               //      (default = \infnty)
          )
{
    int status ;

    //
    // Validate arguments and handle special cases
    //
    if( TopLevel < 0 )                              return(  1001 ) ;
    if( BaseLevel > (TopLevel+1) || BaseLevel < 0 ) return( -1002 ) ;
    if( a_OldMeshes.size() < TopLevel + 1 )           return( -1001 ) ;
    if(   a_Domains.size() < TopLevel + 1 )           return( -1001 ) ;
    if( RefRatios.size() < TopLevel + 1 )           return( -1001 ) ;
    if(      tags.size() < TopLevel + 1 )           return( -1001 ) ;
    if( tags[TopLevel].isEmpty() )               return( -1003 ) ;
    if( a_BlockFactor < 1 )                           return( -1002 ) ;
    if( BufferSize < 1 )                            return( -1002 ) ;
    if( MaxBoxSize < 2*BufferSize )                 return( -1002 ) ;
    if( a_BlockFactor > MaxBoxSize )                  return( -1002 ) ;
    if( FillRatio <= 0.0 || FillRatio > 1.0 )       return( -1002 ) ;


    //
    // reinitialize the output array
    //
    newmeshes.resize( TopLevel+2 ) ;

    //
    // Generate new meshes if requested.
    //
    if( TopLevel+1 > BaseLevel ) 
      {

        Box domaint = a_Domains[BaseLevel];
        Box testdom = coarsen(domaint, a_BlockFactor);
        testdom.refine(a_BlockFactor);
        if(domaint != testdom)
          MayDay::Error("MeshRefine:domain and Blocking Factor incompatible");
        // [NOTE: the following validations are expensive so only do them when
        //        debugging.]
  
#if ! defined( NDEBUG )
      // all existing meshes should be within the boundaries
      for( int i = BaseLevel ; i <= TopLevel ; i++ ) {
          Box minbox = a_OldMeshes[i][0];
          for (int ibox = 0; ibox < a_OldMeshes[i].size(); ++ibox)
          {
              minbox.minBox(a_OldMeshes[i][ibox]);
          }
          assert(a_Domains[i].contains(minbox));
      }
#endif
      // all tagged cells must be within the existing meshes
      // make sure that each level of \var{Domains} is consistent with the 
      // previous level and the refinement ratio.
      for( int i = BaseLevel ; i < TopLevel ; i++ ) 
        {
          if( a_Domains[i+1] != refine( a_Domains[i], RefRatios[i] ) ) 
            MayDay::Error("MeshRefine:domains and refratios incompatible");
        }
      //
      // coarsen old meshes, tags, problem domains by the blocking factor  
      Vector<Vector<Box> > OldMeshes = a_OldMeshes;
      Vector<Box> Domains = a_Domains;
      for (int level = 0; level < tags.size (); ++level)
        {
          int BlockFactor = (a_BlockFactor + RefRatios[level] - 1) / RefRatios[level];
          tags[level].coarsen (BlockFactor);
        }
      for (int level = 0; level < OldMeshes.size (); ++level)
        {
      //
      // coarsen blocking factor, rounding upwards
          int BlockFactor = (a_BlockFactor + RefRatios[level] - 1) / RefRatios[level];
          for (int box = 0; box < OldMeshes[level].size (); ++box)
            {
              OldMeshes[level][box].coarsen (BlockFactor);
            }
          Domains[level].coarsen (BlockFactor);
        }
      //
      // Compute proper nesting domains from BaseLevel up to TopLevel.
      // [NOTE: this leaves pnds[0:BaseLevel-1] empty.]
      //dtg
      Vector<IntVectSet> pnds( TopLevel+1 ) ;
      IntVectSet oldmesh_ivs;
      for (int box = 0; box < OldMeshes[BaseLevel].size(); ++box)
        {
          oldmesh_ivs |= OldMeshes[BaseLevel][box];
        }
      status = makePNDs( pnds ,BaseLevel ,TopLevel ,Domains , oldmesh_ivs,
                         RefRatios ,BufferSize ) ;

      //## -- C++ does not allow jumps around class instantiations
      //##if( status != 0 ) goto _Error1 ;
      if( status != 0 ) {
          if( status < -1000 ) return( -1101 ) ;
          else                 return( -11 ) ;
      }
      //
      // Clip the tagged cells to the proper nesting domains by
      // intersecting the two sets.
      //
      for( int lvl = BaseLevel ; lvl <= TopLevel ; lvl++ ) {
          tags[lvl] &= pnds[lvl] ;
      }
      //
      // Generate new meshes.
      //
  
      // At each level, starting at the top, generate boxes that cover the tag
      // cells using makeBoxes(), refine these boxes and save them in the
      // output \var{newmeshes}.  Take the unrefined boxes, add a buffer zone around
      // each box, coarsen, and union with the tag cells on the next coarser
      // level.  This modifies the tags variable.  To handle \var{BlockFactor},
      // coarsen everything before making the new meshes and then refine the 
      // resulting mesh boxes.
      Vector<Box> lvlboxes ;  // new boxes on this level
      for( int lvl = TopLevel ; lvl >= BaseLevel ; lvl-- ) {
          Box lvldomain ; // domain of this level

          lvldomain = Domains[lvl] ; 

          // Union the meshes from the previous level with the tags on this level
          // to guarantee that proper nesting is satisfied.  On the first
          // iteration this is a no-op because \var{lvlboxes} is empty.
          //[NOTE: for every iteration after the first, \var{lvlboxes} will
          //       already be coarsened by \var{BlockFactor} so it will be at the same
          //       refinement level as \var{tags[lvl]}, which has also been coarsened]
          for( int i = 0 ; i < lvlboxes.size() ; i++ )
              tags[lvl] |= lvlboxes[i] ;
  
      // coarsen blocking factor, rounding upwards
          int BlockFactor = (a_BlockFactor + RefRatios[lvl] - 1) / RefRatios[lvl];
          // make a new mesh at the same level as the tags
          status = makeBoxes( lvlboxes ,tags[lvl] ,pnds[lvl] ,lvldomain
                             ,MaxBoxSize/BlockFactor/RefRatios[lvl], FillRatio ) ;

          //split boxes to size
          int maxsize = MaxBoxSize/RefRatios[lvl]/BlockFactor;
          if(maxsize < 1)
            MayDay::Error("meshrefine: maxsize and blocking factor incompatible");

          for(int idir = 0; idir < SpaceDim; idir++)
            {
              breakBoxes(lvlboxes, maxsize, idir);
            }

          //##if( status != 0 ) goto _Error2 ;
          if( status != 0 ) {
              if( status < -1000 ) return( -1102 ) ;
              else                 return( -12 ) ;
          }
  
  
          // refine the new mesh and save it
          //[NOTE: we have to undo the coarsening by \var{BlockFactor} as well
          //       as refine to the next level.]
          //[NOTE: refine() operates in-place so copy first then refine()
          //       because the unrefined mesh will be needed later.]
          newmeshes[lvl+1] = lvlboxes ;
          for (int ibox = 0; ibox < newmeshes[lvl+1].size(); ++ibox)
          {
              newmeshes[lvl+1][ibox].refine( BlockFactor * RefRatios[lvl] ) ;
          }
          // Make the boxes ready for the next iteration.
          // Don't have to do this for the last iteration.
          if( lvl > BaseLevel ) 
            {
  
              // add the buffer cells to the unrefined new mesh, clip at the domain
              // boundaries and coarsen to the next level so it matches the tags on
              // that level so we can union them together in the next iteration.
              for( int i = 0 ; i < lvlboxes.size() ; i++ ) 
                {
                  lvlboxes[i].grow( Max( 1 ,BufferSize/BlockFactor ) ) ;
                  lvlboxes[i] &= lvldomain ;
                  lvlboxes[i].coarsen( RefRatios[lvl-1] ) ;
                }
  
          }
      }
    }

    //
    // Finally, copy the old mesh levels that didn't change.
    //
    for( int i = 0 ; i <= BaseLevel ; i++ )
        newmeshes[i] = a_OldMeshes[i] ;

    //
    // Done
    //
    return( 0 ) ;

//##// ------------------
//##// Exception handling
//##// ------------------
//##
//##_Error1:  //exception in computing proper nesting domains
//##
//##    // separate exceptions into validation exceptions (which are probably bugs)
//##    // and all others
//##    if( status < -1000 ) return( -1101 ) ;
//##    else                 return( -11 ) ;
//##
//##_Error2:  //exception making boxes to cover tagged cells
//##
//##    // separate exceptions into validation exceptions (which are probably bugs)
//##    // and all others
//##    if( status < -1000 ) return( -1102 ) ;
//##    else                 return( -12 ) ;

}// end of meshRefine()


///////////////////////////////////////////////////////////////////////////////
//
// Function:
// ---------
//  int makePNDs()
//
// Short Description:
// ------------------
//  Compute the proper nesting domains for a subset of the levels in a
//  multilevel adaptive mesh.
//
// Usage:
// ------
//  This is called by \func{meshRefine}.  It wasn't designed to be called
//  directly so if you do so you're on your own.  The output is a
//  \var{Vector<IntVectSet>}.  This function is called to compute the proper
//  nesting domains needed to do mesh refinement.  The function that calls this
//  one must have the mesh and problem domain data structures already defined for
//  all the levels and have allocated a vector<IntVectSet> for the proper nesting
//  domains.  This function computes the proper nesting domains of the levels to
//  be refined (i.e., the levels \var{BaseLevel} and above) and modifies the 
//  \var{pnds} array.  The output vector
//  is undefined for indices correpsonding to mesh levels below \var{BaseLevel}
//  (i.e. [0:\var{BaseLevel}-1]).
//
// Arguments:
// ----------
//  \var{pnds}                     type: Vector<IntVectSet>
//      (output, index[\var{BaseLevel}:\var{TopLevel}]) proper nesting domains
//
//  \var{BaseLevel}                type: int
//      (input, range[0:\var{TopLevel}]) index of the mesh level to use as the
//      source for the proper nesting domains
//
//  \var{TopLevel}                 type: int
//      (input, range[\var{BaseLevel}:\var{OldMeshes.size()-1]) index of the
//      finest mesh level to compute a PND for
//
//  \var{BaseMesh}                 type: Vector<Box>
//      (input) the boxes at level \var{BaseLevel} in the mesh hierarchy.
//
//  \var{Domains}                  type: Vector<Box>
//      [same as for \func{MeshRefine}.]
//
//  \var{RefRatios}                type: Vector<int>
//      [same as for \func{MeshRefine}.]
//
//  \var{BufferSize}              type: int
//      [same as for \func{MeshRefine}.]
//
// Returns:
// --------
//  An int giving an exception number is returned.  The argument
//  \var{pnds} is modified.
//
// Exceptions:
// -----------
//  Exception numbers are divided into 3 groups:
//       0 if no exceptions occurred;
//      <0 if an unrecoverable exception occurred;
//      >0 if a recoverable exception occurred.
//  Unrecoverable exceptions are futher divided into:
//      <-1000 if some input data (argument or global variable) is invalid;
//      >-1000 if an unpredicted exception occurred.
//  The exception values that this function returns are:
//
// References:
// -----------
//  See ??? for a description of Proper Nesting Domains.
//  meshRefine -- calls this function
//
// Numerical Algorithm:
// --------------------
//  The proper nesting domain (PND) of level L+1 is defined as the interior of
//  the PND for level L (the next coarser) except at the boundaries of the
//  problem domain (ie, the level 0 box), in which case the PNDs of levels L
//  and L+1 both contain the boundary.  The PND of the base level is defined as
//  the interior of the union of all the grid boxes in the base level.  Given
//  the PND of level L (denoted by PND[L]) this function computes PND[L+1] by
//  iterating over the coordinate directions and computing the cells to remove
//  from the proper nesting domain due to the boundaries in that direction.
//  For each direction the domain is shifted (first up, then down) and only the
//  part that is shifted _out_ of the PND but remains _inside_ the problem
//  boundary is kept and then shifted back into the PND and subtracted
//  (removed) from it.  So if the shift moves part of the PND outside the
//  problem boundary then no cells will be removed.  If the cells that get
//  shifted out of the PND remain inside the boundary then they will be removed
//  from the PND when they get shifted back.  When all the directions have been
//  handled, the PND is refined to make it correct for level L+1.  The PND of
//  the base level is the union of the mesh boxes in the base level and serves
//  as the starting point.  In set notation the operation is:
//
//            D - ( ( ( (D << d) - D ) * B ) >> d )
//
//  where:
//    d is the current direction (+i,-i,+j,etc)
//    B is the boundary of the problem domain at the current level
//    - is the subtraction (removal) operator
//    * is the intersection operator
//   << is the down-shift operator (shift in negative direction)
//   >> is the up-shift operator (shift in positive direction)
//
//
// Implementation Method:
//  The PNDs are implemented as a vector of \type{IntVectSet}s.  A scratch IntVectSet
//  is used to store the PND of the current level while it is being computed.
//  When the PND for a level is complete, it is refined and used for the next
//  level.  The base PND is computed as the union of the mesh boxes in the base
//  level mesh.  The PNDs for each level are stored in the appropriate elements
//  of the pnds vector as they are computed.  Levels below BaseLevel are not
//  modified.  The code loops through each level starting from Baselevel.  
//  Loop through each coordinate direction applying the operation defined in
//  the "Numerical Algorithm" section above.  Copy the final result into the
//  PNDs vector.  If an exception occurs, do not change PNDs on the level that
//  took the exception.
//
// Implementation Notes:
//  This assumes cell-centers are being manipulated, not vertices.  The whole
//  vector of pnds is passed even though only some are accessed
//  because it is more convenient since these variables are likely to exist
//  already.
//
// Modification history:
//  dbs = David Serafini <dbs@nersc.gov>
//  v0.1  dbs  Mar98  Initial design and coding
//  v2.0  dbs  Apr99  Changed arg list order to conform to ANAG conventions.
//                    Added \var{BufferSize} argument.
//  v2.2  dbs  Oct99  Check alignment of existing mesh with \var{BufferSize}
//
///////////////////////////////////////////////////////////////////////////////


/*  old approach based on IntVectSet operations.....

int
makePNDs( Vector<IntVectSet>&       pnds //output: proper nesting domains,
                                         //        for all levels
         ,const int            BaseLevel //input: index of base level
         ,const int             TopLevel //input: index of finest level
         ,const Vector<Box>&     Domains //input: boundary data for all levels
         ,const Vector<Box>&    BaseMesh //input: mesh at level \var{BaseLevel}
         ,const Vector<int>&   RefRatios //input: mesh refinement ratios
         ,const int           BufferSize //input: number of cells between PNDs
                                         //       at adjacent levels
    )
{
  //
  // Validate inputs
  //
  if( BaseLevel > TopLevel || BaseLevel < 0 ) return( -1002 ) ;
  if(   Domains.size() < TopLevel + 1 )       return( -1001 ) ;
  if( RefRatios.size() < TopLevel + 1 )       return( -1001 ) ;
  // all existing boxes on the base level must be aligned to \var{BuffSize}
  if( BufferSize > 1 ) {
      for( int ibox = 0 ; ibox < BaseMesh.size() ; ibox++ ) {
	  for( int idim = 0 ; idim < SpaceDim ; idim++ ) {
	      if(   BaseMesh[ibox].smallEnd( idim )   % BufferSize != 0 ||
		  ( BaseMesh[ibox].size    ( idim ) ) % BufferSize != 0 )
		  return -1103 ;
	  }
      }
  }


  //
  // Compute the proper nesting domain for each level by shifting and clipping.
  // Refine the result to the new mesh level and repeat.
  //



  // start the PND with all the boxes on the base level

 

  IntVectSet pnd_lvl;
  for (int ibox = 0; ibox < BaseMesh.size(); ibox++)
  {
      pnd_lvl |= BaseMesh[ibox];  //union
  }

  // old approach with IntVectSet operations.....

  IntVectSet tmp_pnd ;            //scratch variable for expressions
  IntVect shift ;                 //amount to shift domains by

  
  for( int lvl = BaseLevel ; lvl <= TopLevel ; lvl++ ) {
    for( int dim = 0 ; dim < SpaceDim ; dim++ ) {

      // set the shift amount for a move in this dimension
      shift = BASISV( dim ); //#test#* BufferSize ;

      //#test#
      // the basic algorithm can fail for \var{BufferSize}>1, so iterate
      // \var{BufferSize} times using a buffer size of 1.
      for( int iter = 0 ; iter < BufferSize ; iter++ ) {
      //#test.end#
      // adjust the PND by removing points in the "up" direction
      //    PND = PND - ( ( ( (PND << d) - PND ) * B ) >> d )
      tmp_pnd = pnd_lvl ;         //initialize
      tmp_pnd.shift( shift ) ;    //move all cells up
      tmp_pnd -= pnd_lvl ;        //take away unshifted cells, leaving
      //                          // cells at this face of the boxes
      tmp_pnd &= Domains[lvl] ;   //intersect with physical domain, keeping
      //                          // only cells inside the boundaries
      tmp_pnd.shift( -shift );    //move cells back into original locations
      pnd_lvl -= tmp_pnd ;        //take remaining cells out of pnd


      // repeat in the 'down' direction
      //    PND = PND - ( ( ( (PND >> d) - PND ) * B ) << d )
      tmp_pnd = pnd_lvl ;         //initialize
      tmp_pnd.shift( -shift ) ;   //move all cells down
      tmp_pnd -= pnd_lvl ;        //take away unshifted cells, leaving
      //                          // cells at this face of the boxes
      tmp_pnd &= Domains[lvl] ;   //intersect with physical domain, keeping
      //                          // only cells inside the boundaries
      tmp_pnd.shift( shift );     //move cells back into original locations
      pnd_lvl -= tmp_pnd ;        //take remaining cells out of pnd

      } //#test#

    } //next dimension

    // save the result as the PND for this level
    pnds[lvl] = pnd_lvl ;

    // refine the result so it can be used for the next level
    pnd_lvl = pnd_lvl.refine( RefRatios[lvl] ) ;

  } //next level

  // Done.
  return( 0 ) ;
}
//end of makePNDs

*/

// new version of makePND based on simple IntVectSet function
int
makePNDs( Vector<IntVectSet>&       pnds //output: proper nesting domains,
                                         //        for all levels
         ,const int            BaseLevel //input: index of base level
         ,const int             TopLevel //input: index of finest level
         ,const Vector<Box>&     Domains //input: boundary data for all levels
         ,const Vector<Box>&    BaseMesh //input: mesh at level \var{BaseLevel}
         ,const Vector<int>&   RefRatios //input: mesh refinement ratios
         ,const int           BufferSize //input: number of cells between PNDs
                                         //       at adjacent levels
    )
{
  //
  // Validate inputs
  //
  if( BaseLevel > TopLevel || BaseLevel < 0 ) return( -1002 ) ;
  if(   Domains.size() < TopLevel + 1 )       return( -1001 ) ;
  if( RefRatios.size() < TopLevel + 1 )       return( -1001 ) ;
  // all existing boxes on the base level must be aligned to \var{BuffSize}
  if( BufferSize > 1 ) {
	for( int ibox = 0 ; ibox < BaseMesh.size() ; ibox++ ) {
	  for( int idim = 0 ; idim < SpaceDim ; idim++ ) {
		if(   BaseMesh[ibox].smallEnd( idim )   % BufferSize != 0 ||
			  ( BaseMesh[ibox].size    ( idim ) ) % BufferSize != 0 )
		  return -1103 ;
	  }
	}
  }

  IntVectSet& pnd_base = pnds[BaseLevel];
  pnd_base = IntVectSet();
  for (int ibox = 0; ibox < BaseMesh.size(); ibox++)
  {
      pnd_base |= BaseMesh[ibox];  //union
  }
  
  for( int lvl = BaseLevel ; lvl <= TopLevel ; lvl++)
	{
	  IntVectSet& pnd = pnds[lvl];
	  
	  pnd.nestingRegion(BufferSize, Domains[lvl]);
	
	  if( (TopLevel - lvl) > 0)
		{
		  pnds[lvl+1] = pnd;
		  pnds[lvl+1].refine(RefRatios[lvl]);
		}
	}
  
  return 0;
}

//
// it's the special bonus version of makePNDs that takes IntVectSets
// as input instead of Vector<Box>
//
/*   old version
int
makePNDs( Vector<IntVectSet>&       pnds //output: proper nesting domains,
                                         //        for all levels
         ,const int            BaseLevel //input: index of base level
         ,const int             TopLevel //input: index of finest level
         ,const Vector<Box>&     Domains //input: boundary data for all levels
         ,const IntVectSet&    BaseMesh //input: mesh at level \var{BaseLevel}
         ,const Vector<int>&   RefRatios //input: mesh refinement ratios
         ,const int           BufferSize //input: number of cells between PNDs
                                         //       at adjacent levels
    )
{
  //
  // Validate inputs
  //
  if( BaseLevel > TopLevel || BaseLevel < 0 ) return( -1002 ) ;
  if(   Domains.size() < TopLevel + 1 )       return( -1001 ) ;
  if( RefRatios.size() < TopLevel + 1 )       return( -1001 ) ;
  // all existing boxes on the base level must be aligned to \var{BuffSize}


  //
  // Compute the proper nesting domain for each level by shifting and clipping.
  // Refine the result to the new mesh level and repeat.
  //

  // start the PND with the entire base level
  IntVectSet pnd_lvl = BaseMesh;

  IntVectSet tmp_pnd ;            //scratch variable for expressions
  IntVect shift ;                 //amount to shift domains by

  for( int lvl = BaseLevel ; lvl <= TopLevel ; lvl++ ) {
    for( int dim = 0 ; dim < SpaceDim ; dim++ ) {

      // set the shift amount for a move in this dimension
      shift = BASISV( dim ); //#test#* BufferSize ;

      //#test#
      // the basic algorithm can fail for \var{BufferSize}>1, so iterate
      // \var{BufferSize} times using a buffer size of 1.
      for( int iter = 0 ; iter < BufferSize ; iter++ ) {
      //#test.end#
      // adjust the PND by removing points in the "up" direction
      //    PND = PND - ( ( ( (PND << d) - PND ) * B ) >> d )
      tmp_pnd = pnd_lvl ;         //initialize
      tmp_pnd.shift( shift ) ;    //move all cells up
      tmp_pnd -= pnd_lvl ;        //take away unshifted cells, leaving
      //                          // cells at this face of the boxes
      tmp_pnd &= Domains[lvl] ;   //intersect with physical domain, keeping
      //                          // only cells inside the boundaries
      tmp_pnd.shift( -shift );    //move cells back into original locations
      pnd_lvl -= tmp_pnd ;        //take remaining cells out of pnd


      // repeat in the 'down' direction
      //    PND = PND - ( ( ( (PND >> d) - PND ) * B ) << d )
      tmp_pnd = pnd_lvl ;         //initialize
      tmp_pnd.shift( -shift ) ;   //move all cells down
      tmp_pnd -= pnd_lvl ;        //take away unshifted cells, leaving
      //                          // cells at this face of the boxes
      tmp_pnd &= Domains[lvl] ;   //intersect with physical domain, keeping
      //                          // only cells inside the boundaries
      tmp_pnd.shift( shift );     //move cells back into original locations
      pnd_lvl -= tmp_pnd ;        //take remaining cells out of pnd

      } //#test#

    } //next dimension


    // save the result as the PND for this level
    pnds[lvl] = pnd_lvl ;

    // refine the result so it can be used for the next level
    pnd_lvl.refine( RefRatios[lvl] ) ;

  } //next level

  // Done.
  return( 0 ) ;
}

*/ 


int
makePNDs( Vector<IntVectSet>&       pnds //output: proper nesting domains,
                                         //        for all levels
         ,const int            BaseLevel //input: index of base level
         ,const int             TopLevel //input: index of finest level
         ,const Vector<Box>&     Domains //input: boundary data for all levels
         ,const IntVectSet&    BaseMesh //input: mesh at level \var{BaseLevel}
         ,const Vector<int>&   RefRatios //input: mesh refinement ratios
         ,const int           BufferSize //input: number of cells between PNDs
                                         //       at adjacent levels
    )
{
  //
  // Validate inputs
  //
  if( BaseLevel > TopLevel || BaseLevel < 0 ) return( -1002 ) ;
  if(   Domains.size() < TopLevel + 1 )       return( -1001 ) ;
  if( RefRatios.size() < TopLevel + 1 )       return( -1001 ) ;
  // all existing boxes on the base level must be aligned to \var{BuffSize}


  pnds[BaseLevel] = BaseMesh;
  
  for( int lvl = BaseLevel ; lvl <= TopLevel ; lvl++)
	{
	  IntVectSet& pnd = pnds[lvl];
	  
	  pnd.nestingRegion(BufferSize, Domains[lvl]);

	  if( (TopLevel - lvl) > 0)
		{
		  pnds[lvl+1] = pnd;
		  
		  pnds[lvl+1].refine(RefRatios[lvl]);
		}

	}
  
  return 0;
}
//end of makePNDs


///////////////////////////////////////////////////////////////////////////////
//
// Function: makeBoxes
//
// Short Description:
//  Construct a set of boxes that cover a set of tagged cells.
//
// Purpose:
//  Given a set of tagged cells defined on a single level of an AMR grid,
//  construct a BoxArray that covers all these cells that minimizes the
//  number of boxes nd maximizes the ratio of tagged cells in each box.
//  This is part of the process of refining a mesh hierarchy based on error
//  estimates.
// 
// Caveat:
//   The mesh that is  created must lie within the proper nesting domain (Pnd)
//
// Usage:
//  Arguments:
//  Returns: 0 if successful
//  Exceptions:
//   -1 if one of the inputs is invalid
//   -2 if the call to \func{splitTags} failed, possibly because of a coding bug here
//   -3 the ProperNestingDomain argument is empty, probably because of a coding bug in
//      \func{MeshRefine}
//   1001 if the \var{FillRatio} value was invalid and was changed to the default value
//
// Usage Notes:
//
// Numerical Algorithm:
//  <Berger-Rigoutsos algorithm>
//
// Implementation Method:
//
// Implementation Notes:
//  This gets tricky when the minbox around a set of tags has a non-zero offset.
//  Have to be careful to remember this when looking at box sizes.
//
// References:
//  M. Berger, I. Rigoutsos, "An Algorithm for Point Clustering and Grid Generation", 
//  _IEEE_Transactions_on_Systems,_Man,_and_Cybernetics_, Vol. 21, No.5, pp. 1278-1286, 
//  Sept./Oct. 1991.
//
// Modification history:
//  dbs = David Serafini <dbs@anag.lbl.gov>
//  dtg = Daniel T. Graves <graves@anag.lbl.gov>
//  dm  = Dave Modiano
//  v0.1  dbs  Mar98   Initial design and coding
//  v0.2  dbs  Jun98   Fix a bug where the split index was being incorrectly
//                     computed as \var{maxlen}/2.   Split the special cases
//                     handling to do \func{Tags.isEmpty} first, before checking
//                     \func{Domain.contains}.
//  v1.2  dbs  Jun98   Changed name of \func{findSplit} to \func{findHole}.
//  v1.3  dtg  Jun98   Added \var{Pnd} to arg list of \func{makeBoxes} and
//                     added a test to ensure that a mesh box is only accepted
//                     if it is within the proper nesting domain.
//  v2.0  dbs  Jul98   Added \var{FillRatio} to arg list to specify the ratio of
//                     how many tagged cells must be in a box before it can be
//                     accepted.
//  v2.0  dbs  Apr99   Changed order of arglist to correspond to ANAG conventions
//             Jun99   and added \var{MaxSize}.
//  v2.1  dm   Oct99   Removed references to \var{BoxArray} for Chombo v0.99.
//  v2.2  dbs  Oct99   Bugfixes for boxes with negative indices.
//
///////////////////////////////////////////////////////////////////////////////


// Constants for Berger-Rigoutsos algorithm

#if ! defined(_BR_MIN_INFLECTION_MAG_)
#define _BR_MIN_INFLECTION_MAG_ ( 3 )
#endif

// Code

int
makeBoxes( Vector<Box>&      mesh        //output boxes
          ,const IntVectSet& Tags        //set of tagged cells to cover
          ,const IntVectSet& Pnd         //proper nesting domain in which 
//                                       // the mesh boxes must live
          ,const Box&        Domain      //boundary of physical domain
          ,const int         MaxBoxSize  //largest number of cells in any dimension
//                                       // of a box
          ,Real              FillRatio   //lowest (#tags/boxsize) for which a box
//                                       // will be accepted (optional)
         )
{
    Box minbx ;                                  //min box around Tags
    int ier = 0 ,ierlast = 0 ;
    Vector<Box> mesh_lo ,mesh_hi ;               //boxes from recursion
    IntVectSet tags_lo ,tags_hi ;                //tags for recursion

    //
    // Handle special cases of no tags
    //
    if( Tags.isEmpty() ) {    //return null box
        mesh.resize(0) ;
        return( 0 ) ;
    }

    //
    // Validate inputs
    //
#ifndef NDEBUG
    IVSIterator j( Tags ) ;
    for( j.begin() ; j.ok() ; ++j )
        if( ! Domain.contains( j() ) ) goto _Error1 ;

    if( Pnd.isEmpty() ) goto _Error4;
#endif

    // The number of tagged cells in a box cannot exceed the number
    // of cells in the box so enforce an upper bound on \var{FillRatio}.
    //[NOTE: 0 or negative values are allowed -- they mean any box
    //       is acceptable.  This will probably be a mistake by the
    //       caller, but there is no obvious lower valid value for
    //       this variable (1e-6 is just as likely a mistake as 0).]
    if( FillRatio > 1.0 ) {
      FillRatio = _BR_MIN_BOX_FILL_RATIO_ ;  //[NOTE: passed by value so we can overwrite it.]

      ier = 1001 ;
    }

    //
    // Handle the case of all tags fitting in one box.
    // This always happens at the bottom of the recursion.
    //
    minbx = Tags.minBox() ;
    if( Tags.numPts() >= minbx.numPts() * FillRatio ) {
        // We want to add this minbox to the mesh because it has enough tagged
        // cells.  Make sure that it is properly nested.  If it is, return.
        // If not, continue with splitting it and recursing on the pieces.
        if( Pnd.contains(minbx)) {
            // no IntVects in the box are outside the PND so this box can
            // be accepted.  If it is larger than the maximum box size, split
            // it into two boxes
            //[NOTE: one of the boxes may violate the FillRatio requirement, but
            //       we will ignore this.]
            mesh.resize( 1 ); 
            mesh[0] = minbx ;
            for( int i = 0 ; i < SpaceDim ; i++ ) {
                // divide the box along this dimension until the box is small enough,
                // replacing the original box with one half and appending the second half
                // to the end of the vector and recursing on both halves
              int num_boxes = mesh.size();
              for (int box = 0; box < num_boxes; ++box)
                {
                  ierlast = splitBox( mesh , box ,i ,MaxBoxSize ) ;
                  if( ierlast != 0 ) ier = ierlast ;
                }
            }
            return( ier ) ;
        }
    }

    //
    // Compute the traces (signatures) of the minbox in each direction, and
    // find a hole in the trace (zero value) and an inflection point (zero
    // Laplacian) in each direction; keep the best of each.
    //[NOTE: both \func{find*} functions return -1 if nothing was found.]
    //[NOTE: \var{infl_val} is modified by \func{findMaxInflectionPoint}.]
    //[NOTE: \var{trace} indexes from 0, so indices into \var{trace} must 
    //       be shifted to be used as indices into the \var{Tags}.]
    //
    int hole_indx[SpaceDim] ,best_hole_dim       //holes in traces
       ,infl_indx[SpaceDim] ,best_infl_dim       //inflection points in traces
       ,infl_val [SpaceDim] ;                    //magnitudes of infl.points
    for( int idim = 0 ; idim < SpaceDim ; idim++ ) {
        Vector<int> trace = makeTrace( Tags ,idim ) ;
        hole_indx[idim] = findSplit( trace ) ;
        infl_indx[idim] = findMaxInflectionPoint( trace
                                                 ,infl_val[idim] ) ;
    }
    // Take the highest index as the best one because we want to take as large
    // a box as possible  (fewer large boxes are better than many small ones)
    best_hole_dim = maxloc( hole_indx ,SpaceDim ) ;
    best_infl_dim = maxloc( infl_indx ,SpaceDim ) ;

    //
    // Split the Tag set at a hole in one of the traces, if there is one, or an
    // inflection point in the Laplacian of the traces.  Failing that, split
    // at the middle of the longest dimension of the enclosing box.
    //
    if( hole_indx[best_hole_dim] >= 0 ) {
      // split at a hole in the trace, adjusting the trace index for the
      // offset into \var{Tags} indices
      ierlast = splitTags( Tags ,best_hole_dim
                          ,hole_indx[best_hole_dim] + minbx.smallEnd( best_hole_dim )
                          ,tags_lo ,tags_hi ) ;
      if( ierlast != 0 ) ier = ierlast ; if( ier < 0 ) goto _Error3 ;

    } else if( infl_indx[best_infl_dim] >= 0 ) {
      // split at an inflection point in the trace, adjusting the trace
      // index for the offset into \var{Tags} indices
      ierlast = splitTags( Tags ,best_infl_dim
                          ,infl_indx[best_infl_dim] + minbx.smallEnd( best_infl_dim ) 
                          ,tags_lo ,tags_hi ) ;
      if( ierlast != 0 ) ier = ierlast ; if( ier < 0 ) goto _Error3 ;

    } else {
      // split on the midpoint of the longest side of \var{minbx}, rounding up,
      // allowing for \var{minbx} to have a non-zero offset
      int longdim ;  //[NOTE: longdim is set by \func{longside}]
      minbx.longside( longdim ) ;  //ignore the return value
      ierlast = splitTags( Tags ,longdim
                          ,( minbx.smallEnd( longdim ) + minbx.bigEnd( longdim ) +1 )/2
                          ,tags_lo ,tags_hi ) ;
      if( ierlast != 0 ) ier = ierlast ; if( ier < 0 ) goto _Error3 ;
    }

    // Recurse on the two halves of the Tags 
    // assert( ! tags_lo.isEmpty() ) ;  assert( ! tags_hi.isEmpty() ) ;
    // wrong logic, you might have empty tags above a split
    if(!tags_lo.isEmpty())
      ier = makeBoxes( mesh_lo ,tags_lo ,Pnd ,Domain ,MaxBoxSize ,FillRatio ) ;
    if( ierlast != 0 ) ier = ierlast ; if( ier < 0 ) goto _Error2 ;
    if(! tags_hi.isEmpty() )
      ier = makeBoxes( mesh_hi ,tags_hi ,Pnd ,Domain ,MaxBoxSize ,FillRatio ) ;
    if( ierlast != 0 ) ier = ierlast ; if( ier < 0 ) goto _Error2 ;

    // combine the results into a single mesh
    mesh = mesh_lo ;
    for(unsigned int i=0; i<mesh_hi.size(); ++i) mesh.push_back(mesh_hi[i]) ;

    // Done
    return( ier ) ;

// Exceptions:

_Error1:
    MayDay::Error( "In makeBoxes: one or more tagged cells are outside the problem domain" );
    return( -1 ) ;
_Error2:
    MayDay::Error( "In makeBoxes: recursion failed on subset of tags" ) ;
    return( -2 ) ;
_Error3:
    MayDay::Error( "In makeBoxes: bug in argument(s) to splitTags()" ) ;
    return( -2 ) ;
_Error4:
    MayDay::Error( "In makeBoxes: empty Proper Nesting Domain argument" ) ;
    return( -3 ) ;
} //end of makeBoxes


///////////////////////////////////////////////////////////////////////////////
//
// Function: splitBox()
//
// Short description:
//  If the box in the given Vector pointed to by the given index is larger
//  than the given maximum size in the given dimension, split it into two
//  boxes and recurse on each.
//
///////////////////////////////////////////////////////////////////////////////

static int
splitBox( Vector<Box>& boxes
         ,const int BoxIndex
         ,const int Dim
         ,const int MaxBoxSize
    )
{
    // See if the box needs to be split in this dimension.  If not, do nothing.

  if( boxes[BoxIndex].size( Dim ) > MaxBoxSize ) 
    {
      // chop() returns the upper half as a new box (which is appended to
      // the vector) and modifies the original box to be the lower half
      int len = 
        ( boxes[BoxIndex].smallEnd( Dim )+ 
          boxes[BoxIndex].bigEnd(   Dim )+1 )/2;

      boxes.push_back(boxes[BoxIndex].chop( Dim , len)); 
      // recurse on the two boxes
      splitBox( boxes ,BoxIndex       ,Dim ,MaxBoxSize ) ;
      splitBox( boxes ,boxes.size()-1 ,Dim ,MaxBoxSize ) ;
    }

  // Done
  return 0 ;
}

///////////////////////////////////////////////////////////////////////////////
//
// Function: makeTrace()
//
// Short description:
//  Count the number of cells in the IVS at every index in the specified
//  coordinate direction.  In other words, make a histogram using the
//  \var{Dir}'th coordinate as data.
//
// Note: Vectors index from 0 so the trace indices have to be shifted from
//       the box indices by the value of the lower bound.
//
// References:
//  The trace computed here is the \Sigma value computed in Berger-Rigoutsos.
//
// Modification History:
//  v0.1  dbs  Mar98   Initial design and coding
//  v2.2  dbs  Oct99   Allow tags to have negative indices.
//
///////////////////////////////////////////////////////////////////////////////

static Vector<int>
makeTrace( const IntVectSet& Ivs ,int Dir )
{
    //
    // Validate inputs
    //
    assert( Dir >= 0 & Dir < SpaceDim ) ;

    //
    // Histogram the coordinates of \var{Ivs} in direction \var{Dir} into
    // \var{trace}.  The index range of \var{trace} must start at 0
    // regardless of the bounds of the minbox of \var{Ivs}
    //
    Box minbox = Ivs.minBox() ;
    int offset = minbox.smallEnd(Dir) ;
    Vector<int> trace( minbox.size(Dir) ,0 ) ;  //initialize to 0
    IVSIterator i(Ivs) ;

    for( i.begin() ; i.ok() ; ++i ) {
      trace[i()[Dir]-offset] += 1 ;
    }

    // Done
    return( trace ) ;
}

///////////////////////////////////////////////////////////////////////////////
//
// Function: findSplit()
//
// Short Description:
//  Given a trace (ie. signature) of a Tag set, find a place in the trace
//  with a zero count.  Assumes the first and last elements of the trace
//  are non-zero.
//
// Usage Note:
//  If there are no acceptable zeros in the trace, this returns -1.
//  \var{Vector}s index from 0, so the result may have to be shifted to be 
//  used as an index into the \var{IntVectSect} which produced the trace.
//
// Modification History:
//  v1.2  dbs  Jun98  Changed name from \func{findSplit} to \func{findHole}
//  v1.3  dbs  Jun98  Changed to ignore leading and trailing zeros in \var{Trace}
//  v2.2  dbs  Oct99  Removed code to handle leading zeros.
//
///////////////////////////////////////////////////////////////////////////////

static int
findSplit( const Vector<int>& Trace )
{
    // look for a place to split at -- begin at the index after the first nonzero
    for( int i = 1 ; i < Trace.size() -1 ; i++ ) {
        if( Trace[i] == 0 ) return( i ) ;
    }

    // nothing acceptable
    return( -1 ) ;
}


///////////////////////////////////////////////////////////////////////////////
//
// Function: findMaxInflectionPoint
//
// Short Description:
//  Find the largest inflection point in the given trace (ie. signature)
//  and return the index and set the value of the inflection in the last
//  argument.
//
// Usage Notes:
//  To be precise, the inflection point is _between_ the elements of the
//  trace, so we have to choose one.  By convention, we assume that the
//  index chosen will be used as the max index in the lower half (and that
//  the chosen+1 will be the first index in the upper half).  So we choose
//  the index that has the larger trace value as the inflection point.
//
//  If there are no inflection points or if none of them are larger than the
//  cutoff tolerance, -1 is returned.
//
// Implementation Note:
//  The smallest acceptable inflection magnitude is the global constant
//  _MIN_INFLECTION_MAG_
//
// Modification history:
//  dbs = David Serafini <dbs@nersc.gov>
//  v0.1  dbs  Mar98   Initial design and coding
//  v0.2  dbs  Jun98   Added 'const' to declaration of 'Trace' argument
//  v2.2  dbs  Oct99   Removed offset argument.  Added checking trace value
//                     when choosing the trace index that corresponds to the
//                     inflection point.
//
///////////////////////////////////////////////////////////////////////////////

static int
findMaxInflectionPoint( const Vector<int>& Trace ,int& MaxVal )
{
    // first compute the discrete Laplacian of the trace
    Vector<int> d2Trace( Trace.size() ,0 ) ;
    for( int i = 1 ; i < Trace.size()-1 ; i++ ) {
        d2Trace[i] = Trace[i-1] - 2*Trace[i] + Trace[i+1] ;
    }

    // find inflection points and save one with the largest magnitude
    int absval ,imax ;
    MaxVal = -1 ;
    for( int i = 2 ; i < Trace.size()-1 ; i++ ) {
        absval = abs( d2Trace[i-1]  - d2Trace[i] ) ;
        if( d2Trace[i-1] * d2Trace[i] < 0  && absval > MaxVal ) {
            imax = i ; MaxVal = absval ;
        }
    }

    // Find the largest inflection point, if one exists and
    // has magnitude large enough and return its index.
    // By convention, the index should indicate the trace element
    // with the larger count, so look at \var{imax} and \var{imax}-1.
    if( MaxVal == -1 )                          return( -1 ) ;
    else if( MaxVal < _BR_MIN_INFLECTION_MAG_ ) return( -1 ) ;
    else if( Trace[imax] >= Trace[imax-1] )     return( imax ) ;
    else                                        return( imax-1 ) ;
}

///////////////////////////////////////////////////////////////////////////////
//
// Function: splitTags
//
// Purpose:
//  Given a direction and an index in that direction, split the IntVectSet of
//  tags into two sets (lo,hi) on either side of the index.  Tags that match
//  the index exactly go into the hi set.
//
// Implementation Method:
//  For each tag in the set, look at the coordinate in the direction specified,
//  and if it is <= the specified index, copy the tag into the lo set.  Else
//  copy it into the hi set.
//
// Usage Note:
//  It is acceptable for the split index to be outside of the box containing Tags.
//  This forces all the Tags to end up in one of the two output sets.
//
// Modification history:
//  dbs = David Serafini <dbs@nersc.gov>
//  v0.1  dbs  Mar98   Initial design and coding
//  v0.2  dbs  Jun98   Changed argument names to better describe the variables.
//  v0.3  dbs  Jun98   Use \func{IntVectSet::chop} instead of splitting by hand.
//
///////////////////////////////////////////////////////////////////////////////

static int
splitTags( const IntVectSet& Tags
          ,const int split_dir ,const int split_indx
          ,IntVectSet& tags_lo ,IntVectSet& tags_hi )
{
    // Validate inputs
    //[NOTE: it is ok if the split_indx is outside the box containing Tags.]
    if( split_dir < 0 || split_dir >= SpaceDim ) return( -1) ;
    if( Tags.isEmpty() ) return( -2 ) ;

    // Copy the whole set into tags_lo and extract the cells that aren't
    // below the split index into tags_hi, keeping the leftover in tags_lo
    // (i.e., the split_indx'th cells go into the high set).
    tags_lo = Tags ;
    tags_hi = tags_lo.chop( split_dir ,split_indx );

    // Done
    return( 0 ) ;
}

///////////////////////////////////////////////////////////////////////////////
//
// Function: maxloc
//
// Purpose: find the maximum value in a Vector<int> or an int[] and return its index.
//
// Usage Notes:
//  Returns -1 if the Vector has no entries.
//
///////////////////////////////////////////////////////////////////////////////

//#unused# static int
//#unused# maxloc( const Vector<int>& V )
//#unused# {
//#unused#     int imax = 0
//#unused#        ,size = V.size() ;
//#unused# 
//#unused#     switch( size ) {
//#unused#     case 0:
//#unused#         imax = -1 ; break ;
//#unused#     case 1:
//#unused#         imax = 0 ;  break ;
//#unused#     default:
//#unused#         for( int i=1 ; i<size ; i++ ) {
//#unused#             if( V[i] > V[imax] ) imax = i ;
//#unused#         }
//#unused#     }
//#unused#     return( imax ) ;
//#unused# }

static int
maxloc( const int* V ,const int Size )
{
    int imax = 0 ;
    for( int i=1 ; i<Size ; i++ ) if( V[i] > V[imax] ) imax = i ;
    return( imax ) ;
}

