/*  _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
/*

** This software is copyright (C) by the Lawrence Berkeley National
** Laboratory.  Permission is granted to reproduce this software for
** non-commercial purposes provided that this notice is left intact.
**  
** It is acknowledged that the U.S. Government has rights to this
** software under Contract DE-AC03-765F00098 between the U.S.  Department
** of Energy and the University of California.
**  
** This software is provided as a professional and academic contribution
** for joint exchange. Thus it is experimental, is provided ``as is'',
** with no warranties of any kind whatsoever, no support, no promise of
** updates, or printed documentation. By using this software, you
** acknowledge that the Lawrence Berkeley National Laboratory and Regents
** of the University of California shall have no liability with respect
** to the infringement of other copyrights by any part of this software.
*/
#ifndef _MESH_REFINE_OBJECT_H_
#define _MESH_REFINE_OBJECT_H_

#include "REAL.H"

template <class TTYPE> class Vector;
class Box;
class IntVectSet;

/// interface to meshRefine
/**
   This class is an interface to the {\tt MeshRefine} functions with
   stored parameters.  Also, it returns the number of levels that
   actually have grids.

   The stored parameters and their defaults values are: \\
   blocking factor, default=1\\
   maximum allowed grid size, default={\tt INT_MAX}\\
   buffer size, default=2\\
   fill ratio default={\tt _BR_MIN_BOX_FILL_RATIO_} (defined in {\tt MeshRefine.H})

   See the {\tt meshRefine} documentationation.

 */

class MeshRefineObject
{
public:
  MeshRefineObject();
  ~MeshRefineObject();

///{\bf Regrid functions}
///
/**
   Computes new grids given tags.  See the {\tt meshRefine}
   documentation.  Also, it returns the number of levels that actually
   have grids.  Levels might not have grids due to an absence of tags or
   a lack of valid space on the coarser level.

   {\bf Arguments:}\\
   a_new_finest_level (modified): the finest level of the new hierarchy.\\
   see {\tt meshRefine} documentation for others.

*/
  void
  regrid(Vector<Vector<Box> >& a_new_grids,
         int& a_new_finest_level,
         Vector<IntVectSet>& a_tags,
         int a_base_level,
         int a_top_level,
         const Vector<Box>& a_prob_domains,
         const Vector<Vector<Box> >& a_old_grids,
         const Vector<int>& a_ref_ratioes)
    const;

///
/**
   Computes new grids given tags.  If there are any tags, all of the
   new levels are expected to have grids.  See the {\tt meshRefine}
   documentation.

*/
  void
  regrid(Vector<Vector<Box> >& a_new_grids,
         IntVectSet& a_tags,
         int a_base_level,
         int a_top_level,
         const Vector<Box>& a_prob_domains,
         const Vector<Vector<Box> >& a_old_grids,
         const Vector<int>& a_ref_ratioes)
    const;
 
///{\bf Modification function}
///
/**
   Sets the blocking factor for subsequent calls to {\tt regrid(...)}.

   See the {\tt meshRefine} documentationation.

 */
  void
  blockFactor(int a_block_factor);

///
/**
   Sets the for subsequent calls to {\tt regrid(...)}.

   See the {\tt meshRefine} documentationation.

 */
  void
  maxSize(int a_max_size);

///
/**
   Sets the buffer size for subsequent calls to {\tt regrid(...)}.

   See the {\tt meshRefine} documentationation.

 */
  void
  bufferSize(int a_buffer_size);

///
/**
   Sets the fill ratio for subsequent calls to {\tt regrid(...)}.

   See the {\tt meshRefine} documentationation.

 */
  void
  fillRatio(Real a_fill_ratio);

/// {\bf Access functions}
///
/**
   Returns the current blocking factor parameter.

   See the {\tt meshRefine} documentationation.

 */
  int
  blockFactor() const;

///
/**
   Returns the current maximum grid size parameter.

   See the {\tt meshRefine} documentationation.

 */
  int
  maxSize() const;

///
/**
   Returns the current buffer size parameter.

   See the {\tt meshRefine} documentationation.

 */
  int 
  bufferSize() const;

///
/**
   Returns the current fill ratio parameter.

   See the {\tt meshRefine} documentationation.

 */
  Real
  fillRatio() const;

private:
  int m_block_factor;
  int m_max_size;
  int m_buffer_size;
  Real m_fill_ratio;
};

#endif
