#include <iostream>
#include <cstdio>
#include <string>
#include <vector>

#include "IntVect.H"
#include "Box.H"
#include "BoxIterator.H"
#include "FArrayBox.H"
#include "BoxLayout.H"
#include "DisjointBoxLayout.H"
#include "LayoutIterator.H"
#include "DataIterator.H"
#include "LevelData.H"

#include "ParmParse.H"

#include "ParmParseIO.H"

ParmParseIO::ParmParseIO () 
  : 
  m_parmparse_ptr ((ParmParse*) NULL),
  m_groupname ("")
{
}


ParmParseIO::~ParmParseIO ()
{
  if (m_parmparse_ptr != NULL)
  {
    delete m_parmparse_ptr;
    m_parmparse_ptr = (ParmParse*) NULL;
  }
}


void
ParmParseIO::openForRead (const std::string& a_filename)
{
  m_filename = a_filename;
  m_parmparse_ptr = new ParmParse (0, 
                                   (char**) NULL, 
                                   (char*) NULL, 
                                   a_filename.c_str ());
}

void
ParmParseIO::openForWrite (const std::string& a_filename)
{
  m_filename = a_filename;
  m_ostrm.open (a_filename.c_str ());
}

void
ParmParseIO::close ()
{
  m_filename = "";

  if (m_parmparse_ptr != NULL)
  {
    delete m_parmparse_ptr;
    m_parmparse_ptr = NULL;
  }

  if (m_ostrm.is_open ())
  {
    m_ostrm.close ();
  }
}

    const std::string&
    ParmParseIO::filename() const
{
    return (m_filename);
}


    const std::string&
    ParmParseIO::groupname() const
{
    return (m_groupname);
}



void
ParmParseIO::setGroup (const std::string& a_groupname)
{
  m_groupname = a_groupname;
}


const std::string
ParmParseIO::getGroup () const
{
  return (m_groupname);
}


void 
ParmParseIO::set (const std::string& a_name, const Real& a_value)
{
  if (m_groupname == "")
  {
    m_ostrm << a_name << " = " << a_value << endl << endl;
  }
  else
  {
    m_ostrm << m_groupname << "." << a_name << " = " << a_value << endl << endl;
  }
}

void 
ParmParseIO::set (const std::string& a_name, const int&  a_value)
{
  if (m_groupname == "")
  {
    m_ostrm << a_name << " = " << a_value << endl << endl;
  }
  else
  {
    m_ostrm << m_groupname << "." << a_name << " = " << a_value << endl << endl;
  }
}

void 
ParmParseIO::set (const std::string& a_name, const std::string&  a_value)
{
  if (m_groupname == "")
  {
    m_ostrm << a_name << " = " << a_value << endl << endl;
  }
  else
  {
    m_ostrm << m_groupname << "." << a_name << " = " << a_value << endl << endl;
  }
}

void 
ParmParseIO::set (const std::string& a_name, const IntVect&  a_value)
{
  if (m_groupname == "")
  {
    m_ostrm << a_name << " = " 
            << D_TERM ( a_value[0], << "  " << 
                        a_value[1], << "  " << 
                        a_value[2] ) << endl << endl;
  }
  else
  {
    m_ostrm << m_groupname << "." << a_name << " = " 
            << D_TERM ( a_value[0], << "  " << 
                        a_value[1], << "  " << 
                        a_value[2] ) << endl << endl;
  }
}


// this is only for cell-centered boxes
void 
ParmParseIO::set (const std::string& a_name, const Box&  a_value)
{
  if (m_groupname == "")
  {
    m_ostrm << a_name << " = " 
            << D_TERM ( a_value.smallEnd (0), << "  " <<
                        a_value.smallEnd (1), << "  " << 
                        a_value.smallEnd (2) ) 
            << "     "
            << D_TERM ( a_value.bigEnd (0), << "  " <<
                        a_value.bigEnd (1), << "  " << 
                        a_value.bigEnd (2) ) 
            << endl << endl;
  }
  else
  {
    m_ostrm << m_groupname << "." << a_name << " = " 
            << D_TERM ( a_value.smallEnd (0), << "  " <<
                        a_value.smallEnd (1), << "  " << 
                        a_value.smallEnd (2) ) 
            << "     "
            << D_TERM ( a_value.bigEnd (0), << "  " <<
                        a_value.bigEnd (1), << "  " << 
                        a_value.bigEnd (2) ) 
            << endl << endl;
  }
}

void 
ParmParseIO::get (const std::string& a_name, Real& a_value) const
{
  if (m_groupname == "")
  {
    m_parmparse_ptr->query (a_name.c_str (), a_value);
  }
  else
  {
    ParmParse pp (m_groupname.c_str ());
    pp.query (a_name.c_str (), a_value);
  }
}

void 
ParmParseIO::get (const std::string& a_name, int&  a_value) const
{
  if (m_groupname == "")
  {
    m_parmparse_ptr->query (a_name.c_str (), a_value);
  }
  else
  {
    ParmParse pp (m_groupname.c_str ());
    pp.query (a_name.c_str (), a_value);
  }
}

void 
ParmParseIO::get (const std::string& a_name, std::string&  a_value) const
{
  if (m_groupname == "")
  {
    m_parmparse_ptr->query (a_name.c_str (), a_value);
  }
  else
  {
    ParmParse pp (m_groupname.c_str ());
    pp.query (a_name.c_str (), a_value);
  }
}

void 
ParmParseIO::get (const std::string& a_name, IntVect&  a_value) const
{
  std::vector<int> vi (SpaceDim);
  if (m_groupname == "")
  {
    m_parmparse_ptr->queryarr (a_name.c_str (), vi, 0, SpaceDim);
  }
  else
  {
    ParmParse pp (m_groupname.c_str ());
    pp.queryarr (a_name.c_str (), vi, 0, SpaceDim);
  }
  D_EXPR ( a_value[0] = vi[0], a_value[1] = vi[1], a_value[2] = vi[2] );
}

// this is only for cell-centered boxes
void 
ParmParseIO::get (const std::string& a_name, Box&  a_value) const
{
  std::vector<int> vi (2*SpaceDim);
  if (m_groupname == "")
  {
    m_parmparse_ptr->queryarr (a_name.c_str (), vi, 0, 2*SpaceDim);
  }
  else
  {
    ParmParse pp (m_groupname.c_str ());
    pp.queryarr (a_name.c_str (), vi, 0, 2*SpaceDim);
  }
  IntVect small;
  D_EXPR ( small[0] = vi[0], 
           small[1] = vi[1], 
           small[2] = vi[2] );
  IntVect big;
  D_EXPR ( big[0] = vi[SpaceDim], 
           big[1] = vi[SpaceDim+1], 
           big[2] = vi[SpaceDim+2] );
  a_value = Box(small, big);
}


void
write (ParmParseIO& a_ppio, const BoxLayout& a_layout)
{
  const std::string old_groupname = a_ppio.m_groupname;
  a_ppio.m_groupname += ".boxlayout";

  const int num_boxes = a_layout.size ();
  char* box_str = new char[int (log10 (Real(num_boxes))+1)+2];

  a_ppio.set ("num_boxes", num_boxes);

  LayoutIterator lit = a_layout.layoutIterator ();
  for (lit.begin (); lit.ok (); ++lit)
  {
    sprintf (box_str, "%d", lit ().intCode ());
    std::string label = std::string ("box_") + box_str;
    a_ppio.set (label, a_layout[lit ()]);
  }
  delete[] box_str;

  a_ppio.m_groupname = old_groupname;
}


int
read (ParmParseIO& a_ppio, Vector<Box>& a_boxes)
{
  const std::string old_groupname = a_ppio.m_groupname;
  a_ppio.setGroup(old_groupname + ".boxlayout");

  if (!a_ppio.found ("num_boxes"))
  {
    return (1);
  }
  else
  {
    int num_boxes;
    a_ppio.get ("num_boxes", num_boxes);
    a_boxes.resize(num_boxes);
    char* box_str = new char[int (log10 (Real(num_boxes))+1)+2];

    for (int i = 0; i < num_boxes; ++i)
    {
      sprintf (box_str, "%d", i);
      std::string label = std::string ("box_") + box_str;
      if (!a_ppio.found(label))
      {
        return (2);
      }
      a_ppio.get (label, a_boxes[i]);
    }
    delete[] box_str;
    a_ppio.setGroup(old_groupname);

 
    return (0);
  }
}


bool
ParmParseIO::found (const std::string& a_name) const
{
  if (m_groupname == "")
  {
    return (m_parmparse_ptr->contains (a_name));
  }
  else
  {
    ParmParse pp (m_groupname.c_str ());
    return (pp.contains (a_name));
  }
}


void
writeFArrayBox(ostream& a_fabstrm,
               const FArrayBox& a_data)
{
  const Box domain = a_data.box();
  BoxIterator bit(domain);
  const int num_comps = a_data.nComp();
  for (int comp = 0; comp < num_comps; ++comp)
  {
    for (bit.begin(); bit.ok(); ++bit)
    {
      a_fabstrm << a_data(bit(), comp) << " ";
    }
    a_fabstrm << endl << endl;
  }
}


void
write (ParmParseIO& a_ppio, 
       const LevelData<FArrayBox>& a_data, 
       const std::string& a_name)
{
  const int num_comps = a_data.nComp();
  a_ppio.set("num_comps", num_comps);

  const IntVect& ghost = a_data.ghostVect();
  a_ppio.set("ghost_cells", ghost);

  const BoxLayout& layout = a_data.boxLayout();
  const int num_boxes = layout.size ();
  char* box_str = new char[int(log10(Real(num_boxes))+1)+2];
  DataIterator dit = layout.dataIterator();
  for (dit.begin(); dit.ok(); ++dit)
  {
    sprintf (box_str, "%d", dit().intCode());
    const std::string label = a_name + ".fabfile_" + box_str;
    const std::string fabfilename = 
      a_ppio.filename() + "." + 
      a_ppio.groupname() + "." + 
      label + ".fab";
    a_ppio.set(label, fabfilename);
    ofstream fabstrm (fabfilename.c_str());
    writeFArrayBox (fabstrm, a_data[dit()]);
    fabstrm.close();
  }
  delete[] box_str;
}


int
readFArrayBox(istream& a_fabstrm,
              FArrayBox& a_data)
{
  const Box domain = a_data.box();
  BoxIterator bit(domain);
  const int num_comps = a_data.nComp();
  for (int comp = 0; comp < num_comps; ++comp)
  {
    for (bit.begin(); bit.ok(); ++bit)
    {
      a_fabstrm >> a_data(bit(), comp);
    }
  }
  return (0);
}


int
read  (ParmParseIO& a_ppio, 
                  LevelData<FArrayBox>& a_data, 
                  const std::string& a_name,
                  const DisjointBoxLayout& a_layout)
{
  if (!a_ppio.found("num_comps"))
  {
    return (1);
  }
  else
  {
    int num_comps;
    a_ppio.get("num_comps", num_comps);

    IntVect ghost = IntVect::Zero;
    a_ppio.get("ghost_cells", ghost);

    a_data.define(a_layout, num_comps, ghost);

    const int num_boxes = a_layout.size ();
    char* box_str = new char[int(log10(Real(num_boxes))+1)+2];
    DataIterator dit = a_layout.dataIterator();
    for (dit.begin(); dit.ok(); ++dit)
    {
      sprintf (box_str, "%d", dit().intCode());
      const std::string label = a_name + ".fabfile_" + box_str;
      std::string fabfilename;
      a_ppio.get(label, fabfilename);
      ifstream fabstrm (fabfilename.c_str());
      const int fab_status = readFArrayBox (fabstrm, a_data[dit()]);
      if (fab_status != 0)
      {
        return (2);
      }
      fabstrm.close();
    }
    delete[] box_str;
  }
  return (0);
}


