// Pool.cpp


/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/

//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.


#include "Pool.H"
#include "MayDay.H"

// Constructor sets Pool parameters, but doesn't actually allocate any memory.
Pool::Pool(int ptrSize, int poolSize, int alignment)
	: pool_(0),              // An array of pointers to pools
	ptrSize_(ptrSize),       // Chunk size (bytes) in each pool
	poolSize_(poolSize),     // Number of chunks in each pool
	alignment_(alignment),   // allocated memory to be aligned on this boundary.
	next_(0)                 // Points to next available chunk or is null.
{
	if (poolSize_ < 1)
		MayDay::Error("Pool::Pool non positive poolSize.");
	if ((unsigned int)ptrSize_ < sizeof(void*))     // Minimum chunk size is size of pointer.
		ptrSize_ = sizeof(void*);
	if ((unsigned int)alignment_ < sizeof(int))   // alignment_ checked out only for 4 and 8.
		alignment_ = sizeof(int);
	while (ptrSize_ % alignment_ != 0)  // chunk size must be multiple of alignment_
		ptrSize_++;
}

Pool::~Pool() {
	for (int i = 0; i < pool_.size(); i++)
		delete [] pool_[i];
}

void* Pool::getMoreMemory() {
	// Create another pool
	char* po;
	if (alignment_ == sizeof(int))
	  po = (char*)new int [ptrSize_ * poolSize_ / sizeof(int)];
 	else if (alignment_ == sizeof(double))
	  po = (char*)new double [ptrSize_ * poolSize_ / sizeof(double)];
	else
	  MayDay::Error("Pool::Pool : alignments other than int or double not supported");
	
	pool_.push_back(po);
	// Thread pool:
	// The first few bytes in each chunk is interpreted as a pointer
	// which points to the next free chunk.  With multiple pools, and random
	// news and deletes, the pools will become threaded together.  This is ok.
	char* p = po;
	for (int i = 0; i < poolSize_-1; i++) {
		*(char**)p = p + ptrSize_;  // Chunk at i points to chunk at i+1.
		p += ptrSize_;
	}
	*(char**)p = 0;  // Chunk at end of list points to null
	return po;       // return first chunk.
}

/// getPtr and returnPtr must be as fast as possible!

void* Pool::getPtr() {

  if (next_ == 0) next_ = getMoreMemory(); 
  void* result = next_;  // result points to first free chunk in list
  next_ = *(void**)next_;  // point next_ to next free chunk in list.
  return result;
}

///
void Pool::returnPtr(void* ptr) {

  *(void**)ptr = next_;  // Stamp *ptr with the first free chunk.
  next_ = ptr;           // Make ptr the first free chunk.

}
