/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/

/*
** This software is copyright (C) by the Lawrence Berkeley National
** Laboratory.  Permission is granted to reproduce this software for
** non-commercial purposes provided that this notice is left intact.
**  
** It is acknowledged that the U.S. Government has rights to this
** software under Contract DE-AC03-765F00098 between the U.S.  Department
** of Energy and the University of California.
**  
** This software is provided as a professional and academic contribution
** for joint exchange. Thus it is experimental, is provided ``as is'',
** with no warranties of any kind whatsoever, no support, no promise of
** updates, or printed documentation. By using this software, you
** acknowledge that the Lawrence Berkeley National Laboratory and Regents
** of the University of California shall have no liability with respect
** to the infringement of other copyrights by any part of this software.
**  
*/

#ifndef CH_REALVECT_H
#define CH_REALVECT_H

//
// $Id: RealVect.H,v 1.3 2000/12/01 19:45:13 graves Exp $
//

#include <cstddef>
#include <cstdlib>
#include <cstring>
#include <iostream>


class HDF5Handle;

#include "SPACE.H"
#include "Misc.H"
#include "REAL.H"
#include <cassert>

//
//@Man:
//@Memo: An Integer Vector in SpaceDim-Dimensional Space
/*@Doc:

  The class RealVect is an implementation of an Real vector in a
  SpaceDim-dimensional space.  
  RealVect values are accessed using the operator[] function, as for a normal
  C++ array.  In addition, the basic arithmetic operators have been overloaded
  to implement scaling and translation operations.
*/

class RealVect
{
public:

  /// {\bf Constructors and Accessors}

  ///
  /**
     Construct an RealVect whose components are zero.

  */
  RealVect ();

  ///
  /**
     Construct an RealVect given the specific values for its
     coordinates.  D\_DECL is a macro that sets the constructor to
     take CH\_SPACEDIM arguments.  

  */
  RealVect (D_DECL(int i, int j, int k));

  ///
  /**
     The copy constructor.

  */
  RealVect (const RealVect& rhs);

  ///
  /**
     The assignment operator.

  */
  RealVect& operator= (const RealVect& rhs);

  ///
  /**
     Returns a modifiable lvalue reference to the i'th coordinate of the
     RealVect.

  */
  inline
  Real& operator[] (int i);

  ///
  /**
     Returns the i'th coordinate of the RealVect.

  */
  inline
  const Real& operator[] (int i) const;


  /// {\bf Comparison Operators}

  ///
  /**
     Returns true if this RealVect is equivalent to argument RealVect.  All
     comparisons between analogous components must be satisfied.  

  */
  bool operator== (const RealVect& p) const;

  ///
  /**
     Returns true if this RealVect is different from argument RealVect.
     All comparisons between analogous components must be satisfied.  

  */
  bool operator!= (const RealVect& p) const;

  ///
  /**
     Modifies this RealVect by addition of a scalar to each component.

  */
  RealVect& operator+= (Real s);

  ///
  /**
     Modifies this RealVect by component-wise addition with argument.

  */
  RealVect& operator+= (const RealVect& p);

  ///
  /**
     Returns component-wise sum of this RealVect and argument.

  */
  RealVect operator+ (const RealVect& p) const;

  ///
  /**
     Return an RealVect that is this RealVect with a scalar added to
     each component.

  */
  RealVect operator+ (Real s) const;

  ///
  /**
     Modifies this RealVect by subtraction of a scalar from each
     component.

  */
  RealVect& operator-= (Real s);

  ///
  /**
     Modifies this RealVect by component-wise subtraction with argument.

  */
  RealVect& operator-= (const RealVect& p);

  ///
  /**
     Returns component-wise difference of this RealVect and argument.
  */
  RealVect operator- (const RealVect& p) const;

  ///
  /**
     Returns an RealVect that is this RealVect with a scalar s subtracted
     from each component.

  */
  RealVect operator- (Real s) const;
 
  ///
  /**
     Modifies this RealVect by multiplication of a scalar to each
     component.  

  */
  RealVect& operator*= (Real s);

  ///
  /**
     Modifies this RealVect by component-wise multiplication with argument.

  */
  RealVect& operator*= (const RealVect& p);

  ///
  /**
     Returns component-wise product of this RealVect and argument.

  */
  RealVect operator* (const RealVect& p) const;

  ///
  /**
     Returns an RealVect that is this RealVect with each component
     multiplied by a scalar.

  */
  RealVect operator* (Real s) const;

  ///
  /**
     Modifies this RealVect by division by a scalar of each component.

  */
  RealVect& operator/= (Real s);

  ///
  /**
     Modifies this RealVect by component-wise division with RealVect
     argument.

  */
  RealVect& operator/= (const RealVect& p);

  ///
  /**
     Returns component-wise division of this RealVect by RealVect argument.

  */
  RealVect operator/ (const RealVect& p) const;

  ///
  /**
     Returns component-wise division of this RealVect by scalar s.

  */
  RealVect operator/ (Real s) const;


  ///
  /**
   Returns a basis vector in the given coordinate direction; eg.
   RealVect3d BASISV(1) == (0,1,0).  Note that the coordinate
   directions are zero based.  

  */
  friend inline RealVect BASISREALV(int dir);

  ///
  /**
     This is an RealVect all of whose components are equal to zero.
     
  */
  static const RealVect Zero;

  ///
  /**
     This is an RealVect all of whose components are equal to one.

  */
  static const RealVect Unit;

protected:

  Real vect[SpaceDim];

};


#endif /*CH_RealVect_H*/

