
/*  _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.

// RefCountedPtr.h
// ==================


/*
	A reference counting handle class.
	
	This is to be used as a pointer to class T.  
	This will feel and smell just like a built-in pointer except:
	1.  There is no need to call delete on the pointer.
	2.  The default copy constructor and assignment implement ref-counting.
	3.  The user may call isNonUnique to determine if this pointer is
	    the only pointer to the data.  This can be used to hide the
	    ref-counting behavior of a class.
	4.  Checks for dereference of a null pointer.
*/

#ifndef REFCOUNTEDPTR_H
#define REFCOUNTEDPTR_H

#include "MayDay.H"

//
template <class T>
class RefCountedPtr {
public:
	//
	inline RefCountedPtr(T* ptr = 0);
	//
	inline RefCountedPtr(const RefCountedPtr<T>& other);
	//
	inline ~RefCountedPtr();
	//
	inline const RefCountedPtr<T>& operator =(const RefCountedPtr<T>& rhs);
	//
	inline T* operator ->();
	//
	inline bool isNull() const;
	//
	inline const T* operator ->() const;
	//
	inline T& operator *();
	//
	inline const T& operator *() const;
	//
	// auto conversion to regular const pointer where required.
	inline operator const T* () const;
	//
	inline bool isNonUnique() const;
	
	inline int refCount() const;
	
protected:
	T* ptr_;
	int* refCount_;
private:
};

template <class T> inline
RefCountedPtr<T>::RefCountedPtr(T* ptr)
	: ptr_(ptr),
		refCount_(0)
{
	if (ptr_) {
		refCount_ = new int;
		if (refCount_ == 0)
			MayDay::Error("RefCountedPtr::RefCountedPtr(T* ptr) out of memory");
		*refCount_ = 1;
	}
}

template <class T> inline
RefCountedPtr<T>::RefCountedPtr(const RefCountedPtr<T>& other)
	: ptr_(other.ptr_),
		refCount_(other.refCount_)
{
	if (refCount_ != 0)
		++(*refCount_);
}

template <class T> inline
RefCountedPtr<T>::~RefCountedPtr() {
	if (refCount_ != 0 && --(*refCount_) == 0) {
		delete ptr_;
		ptr_ = 0;
		delete refCount_;
		refCount_ = 0;
	}
}

template <class T> inline
const RefCountedPtr<T>& RefCountedPtr<T>::operator =(const RefCountedPtr<T>& rhs) {
	if (ptr_ != rhs.ptr_) {
		if (refCount_ != 0 && --(*refCount_) == 0) {
			delete ptr_;
			delete refCount_;
		}
		ptr_ = rhs.ptr_;
		refCount_ = rhs.refCount_;
		if (refCount_ != 0)
			++(*refCount_);
	}
	return *this;
}

template <class T> inline
bool RefCountedPtr<T>::isNull() const {
  return (ptr_ == 0);
}

template <class T> inline
T* RefCountedPtr<T>::operator ->() {
	if (ptr_ == 0)
		MayDay::Error("RefCountedPtr<T>::operator ->() on null pointer");
	return ptr_;
}

template <class T> inline
const T* RefCountedPtr<T>::operator ->() const {
	if (ptr_ == 0)
		MayDay::Error("RefCountedPtr<T>::operator ->() on null pointer");
	return ptr_;
}

template <class T> inline
T& RefCountedPtr<T>::operator *() {
	if (ptr_ == 0)
		MayDay::Error("RefCountedPtr<T>::operator *() on null pointer");
	return *ptr_;
}

template <class T> inline
const T& RefCountedPtr<T>::operator *() const {
	if (ptr_ == 0)
		MayDay::Error("RefCountedPtr<T>::operator *() on null pointer");
	return *ptr_;
}

template <class T> inline
RefCountedPtr<T>::operator const T* () const {
	return ptr_;
}

template <class T> inline
bool RefCountedPtr<T>::isNonUnique() const {
	return refCount_ == 0 ? false : *refCount_ != 1;
}

template <class T> inline
int RefCountedPtr<T>::refCount() const {
	return refCount_ == 0 ? 0 : *refCount_;
}

#endif
