/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
// SPMD.H
//
// These are the inevitable functions that people
// can't avoid using when writing a SPMD program.
// It is a minimal set from which more elaborate
// functionality can be generated.  As always, a
// user is free to utilize the entire MPI programming
// on their own platform.  The functions are 
// assured to work on all machines supported.

//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
#ifndef SPMD_H
#define SPMD_H

#include "REAL.H"
#include "Vector.H"
#include "Box.H"
#include "MayDay.H"
class IntVectSet;


/// local process ID
/** 
    Returns the ID of the locally running process in the range 0 <=
    procID() < numProc().  This has no relation to the operating system
    pid.  There is always a procID() == 0.  */
int procID();

/// number of parallel processes
/** 
    Returns the number of parallel processes running.
    Always returns at least 1. */
unsigned int numProc();


template <class T>
int linearSize(const T& inputT); 

template <class T>
void linearIn(T& a_outputT, const void* const inBuf); 

template <class T>
void linearOut(void* const a_outBuf, const T& inputT); 

///gather from multiple processes
/**
  Gathers a_input of every process onto Vector<T> a_outVec on process
  number a_dest.  It is an error if a_dest < 0 or a_dest >= numProc().
  a_outVec is numProc() long with the value of a_input on every
  process in its elements.\\ 

T must have have the following functions:
  \\
  
  int linearSize<T>(const T& inputT); \\
  void linearIn<T>(T& a_outputT, const void* const inBuf); \\
  void linearOut<T>(void* a_outBuf, const T& inputT); \\ 

*/
template <class T>
void gather(Vector<T>& a_outVec, const T& a_input, int a_dest);

template < >
void gather(Vector<IntVectSet>& a_outVec, 
            const IntVectSet& a_input, 
            int a_dest);

///broadcast to every process
/**
  Broadcasts a_inAndOut to every process from process number a_src.
  It is an error if a_src < 0 or a_src >= numProc().  \\

  T must have have the following functions: \\
  
  int linearSize<T>(const T& inputT); \\
  void linearIn<T>(T& a_outputT, const void* const inBuf); \\
  void linearOut<T>(void* a_outBuf, const T& inputT); \\ */
template <class T>
void broadcast(T& a_inAndOut,  int a_src);

//integer specialization of linearSize
template < >
int linearSize(const int& a_input); 

//integer specialization of linearIn
template < >
void linearIn(int& a_outputT, const void* const inBuf); 

//integer specialization of linearOut
template < >
void linearOut(void* const a_outBuf, const int& a_inputT); 

//Real specialization of linearSize
template < >
int linearSize(const Real& a_input); 

//Real specialization of linearIn
template < >
void linearIn(Real& a_outputT, const void* const a_inBuf); 

//Real specialization of linearOut
template < >
void linearOut(void* const a_outBuf, const Real& a_inputT); 

//Box specialization of linearSize
template < >
int linearSize(const Box& a_input); 

//Box specialization of linearIn
template < >
void linearIn(Box& a_outputT, const void* const a_inBuf); 

//Box specialization of linearOut
template < >
void linearOut(void* const a_outBuf, const Box& a_inputT); 

//Vector<T> specialization of linearSize
template <class T>
int linearSize(const Vector<T>& a_input); 

//Vector<T> specialization of linearIn
template <class T>
void linearIn(Vector<T>& a_outputT, const void* const a_inBuf); 

//Vector<T> specialization of linearOut
template <class T>
void linearOut(void* const a_outBuf, const Vector<T>& a_inputT); 

class SerialTask
{
public:
    enum task { compute=0 };
};

// return id of unique processor for special serial tasks
int
uniqueProc(const SerialTask::task& a_task);

#ifdef MPI
#include <mpi.h>
struct Chombo_MPI{ static MPI_Comm comm;};

#else
// this can be changed for debugging parallel code in serial
extern int num_procs ;
#endif

#include "SPMDI.H"

#endif // SPMD.H

