/*
** This software is copyright (C) by the Lawrence Berkeley National
** Laboratory.  Permission is granted to reproduce this software for
** non-commercial purposes provided that this notice is left intact.
**  
** It is acknowledged that the U.S. Government has rights to this
** software under Contract DE-AC03-765F00098 between the U.S.  Department
** of Energy and the University of California.
**  
** This software is provided as a professional and academic contribution
** for joint exchange. Thus it is experimental, is provided ``as is'',
** with no warranties of any kind whatsoever, no support, no promise of
** updates, or printed documentation. By using this software, you
** acknowledge that the Lawrence Berkeley National Laboratory and Regents
** of the University of California shall have no liability with respect
** to the infringement of other copyrights by any part of this software.
**  
*/

#ifndef _TREE_H
#define _TREE_H

#include <cstdlib>

#include "REAL.H"
#include "IntVect.H"
#include "BaseFab.H"
#include "BoxIterator.H"
#include "Vector.H"

//#include "TreeNode.H"

//// class Leaf
class Leaf
{
public:

    enum LeafType {Irregular=0, Regular, Covered,NUMLEAFTYPES};

    Leaf();
    ~Leaf();

// Locations in linear array corresponding to Irregular VoFs 
// located at this Leaf (defined only if lt == Irregular. 

    Vector<int> irregLocs;

    LeafType getLeafType();

// LeafType.
    LeafType lt;

};
/*
class TreeNode:: 

These classes implement a slightly specialized form of a
2^d-ary tree, where d = CH_SPACEDIM, used to represent irregular,
multiply-covered regions on a rectangular grid. There 
are a total of (depth + 1) levels, depth > 1, numbered from 
level = 0 (a subTree representing a Box containing a single cell)
to the maximum number of levels (at the root). Each TreeNode contains 
a Leaf lf, whose type can be Irregular, Regular, or Covered, or it can be 
undefined (hasLeaf == false).  An Irregular Leaf represents a
cell that is multiply-covered, and can only be be contained in 
a BaseTree with level==depth. Note that hasLeaf is always false 
for the root.

*/
class TreeNode
{
public:

    TreeNode();
// constructor for sub-tree
    TreeNode(TreeNode* Parent, const Box& Domain);
// destructor
    virtual ~TreeNode();

// Adds Leaf to tree, or augments irregular Leaf.
    virtual void addLeaf(const IntVect& iv, Leaf::LeafType lt);

// Adds Box to tree, or augments irregular Leaf.
    virtual void addBox(const Box& B, Leaf::LeafType lt);

// Gets Leaf corrsponding to input IntVect.
    Leaf& getLeaf(const IntVect& iv);

// Change non-Irregular leaf to Irregular leaf
    Leaf changeToIrregular(const IntVect& iv);

// Deletes all of lt in box b
    virtual void deleteBox(const Box& b, Leaf::LeafType lt);
    virtual void deleteBox(const Box& b);

// Deletes non-Irregular IntVect from TreeNode.
    virtual void deleteLeaf(const IntVect& iv);

// Does this location have a Leaf ?
    bool isLeaf(const IntVect& iv) const;

// irregular locations if vof type is Irregular
    Vector<int> getIrregLocs(const Leaf& lf) const; 
    Vector<int> getIrregLocs(const IntVect& iv) const;

    Box getDomain() const{return domain;}
// return irregular size
    int getIrregSize();

// WARNING: need to decide about permissions, access to data members.

    TreeNode* root;

// Pointer to tree that owns the parent.
    TreeNode* parent;

// children of *this. children.box() is a 2x2 / 2x2x2 box, except for
// the root, for which it can be any nonempty box.
// Leaf contained in *this; only either children or lf can be defined

    BaseFab<TreeNode* > children;

    int countChildren();

// Leaf member. Set to default at initialization.
    Leaf lf;

// Boolean that says whether this tree has a leaf.
    bool hasLeaf;

// depth of *this, where level==0 for root.
    int level;

// Box defining the part of the overall domain represented by the tree.
    Box domain;

// Pointer in a DLL to next tree containing leaf of this type.
    TreeNode* nextLeafLoc;

// Pointer in a DLL to previous tree containing leaf of this type.
    TreeNode* prevLeafLoc;

private:
    //nonfunctional copy constructors and = operators disallows them
    TreeNode(const TreeNode& ebis_in)
        {
            cout << "Tree has no copy constructor" << endl;
            abort();
        }
    void operator=(const TreeNode& ebis_in)
        {
            cout << "Tree has no = operator" << endl;
            abort();
        }
public:

// Refinement ratio between levels of the tree.
    int nRef;

// Number of Children that have covered leaves.
    long nCovered;

// Number of Children that have regular leaves.
    long nRegular;

// Number of Children that have irregular leaves.
    long nIrregular;

// Number of Children that have leaves.
    long nLeaves;

// Number of Children.
    int numChildren;

// log_2(nRef).
    int log2nref;

// 
    TreeNode* getTreeWithLeaf(const IntVect& Iv);

//
    const TreeNode* getTreeWithLeaf(const IntVect& Iv) const;

// 
    TreeNode* expandChildTree(const IntVect& Iv);

// Creates new levels 
    TreeNode* getNewLevel(int LBase, IntVect& Iv);
    bool hasChildren();
    IntVect getChildIndex(const IntVect& Iv);
    void expandTreeWithLeaf();

// Sets Leaf and associated DLL pointers.
    void setLeaf(Leaf::LeafType lt);

// Unsets DLL pointers.
    void unSetLeaf();

// Refinement ratio to leaf.
    int getNRefToLeaf(int level) const;
    void deleteChildren();

// Takes a set of children consisting entirely of leaves of the 
// same nonIrregular type and combines them into a single leaf.

    void agglomTree(Leaf::LeafType Lt);

// Increments Leaf counters.

    void incRCCount(long iinc, Leaf::LeafType Lt);

// Resets leaf counters.

    void resetRCCount();
};


class Tree : public TreeNode
{
    friend class LeafIterator;
public:
    Tree();

    ~Tree();
// Array of values for coarsening - only defined for Root Tree.
    Vector<int> refToLeafArray;

// Array of initial pointers for DLL - only defined for Root Tree.
    Vector<TreeNode*> startPtrs;

// Total number of irregular leaves - only define for root Tree.
// Specialized versions of {add,delete}{Box,Leaf} for root tree.

    int irregSize;

    int unitBoxSize;

    void addBox(const Box& B, Leaf::LeafType Lt);
    void deleteBox(const Box& B, Leaf::LeafType Lt);
    void deleteBox(const Box& B);
    void addLeaf(const IntVect& Iv,Leaf::LeafType Lt);
    void deleteLeaf(const IntVect& Iv);
    int getIrregSize();

private:

// Expands root Tree if B is not contained in domain.

    void expandTree(const Box& B,Leaf::LeafType Lt);

// Compresses root Tree if it has only one child.
    void compressTree();

};
class LeafIterator
{
public:

    LeafIterator(const Tree* Tr , Leaf::LeafType Lt);
    LeafIterator();
    void define(const Tree* Tr , Leaf::LeafType Lt);
    ~LeafIterator();
    void begin();
    bool ok();
    void next();
    const Box& getDomain();

protected:

    const Tree* rootPtr;
    TreeNode* currentPtr;
    Leaf::LeafType lt;

};
#endif
