/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
// Tree.cpp
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.

#include <cmath>
#include "Tree.H"

#if(CH_SPACEDIM == 2)
#define LOG2NREF 1
#endif
#if(CH_SPACEDIM == 3)
#define LOG2NREF 1
#endif



Leaf::Leaf()
{
}

Leaf::~Leaf() {}

// Default constructor for TreeNode - used only to construct root.

TreeNode::TreeNode()
{
}
TreeNode::TreeNode(TreeNode* Pt,const Box& Domain):log2nref(LOG2NREF)
{
    hasLeaf = false;
    domain = Domain;
    resetRCCount();
    parent=Pt;
    root = parent->root;
    assert(root->parent == root);
    nRef = root->nRef;
    level=(parent->level) - 1;
    numChildren = 0;
    parent->numChildren +=1;
    if (level > 0)
    {
        Box childDomain = coarsen(domain,getNRefToLeaf(level-1));
    	children.resize(childDomain,1);
    	children.setVal((TreeNode *) NULL);
    }
}

TreeNode::~TreeNode()
{
    if ((hasLeaf)&& (lf.lt != Leaf::Irregular))
    {
        unSetLeaf();
    }
    if (this != root)
    {
	parent->numChildren -= 1;
    }
}
Vector<int> 
TreeNode::getIrregLocs(const Leaf& Lf) const 
{
    return Lf.irregLocs;
}

Vector<int> 
TreeNode::getIrregLocs(const IntVect& Iv) const 
{
    if (hasLeaf) 
    {
        return lf.irregLocs;
    }
    else 
    {
	IntVect ivChild = coarsen(Iv,getNRefToLeaf(level-1));
	assert(children(ivChild) != (TreeNode *) NULL);
	return children(ivChild)->getIrregLocs(Iv);
    }
}

Leaf& 
TreeNode::getLeaf(const IntVect& Iv) 
{
    assert(domain.contains(Iv));
    if (hasLeaf) 
// TreeNode has a leaf. Return reference to that leaf.
    {
	return lf;
    }
// TreeNode does not have leaf. Find child that might contain leaf, 
// and make recursive call.
    else
    {
	IntVect ivChild = coarsen(Iv,getNRefToLeaf(level-1));
	assert(children(ivChild) !=  (TreeNode *) NULL);
	return children(ivChild)->getLeaf(Iv);
    }
}

void 
TreeNode::addLeaf(const IntVect& Iv, Leaf::LeafType Lt) 
{
    assert(domain.contains(Iv));
    if (hasLeaf)
    {
// TreeNode has a leaf. Increment lf.irregLocs if leaf is Irregular	
// and return reference to that leaf.
    	assert(lf.lt == Lt);
	if (Lt == Leaf::Irregular) 
	{
	    lf.irregLocs.push_back(((Tree *) root)->irregSize);
	    ((Tree *) root)->irregSize++;
	}
	return;
    }
    else 
    {
// TreeNode does not have leaf. 
	if (level > 0)

// level > 0, so have to add a leaf node to one of the children.
// Compute which child:
	{
	    IntVect ivChild = coarsen(Iv, getNRefToLeaf(level-1));

// Check to see whether appropriate child already exists. 
// If not, create one. 
            
	    if (children(ivChild) ==  (TreeNode *) NULL) 
	    {
	    	Box childDomain =
                    refine(Box(ivChild,ivChild),getNRefToLeaf(level-1));
		children(ivChild) = new TreeNode(this, childDomain);
		assert(numChildren == countChildren());
	    }

// Recursive call to addLeaf.
	    if ((Lt == Leaf::Irregular)) 
	    {
	        children(ivChild)->addLeaf(Iv,Lt);
                return;
	    }
	    else  
// Leaf is not Irregular : must add Leaf and check for agglomeratation.
	    {
		children(ivChild)->addLeaf(Iv,Lt);

// Now, check to see whether all of the child nodes have identical
// Regular or covered Leaves. If they do, delete the subtrees, and
// replace with a single leaf.

                agglomTree(Lt);

            }
        }
	else

// level = 0, but hasLeaf == false. Call setLeaf, and, if leaf
// is Irregular, call push_back on lf.irregLocs, and return reference to
// lf.

	{
	    setLeaf(Lt);
	    if (Lt == Leaf::Irregular)
	    {
	        lf.irregLocs.push_back(((Tree *) root)->irregSize);
	        ((Tree *) root)->irregSize++;
	    }
	    return;
	}
    }
}
void 
TreeNode::incRCCount(long iinc, Leaf::LeafType Lt)
{
    if (Lt == Leaf::Regular)
        nRegular = nRegular + iinc;
    if (Lt == Leaf::Covered)
        nCovered = nCovered + iinc;
    if (Lt == Leaf::Irregular)
        nIrregular = nIrregular + iinc;
    nLeaves = nLeaves + iinc;
}
void 
TreeNode::resetRCCount()
{
    nRegular = 0;
    nCovered = 0;
    nIrregular = 0;
    nLeaves = 0;
}
void
TreeNode::agglomTree(Leaf::LeafType Lt)
{
// Now, check to see whether all of the child nodes have identical
// Regular or covered Leaves. If they do, delete the subtrees, and
// replace with a single leaf.
    bool agglomFlag = false;
    if (this == root) return;
    const Box& b = children.box();
    assert( !b.isEmpty());
    assert( b.numPtsOK());
    long bcount = b.numPts();
    assert(nRegular >= 0);
    assert(nCovered >= 0);
    assert((nRegular+nCovered) <= bcount);
    
    if(Lt == Leaf::Regular)
    {
        if(nRegular == bcount) agglomFlag = true;
    }
    if(Lt == Leaf::Covered)
    {
        if(nCovered == bcount) agglomFlag = true;
    }
    if (agglomFlag) 
// Children can be combined into a single leaf. Delete children, and 
// call setLeaf for current leaf.
    {	
        deleteChildren();
        setLeaf(Lt);
        return;
    }
    else
// Children cannot be combined.
    {
        return;
    }
}

void TreeNode::addBox(const Box& B,Leaf::LeafType Lt)
{
    assert (Lt != Leaf::Irregular);
    assert (numChildren == countChildren());

// Can only add a box that is contained in the domain and for 
// a LeafType that is not irregular.

    if (hasLeaf) 
    {
    	assert(Lt == lf.lt);
    	return;
    }
    else if ((B == domain)&&(this != root))
    {
        deleteChildren();
    	setLeaf(Lt);
	return;
    }
    else
    {
        int icoarse =getNRefToLeaf(level-1);
	Box BChildren = coarsen(B,icoarse);
        BoxIterator bit(BChildren);
        for(bit.begin(); bit.ok(); ++bit)
        {
            IntVect iv = bit();
            Box biv(iv,iv);
            Box bChild = refine(biv,icoarse);
            Box binter= bChild;
            binter &= B;
            if (!binter.isEmpty())
            {
                if (children(iv) == (TreeNode *) NULL)
		{
		    assert(numChildren == countChildren());
                    children(iv) = new TreeNode(this,bChild);
		    assert(numChildren == countChildren());
		}
		assert (children(iv) != (TreeNode *) NULL);
                children(iv)->addBox(binter,Lt);
            }
        }
        // snip branches if possible
        if(level != 0)
            agglomTree(Lt);
    }
}
TreeNode* TreeNode::getTreeWithLeaf(const IntVect& Iv) 
{
    if (hasLeaf)
    {
        return this;
    }
    else 
    {
	IntVect ivChild;
	ivChild = coarsen(Iv, getNRefToLeaf(level-1));
        if (children(ivChild) != (TreeNode *) NULL)
        {
            return children(ivChild)->getTreeWithLeaf(Iv);
        }
        else
// Search of tree leads to child with (TreeNode*) NULL pointer. 
	{
	    return (TreeNode *) NULL;
	}
    }
}
const TreeNode* TreeNode::getTreeWithLeaf(const IntVect& Iv) const
{
    if (hasLeaf)
    {
        return this;
    }
    else 
    {
	IntVect ivChild;
	ivChild = coarsen(Iv, getNRefToLeaf(level-1));
        if (children(ivChild) != (TreeNode *) NULL)
        {
            return children(ivChild)->getTreeWithLeaf(Iv);
        }
        else
// Search of tree leads to child with (TreeNode*) NULL pointer. 
	{
	    return (TreeNode *) NULL;
	}
    }
}

TreeNode* TreeNode::expandChildTree(const IntVect& Iv) 
{
// Finds subtree containing leaf with Iv. If level > 0,
// splits leaf into subtrees each of which has leaf of same type as 
// original subtree, unSets itself and calls itself recursively on 
// subtree containing the leaf at Iv. If level == 0, returns this.

    if (level > 0)
    {
	assert(parent != (TreeNode *) NULL);
	Box b = children.box();

        BoxIterator bit(b);
        for(bit.begin(); bit.ok(); ++bit)
        {
            IntVect tiv = bit();
	    assert(children(tiv) == (TreeNode *) NULL);
            Box childDomain;
            childDomain = refine(Box(tiv,tiv),getNRefToLeaf(level-1));
            children(tiv) = new TreeNode(this, childDomain);
            assert(numChildren == countChildren());
            children(tiv)->setLeaf(lf.lt);
        }
	unSetLeaf();
	int refToLeaf = getNRefToLeaf(level-1);
    	IntVect ivChild = coarsen(Iv, refToLeaf);
        return children(ivChild)->expandChildTree(Iv);
    }
    else 
    {
	return this;
    }
}

Leaf TreeNode::changeToIrregular(const IntVect& Iv) 
{
    if (level > 0)
    {
// Set up for recursive call to changeToIrregular.
    	if (hasLeaf)
// current tree has leaf. Expand into subtrees.
	{
    	    assert(lf.lt != Leaf::Irregular);
	    expandTreeWithLeaf();
	}
// find child containing Iv, and call changeToIrregular
// recursively on that child.

    	IntVect ivChild = coarsen(Iv,getNRefToLeaf(level-1));
// There is no subtree past this point. This is an error; in that case,
// assert(0). We may wish to change this later, so that instead addLeaf
// is invoked.
    	assert(children(ivChild) != (TreeNode *) NULL);
        return children(ivChild) -> changeToIrregular(Iv);
    }
    else
    {
// Level == 0. unSet leaf, reset as Irregular, and return.
	unSetLeaf();
	setLeaf(Leaf::Irregular);
        lf.irregLocs.push_back(((Tree *) root)->irregSize);
        ((Tree *) root)->irregSize++;
	return lf;
    }
}
void TreeNode::deleteChildren() 
{
    assert (numChildren == countChildren());
    if (numChildren > 0)
    {
        const Box& b = children.box();
        BoxIterator bit(b);
        for (bit.begin(); bit.ok(); ++bit) 
        {
            IntVect iv = bit();
            if (children(iv) != (TreeNode *) NULL)
            {
                children(iv)->deleteChildren();
                delete children(iv);
                children(iv) = (TreeNode *) NULL;
    		assert (numChildren == countChildren());
            }
        }
    }
}
void TreeNode::deleteLeaf(const IntVect& Iv) 
{
    if (level > 0)
    {
// Set up for recursive call to deleteLeaf.
    	if (hasLeaf)
// current tree has leaf. Expand into subtrees.
	{
    	    assert(lf.lt != Leaf::Irregular);
	    expandTreeWithLeaf();
	}
// find child containing Iv, and call deleteLeaf recursively on that child.

    	IntVect ivChild = coarsen(Iv,getNRefToLeaf(level-1));
    	if (children(ivChild) !=  (TreeNode *) NULL)
    	{
    	    children(ivChild) -> deleteLeaf(Iv);
	}
	else
	{
// There is no subtree past this point. In that case, 
// deleteLeaf is a no-op, and we simply return.
	    return;
	}

// have completed recursive call to deleteLeaf. Now check to see
// whether child has any children. If not, the child can be deleted,
// and the pointer set to (TreeNode*) NULL.

        assert 
            ((children(ivChild)->numChildren) == children(ivChild)->countChildren());
        if (numChildren > 0)
	{
    	    if ((children(ivChild) != (TreeNode *) NULL) && 
                (children(ivChild)->numChildren == 0))
    	    {
    	    	delete children(ivChild);
    	    	children(ivChild) = (TreeNode *) NULL;
		assert(numChildren == countChildren());
	    }
	}
    }
    else
    {
// Level == 0. unSet leaf, and return.
	unSetLeaf();
	return;
    }
}


void TreeNode::deleteBox(const Box& B, Leaf::LeafType Lt)
{
    deleteBox(B);
}


void TreeNode::deleteBox(const Box& B)
{
// delete Box B from tree. We make no assumptions that the 
// box is contained in domain.
    if (level > 0)
    {
// Set up for recursive call to deletebox.
    	if (hasLeaf)
// current tree has leaf. Expand into subtrees.
	{
    	    assert(lf.lt != Leaf::Irregular);
            if(B.contains(domain))
            {
                unSetLeaf();
                return;
            }
            else
            {
                expandTreeWithLeaf();
            }
	}

	Box BChildren = children.box();
        BoxIterator bit(BChildren);
        for(bit.begin(); bit.ok(); ++bit)
        {
            IntVect iv = bit();
	    Box bChild = refine(Box(iv,iv),getNRefToLeaf(level-1));
	    if (children(iv) !=  (TreeNode *) NULL)
	    {
		if (B.intersects(bChild))
		{
                    Box bdel  = B&bChild;
	    	    children(iv)->deleteBox(bdel);
		}

// have completed recursive call to deleteBox. Now check to see
// whether child has any children. If not, the child can be deleted,
// and the pointer set to (TreeNode*) NULL.

    		if (numChildren > 0)
		{
    		    if ((children(iv) != (TreeNode *) NULL) && 
            		(children(iv)->numChildren == 0)
			&& !(children(iv)->hasLeaf) )
    		    {
    	    		delete children(iv);
    	    		children(iv) = (TreeNode *) NULL;
			assert(numChildren == countChildren());
		    }
    		}
	    }
	}
    }
    else
    {
// Level == 0. unSet leaf, and return.
	unSetLeaf();
	return;
    }
}
TreeNode*
TreeNode::getNewLevel(int LBase,IntVect& Iv)
{
    assert (numChildren == countChildren());
    if (numChildren == 0)
    {
	Box childrenBox = coarsen(domain,getNRefToLeaf(level-1));
	children.resize(childrenBox,1);
	children.setVal((TreeNode *) NULL);
    }
    
    IntVect ivChild = coarsen(Iv,getNRefToLeaf(level-1));

    if (children(ivChild,0) == (TreeNode *) NULL)
    {
        IntVect ivChild = coarsen(Iv,getNRefToLeaf(level - 1));
        Box childDomain(ivChild,ivChild);
        childDomain.refine(getNRefToLeaf(level-1));
    	children(ivChild,0) = new TreeNode(this,childDomain);
        assert(numChildren == countChildren());
    }
    if ((level - 1) == LBase) 
    {
        assert(numChildren == countChildren());
	return children(ivChild,0);
    }
    else
    {
        assert(numChildren == countChildren());
	return children(ivChild,0)->getNewLevel(LBase,Iv);
    }
}
int 
TreeNode::getNRefToLeaf(int ilevel) const
{
    return(((Tree *) root)->refToLeafArray[ilevel]);
}
void TreeNode::expandTreeWithLeaf()
{
// converts TreeNode with leaf to TreeNode with subtrees, all having
// leaves of the same type as the original tree.

    int reftoleaf = getNRefToLeaf(level-1);
    Box b = coarsen(domain,reftoleaf);
    children.resize(b,1);
    children.setVal((TreeNode *) NULL);
    BoxIterator bit(b);
    for (bit.begin(); bit.ok(); ++bit)
    {
        IntVect tiv = bit();
    	assert(children(tiv) == NULL);
        Box childDomain = refine(Box(tiv,tiv),reftoleaf);
	children(tiv) = new TreeNode(this,childDomain);
        assert(numChildren == countChildren());
	children(tiv)->setLeaf(lf.lt);
    }
    unSetLeaf();
}
void TreeNode::setLeaf(Leaf::LeafType Lt)
{
    assert(parent->root == root);
    lf.lt = Lt;
    parent->incRCCount(1,Lt);
    hasLeaf = true;
    TreeNode* oldstart = ((Tree *) root)->startPtrs[Lt];
    if (oldstart != NULL) 
  	oldstart->prevLeafLoc = this;
    ((Tree *) root)->startPtrs[Lt] = this;
    prevLeafLoc = NULL;
    nextLeafLoc = oldstart;
    numChildren = 0;
}

void TreeNode::unSetLeaf()
{
    assert(parent->root == root);
    assert((lf.lt != Leaf::Irregular));
    parent->incRCCount(-1,lf.lt);
    hasLeaf = false;
    if (prevLeafLoc !=  NULL)
    {
	prevLeafLoc->nextLeafLoc = nextLeafLoc;
    }
    else
    {
  	((Tree*) parent->root)->startPtrs[lf.lt] = nextLeafLoc;
    }
    if (nextLeafLoc != NULL)
	nextLeafLoc->prevLeafLoc = prevLeafLoc;
    prevLeafLoc =  NULL;
    nextLeafLoc =  NULL;
}

bool TreeNode::isLeaf(const IntVect& iv) const
{
    bool retvalue;
    if(domain.contains(iv))
    {
        const TreeNode* trtemp = getTreeWithLeaf(iv);
        retvalue = trtemp != (TreeNode*) NULL; 
    }
    else
        retvalue = false;
    return retvalue;
}
int TreeNode::countChildren()
{
    if (children.box().isEmpty()) return 0;
    BoxIterator bit(children.box());

    int number = 0;
    for (bit.begin(); bit.ok(); ++bit)
    {
        if (children(bit(),0) != (TreeNode *) NULL)
        {
            number = number + 1;
        }
    }
    return number;
}

Tree::Tree()
{
    numChildren = 0;
    nIrregular = 0;
    nRegular = 0;
    nCovered = 0;
    nLeaves = 0;
    irregSize = 0;
    nRef = 2;
    level = 1;
    root = this;
    parent = this;
    hasLeaf = false;
    startPtrs.resize(Leaf::NUMLEAFTYPES, (TreeNode *) NULL);
    domain = Box(IntVect::TheZeroVector(), IntVect::TheUnitVector());
    children.resize(domain,1);
    children.setVal((TreeNode *) NULL);
    refToLeafArray.resize(0);
    refToLeafArray.push_back(1);
    refToLeafArray.push_back(2);

    unitBoxSize = 1;
    for (int i = 0;i < SpaceDim ; i++)
    {
	unitBoxSize = unitBoxSize*nRef;
    }
}
Tree::~Tree()
{
    deleteChildren();
}
void 
Tree::addLeaf(const IntVect& Iv, Leaf::LeafType Lt) 
{
    if (!domain.contains(Iv))
    {
	Box b(Iv,Iv);
	expandTree(b,Lt);
    }
    TreeNode::addLeaf(Iv,Lt);
}

#if CH_SPACEDIM == 3
Box debug_domain(IntVect(0,0,0),IntVect(3,3,3));
Box debug_b(IntVect(2,1,0),IntVect(2,1,0));
#endif

void
Tree::addBox(const Box& B, Leaf::LeafType Lt)
{
    if (!domain.contains(B))
    {
	expandTree(B,Lt);
    }
    TreeNode::addBox(B,Lt);
}
void 
Tree::compressTree()
{
// compressTree should only be called if numChildren == 1 and if the
// child has no Leaf.

    assert (numChildren == countChildren());
    assert(numChildren == 1);
    assert(root == this);
    assert (nLeaves == 0);

// Find non-null child.

    IntVect ivChild;
    BoxIterator itChild(children.box());
    TreeNode* tmp = (TreeNode* ) NULL;
    for (itChild.begin(); 
         (itChild.ok() && (tmp == (TreeNode *) NULL));
         ++itChild)
    {
	if (children(itChild(),0) != (TreeNode *) NULL)
	{
	    ivChild = itChild();
    	    tmp = children(ivChild);
	}
    }

    assert(tmp != (TreeNode *) NULL);

// Check to see whether Child has children. If so, copy them
// Onto root's children.

    BaseFab<TreeNode* >& grandChildren = tmp->children;

    Box newChildBox = grandChildren.box();
    children.resize(newChildBox);
    children.copy(grandChildren);

// Reset parent pointers for the children.

    BoxIterator bit(newChildBox);
    for (bit.begin(); bit.ok(); ++bit)
    {
	TreeNode* childPtr = children(bit());
	if (childPtr != (TreeNode*) NULL)
	{
	    childPtr->parent = root;
	}
    }

// Reset data members of root Tree.

    numChildren = tmp->numChildren;
    nRegular = tmp->nRegular;
    nIrregular = tmp->nIrregular;
    nCovered = tmp->nCovered;
    nLeaves = tmp->nLeaves;
    level = level - 1;

// We are finished with compressing one level out of Tree - 
// we can now delete tmp.

    delete tmp;

// WARNING:: the following is a hack to undo the decrementing 
// of the numChildren counter by delete !!!

    numChildren += 1;

// If numChildren == 1, and the child doesn't have a leaf
// Tree can be further compressed. Call compressTree recursively.
/*
    cout << "Tree compressed by one level, level = " << level << endl;
*/
    assert (numChildren == countChildren());
    if ((numChildren == 1) && (nLeaves == 0)) compressTree();
} 
void Tree::expandTree(const Box& B,Leaf::LeafType Lt)
{
    assert(!domain.contains(B));
    assert(root == this);

// Build covering dyadic box.

    Box newDomain;
    assert (numChildren == countChildren());
    if (numChildren != 0) 
    {
	newDomain = minBox(B,domain);

    }
    else
    {
	if (B.numPts() > 1) 
	{
	    newDomain = B;
	}
	else 
	{
	    newDomain = refine(coarsen(B,nRef),nRef);
	}
    }
    int refRatio = nRef;
    int depth = 1;
    Box prevDomain;
    
    do
    {
	prevDomain = newDomain;
        newDomain.coarsen(nRef);
	if ((prevDomain != newDomain) && (newDomain.numPts() > 1))
	{
            depth++;
	    refRatio *= nRef;
	}
    } while ((newDomain != prevDomain)&& (newDomain.numPts() > 1)); 

// Special case for case that newDomain contains positive and negative
// values. Easy check: what happens if input domain is [-1,0] x [-1,0].

    if (newDomain.numPts() > 1)
    {
	IntVect iv = newDomain.smallEnd();
	newDomain = Box(IntVect::TheZeroVector(), IntVect::TheUnitVector()) + iv;
	refRatio /= nRef;
    } 

    newDomain.refine(refRatio);

// Copy children of current root into oldChildren, and save counter
// information.
    
    BaseFab<TreeNode* > oldChildren(children.box(),1);
    oldChildren.copy(children);
    int oldNumChildren = numChildren;
    int oldLevel = level;
    //#int oldNCovered = nCovered;     //NOTE: not used
    //#int oldNRegular = nRegular;     //       "   "
    //#int oldNIrregular = nIrregular; //       "   "
    //#int oldNLeaves = nLeaves;       //       "   "
    //#Box oldDomain = domain;         //       "   "

// Reset data members of base Tree.

    domain = newDomain;
    level = depth + 1;
    numChildren = 0; 
// Note that numChildren gets incremented by getNewLevel.
    nCovered = 0;
    nRegular = 0;
    nIrregular = 0;
    nLeaves = 0;

    for (int lev = oldLevel+1; lev < level+1; lev++)
    {
	if (!(refToLeafArray.size() < lev))
	{
            int newRef = refToLeafArray[lev-1]*nRef;
            refToLeafArray.push_back(newRef);
	}
    }


// Get new parent for old children, and reset values for new parent,
// parent pointers for old children.

    Box newChildrenDomain = coarsen(domain,refToLeafArray[level-1]);
    children.resize(newChildrenDomain,1);
    children.setVal((TreeNode *) NULL);

    if (oldNumChildren > 0) 
    {
	Box oldChildrenDomain = oldChildren.box();
	BoxIterator bit(oldChildrenDomain);
	for (bit.begin(); bit.ok(); ++bit)
	{
	    IntVect ivChild = bit();

	    IntVect iv = ivChild*getNRefToLeaf(oldLevel-1);
	    TreeNode* childPtr = oldChildren(ivChild);
	    if (childPtr != (TreeNode *) NULL)
	    {
    	        TreeNode* newParent = getNewLevel(oldLevel,iv);
		assert(newParent->numChildren ==
			newParent->countChildren());
    	        newParent->children(ivChild) = childPtr;
	        childPtr->parent = newParent;
		newParent->numChildren += 1;
		assert(newParent->numChildren ==
			newParent->countChildren());

    	        if (childPtr->hasLeaf)
	        {
		    newParent->incRCCount(1,(childPtr->lf).lt);
    		    newParent->agglomTree((childPtr->lf).lt);
	        }
	    }
	}
    }
/*
    cout << "Tree expanded: old level = " << oldLevel <<", new level = "
    << level << ",domain = " << domain << endl;
*/
}
void
Tree::deleteLeaf(const IntVect& Iv) 
{
    TreeNode::deleteLeaf(Iv);
    assert (numChildren == countChildren());
    if ((numChildren == 1)&&(nLeaves == 0)) compressTree();
}

void
Tree::deleteBox(const Box& B,Leaf::LeafType Lt)
{
    deleteBox(B);
}

void
Tree::deleteBox(const Box& B)
{
    TreeNode::deleteBox(B);
    assert (numChildren == countChildren());
    if ((numChildren == 1)&&(nLeaves == 0)) compressTree();
}

int
Tree::getIrregSize()
{
    return irregSize;
}
LeafIterator::LeafIterator(const Tree* RootPtr,Leaf::LeafType Lt)
{
    define(RootPtr,Lt);
}
void LeafIterator::define(const Tree* RootPtr,Leaf::LeafType Lt)
{
    rootPtr = RootPtr;
    lt = Lt;
}
LeafIterator::LeafIterator()
{
    rootPtr =  (Tree* ) NULL;
}
LeafIterator::~LeafIterator()
{
}
void LeafIterator::begin()
{
    currentPtr = rootPtr->startPtrs[lt];
}

void LeafIterator::next()
{
    currentPtr = currentPtr->nextLeafLoc;
}

bool LeafIterator::ok()
{
    return (currentPtr != (TreeNode *) NULL);
}

const Box& LeafIterator::getDomain() 
{
    return currentPtr->domain;
}
