/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/


// TreeIntVectSet.H
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
// ============

#ifndef TREEINTVECTSET_H
#define TREEINTVECTSET_H

#include "Box.H"
#include "Pool.H"

///
/**
   Set calculus class based on trees.
 */
class TreeIntVectSet
{
public:
  ///
  inline TreeIntVectSet();

  ///
  TreeIntVectSet(const Box&);

  ///
  TreeIntVectSet(const TreeIntVectSet& a_sivs);

  ///
  ~TreeIntVectSet();

  ///
  void define(const Box&);

  ///
  void define(const TreeIntVectSet& a_sivs);

  ///
  TreeIntVectSet& operator=(const TreeIntVectSet& a_sivs);

  ///or
  TreeIntVectSet& operator|=(const TreeIntVectSet& s_sivs);

  ///
  TreeIntVectSet& operator|=(const IntVect& a_iv);

  ///
  TreeIntVectSet& operator|=(const Box& a_box);

  ///and
  TreeIntVectSet& operator&=(const TreeIntVectSet& s_sivs);

  ///
  TreeIntVectSet& operator&=(const Box& a_box);

  ///not
  TreeIntVectSet& operator-=(const TreeIntVectSet& s_sivs);

  ///
  TreeIntVectSet& operator-=(const IntVect& a_iv);

  ///
  TreeIntVectSet& operator-=(const Box& a_box);

  ///
  /**
     Returns Vector<Box> representation of this IntVectSet.
  */
  Vector<Box> createBoxes() const;

  
  void createBoxes(Vector<Box>& boxes, int& size) const;

  ///
  bool contains(const IntVect& iv) const;

  ///
  bool contains(const Box& box) const;

  ///
  /**
     somewhat expensive, but shouldn't be ;-)
  */
  TreeIntVectSet chop(int idir, int chop_pnt); 

  ///
  /**
     expensive
  */
  void grow(int igrow);           

  ///
  /**
     expensive
  */
  void grow(int idir, int igrow);

  ///
  /**
     fast if iref is power of 2
  */
  void refine(int iref = 2); 

  ///
  /**
     //fast if iref is power of 2
  */
  void coarsen(int iref = 2); 

  ///
  /**
     very very fast
  */
  void shift(const IntVect& iv); 
  
  ///
  void clear();

  ///
  void nestingRegion(int radius, const Box& domain);

  ///
  inline const Box& minBox() const;

  ///
  inline bool isEmpty() const;

  ///
  int numPts() const;

  friend void dumpTree(const TreeIntVectSet* set);

  ///
  void compact() const;  // logically const operation;

private:

  friend class TreeIntVectSetIterator;

  struct TreeNode {
    TreeNode* nodes;
    TreeNode():nodes(0){;}
  };
  
  TreeNode m_tree;
  Box m_minBox, m_spanBox;
  int m_depth;


  //===============
  static Pool treeNodePool;

  static void   quadrantBox(const Box& inputBox, int quadrant, Box& outputQuadrant);
  static void clearTree(TreeNode& tree);
  static void expandNode(TreeNode& node);
  void growTree();


  static int  oppositeQuadrant(int index);
  static bool nextIntVect(const Box& box, IntVect& iv);
  static void nextNode(int& currentDepth);
  static void cloneNode(const TreeNode& src, TreeNode& dest);
  // data structures that are needed again and again in the routines
  // that only change their size during grow() operations.  Keep them
  // around so that we don't have to create and destroy them on every function
  // call
  static Vector<int> index;
  static Vector<TreeNode*> parents;
  static Vector<Box> boxes;

  const static int  nodeSize = D_TERM(2,*2,*2);
  static TreeNode full; 
  friend struct Flag;

};


class TreeIntVectSetIterator
{
public:
  TreeIntVectSetIterator();
  TreeIntVectSetIterator(const TreeIntVectSet& ivs);
  void define(const TreeIntVectSet& ivs);
  const IntVect& operator()() const ;
  bool ok() const;
  void operator++();
  void begin();
  void end();
  void clear();
private:
  const TreeIntVectSet* m_ivs;
  Vector<const TreeIntVectSet::TreeNode*> nodes;
  Vector<Box>   boxes;
  Vector<int>   index;
  int           m_depth;
  IntVect       m_current;

  void findNextNode();
  void findNext(); // parse tree, starting at [depth, index], set m_current
  //when you find an entry, might run to end of iterator.
};

inline
TreeIntVectSetIterator::TreeIntVectSetIterator():m_ivs(0), m_depth(-1) {;}

inline 
TreeIntVectSetIterator::TreeIntVectSetIterator(const TreeIntVectSet& ivs)
{
  define(ivs);
}

inline
void TreeIntVectSetIterator::clear()
{
  m_depth = -1;
  m_ivs = 0;
}

inline
void TreeIntVectSetIterator::define(const TreeIntVectSet& ivs)
{
  m_ivs = &ivs;
  int max = ivs.index.size();
  if(boxes.size() < max)
    {
      boxes.resize(max);
      index.resize(max);
      nodes.resize(max);
    }
  begin();
}

inline
bool TreeIntVectSetIterator::ok() const
{
  return m_depth >= 0;
}

inline
void TreeIntVectSetIterator::end()
{
  m_depth = -1;
}  

inline
TreeIntVectSet& TreeIntVectSet::operator|=(const IntVect& iv)
{
  return *this|=Box(iv, iv);
}

inline
TreeIntVectSet& TreeIntVectSet::operator-=(const IntVect& iv)
{
  return *this-=Box(iv, iv);
}

inline
void TreeIntVectSetIterator::operator++()
{
  findNext();
}

inline 
const IntVect& TreeIntVectSetIterator::operator()() const
{
  return m_current;
}

//=======================================================

inline
TreeIntVectSet::~TreeIntVectSet()
{
  clearTree(m_tree);
}
inline
TreeIntVectSet::TreeIntVectSet(const TreeIntVectSet& a_tivs)
{
  define(a_tivs);
}

inline
TreeIntVectSet::TreeIntVectSet(const Box& a_box)
{
  define(a_box);
}

inline
TreeIntVectSet::TreeIntVectSet()
{
  m_tree.nodes = 0;
  m_depth=1;
  
}

inline
const Box&
TreeIntVectSet::minBox() const
{
  return m_minBox;
}

inline bool TreeIntVectSet::isEmpty() const
{
  return m_tree.nodes == 0;
}

inline void TreeIntVectSet::nextNode(int& depth)
{
  index[0] = 0;
  index[depth]++;
  while(index[depth] == nodeSize)
    {
      index[depth] = 0;
      depth--;	      
      index[depth]++;
    }
}

#endif //  TREEINTVECTSET_H
