/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/


// TreeIntVectSet.cpp
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
// ============


#include "TreeIntVectSet.H"
#include "Vector.H"
#include "Arena.H"
#include "RefCountedPtr.H"
#include "DisjointBoxLayout.H"
using std::ostream;
using std::cout;
using std::endl;


// The implementation of the tree here is a little
// weird for some.  It is a strictly 2-ary tree. 
// the terminal nodes of the tree represent a 2-ary
// box of information, not just a point.  it uses
// the bit pattern in the pointer data holder to 
// store the true/false information for the points in the
// 2-ary box. This requires 8 bits for the 3D case, a bit
// for each point in the box.  hence the appearance of
// the magic '256' in the code (2^8).  TreeNode pointers
// that have a value less than '256' are not pointing at
// memory, but holding the leaf information.

//
//  Further details of how non-recursive TreeNode design works: 
//
//  (for a 2D tree) 
//
//    (m_tree)    
//           + -- 0  
//
//             (a)+ - 0
//                    1
//                1   +
//                    1  <------you are here
//              
//                + - + - 0
//                    0   1
//                    0   1
//                    0   0
//
//  for the node indicated, the 'index' vector  would contain
//
//  index=[ 0      1    3  ...............]            
//parents=[&m_tree &a   ..................]
//
// or directly refered to as m_tree.nodes[1].nodes[3]
//
//  the tree indicates a covering of an index space in either
//  1 or 0.   1 is stored in the tree by pointing at the static
//  'full' data member, 0 is stored as a 0.
//
//  every 'nodes' member of a TreeNode object can be either
//
//   0, &full, or a pointer .
//
//   The interpretation of the tree depends on what m_spanBox is.
//     nodes[i] indicates whether the i'th quadrant of the parent
//     Box is full, empty, or needs to be parsed deeper.
//
//


// handy macro for tree parsing operations....

#define NEXT(index_local, depth)                 \
      index_local[0] = 0;                        \
	  index_local[depth]++;                      \
		  while(index_local[depth] == nodeSize)  \
			{                                    \
			  index_local[depth] = 0;            \
			  depth--;	                         \
			  index_local[depth]++;              \
            }                


void TreeIntVectSet::define(const Box& a_box)
{

  clearTree(m_tree);
  m_minBox = a_box;

  if(a_box.isEmpty())
    {
      m_spanBox = Box();
      return;
    }

//    int size = 0;
//    for(int i=0; i<SpaceDim; ++i) size = Max(size, a_box.size(i));
//    int clippedSize = (size/2)*2;
//    if(clippedSize < size) size = clippedSize+2;

//    int minBoxSize = 2;
//    m_depth = 1;
//    while(minBoxSize < size) 
//      {
//        minBoxSize*=2;
//        m_depth++;
//      }

//    int m = index.size();
//    m = Max(m_depth+1, m);
//    if(m > index.size())
//      {
//        index.resize(m);
//        parents.resize(m);
//        boxes.resize(m);
//      }

//    m_spanBox = Box(a_box.smallEnd(), 
//  		 a_box.smallEnd()+(minBoxSize-1)*IntVect::TheUnitVector());
  

  *this |= a_box;

}

void TreeIntVectSet::clear()
{
  clearTree(m_tree);
  m_spanBox = Box();
  m_minBox  = Box();
  m_depth = 1;
}


TreeIntVectSet& TreeIntVectSet::operator=(const TreeIntVectSet& rhs)
{
  define(rhs);
  return *this;
}
void TreeIntVectSet::define(const TreeIntVectSet& a_tivs)
{
  if(this == &a_tivs) return;

  clearTree(m_tree);

  m_spanBox = a_tivs.m_spanBox;
  m_minBox  = a_tivs.m_minBox;
  m_depth  = a_tivs.m_depth;

  if(a_tivs.m_tree.nodes == 0 || a_tivs.m_tree.nodes == &full)
    {
      m_tree = a_tivs.m_tree;
      return;
    }

  expandNode(m_tree);

  // perform direct Tree-to-Tree copy operation.
  // this is a little more involved than just
  // adding the boxes from one to the other, but
  // it should be much more efficient and faster

  cloneNode(a_tivs.m_tree, m_tree);

}

void TreeIntVectSet::cloneNode(const TreeNode& src, TreeNode& dest)
{
  static Vector<const TreeNode*> otherParents;
  if(parents.size() > otherParents.size())
    otherParents.resize(parents.size());
    
  if(src.nodes ==0 || src.nodes == &full)
	{
	  dest=src;
	  return;
	}
  if(dest.nodes == 0 || dest.nodes == &full)
	{
	  expandNode(dest);
	}

  otherParents[0] = &(src);
  parents[0]  = &(dest);
  index[1] = 0;
  int depth = 1;
  while(depth!=0) 
    {
      const TreeNode* otherParent  = otherParents[depth-1];
      TreeNode* thisParent   = (TreeNode*)parents[depth-1];
	  int ind = index[depth];
      const TreeNode& otherCurrent = otherParent->nodes[ind];
      TreeNode& thisCurrent  = thisParent->nodes[ind];

      if(otherCurrent.nodes == 0 || otherCurrent.nodes == &full) // terminal
	{
	  thisCurrent = otherCurrent;
	  nextNode(depth);
	}
      else  // not terminal node, add nodes and traverse deeper
	{
	  expandNode(thisCurrent);
	  otherParents[depth] = &(otherCurrent);
	  parents[depth]  = &(thisCurrent);
	  depth++;
	  index[depth] = 0;
	}
    }
}

void TreeIntVectSet::clearTree(TreeNode& tree)
{
  static Vector<TreeNode*> parents_local;
  static Vector<int>       index_local;
  parents_local.resize(TreeIntVectSet::parents.size());
  index_local.resize(TreeIntVectSet::index.size());
  index_local[0] = 0;
  // quick check to eliminate the work of tiny tree deletion
  if(tree.nodes == 0 || tree.nodes == &full) {tree.nodes = 0; return; }

  int depth = 1;

  parents_local[0] = &(tree);
  index_local[1] = 0;
  while(depth != 0)
    {
      TreeNode* parent =  parents_local[depth-1];
      int ind = index_local[depth];
      //Vector<int>& indexRef = index;
      //Vector<TreeNode*>& parentRef = parents;
      TreeNode& current = parent->nodes[ind];
      if(current.nodes == 0 || current.nodes == &full)
	{
	  index_local[depth]++;
	  while(index_local[depth] == nodeSize)
	    {
	      index_local[depth] = 0;
	      //delete[] parents[depth-1]->nodes;
	      treeNodePool.returnPtr(parents_local[depth-1]->nodes);
	      depth--;
	      index_local[depth]++;
	    }
	}
      else{
	parents_local[depth] = &(current);
	depth++;
	index_local[depth] = 0;
      }
    }
  tree.nodes = 0;
}

void TreeIntVectSet::refine(int iref)
{
  if(iref == 1) return;
  assert(iref >= 1);
  int refinements = iref/2;
  assert(refinements*2 == iref); // check iref for power of 2
  m_spanBox.refine(iref);
  m_minBox.refine(iref);
  m_depth+=iref/2;
  int m=index.size();
  m = Max(m_depth+1, m);
  if(m > index.size())
    {
      index.resize(m);
      parents.resize(m);
      boxes.resize(m);
    }
}

void TreeIntVectSet::shift(const IntVect& iv)
{
  static Vector<Box> boxs;
  int size;
  
  createBoxes(boxs, size);
  clear();
  for(int i=0; i < size; ++i)
    {
      boxs[i].shift(iv);
    }
  for(int i=0; i<size; ++i)
    {
      *this |= boxs[i];
    } 
}

void TreeIntVectSet::grow(int igrow)
{
  static Vector<Box> boxs;
  int size;
  
  createBoxes(boxs, size);
  clear();
  for(int i=0; i < size; ++i)
    {
      boxs[i].grow(igrow);
    }
  for(int i=0; i<size; ++i)
    {
      *this |= boxs[i];
    } 
}

void TreeIntVectSet::grow(int idir, int igrow)
{
  static Vector<Box> boxs;
  int size;

  createBoxes(boxs, size);
  clear();
  for(int i=0; i<size; ++i)
    {
      boxs[i].grow(idir, igrow);
    }
  for(int i=0; i<size; ++i)
    {
      *this |= boxs[i];
    } 
}

/*
TreeIntVectSet TreeIntVectSet::chop(int dir, int chop_pnt)
{
  if(m_minBox.smallEnd(dir) >= chop_pnt)
	{
	  TreeIntVectSet rtn(*this);
	  clear();
	  return rtn;
	}
  if(m_minBox.bigEnd(dir) < chop_pnt)
	{
	  return TreeIntVectSet();
	}


  static Vector<Box> boxs;
  int size;
  createBoxes(boxs, size);

  clear();
  TreeIntVectSet rtn;
  for(int i=0; i<size; ++i)
    {
      Box& box = boxs[i];
      if(box.smallEnd(dir) >= chop_pnt)
		rtn |= box;
      else if(box.bigEnd(dir) < chop_pnt)
		*this |= box;
      else
		{
		  Box hi = box.chop(dir, chop_pnt);
		  rtn |= hi;
		  *this |= box;
		}
    }
  return rtn;
}

*/

TreeIntVectSet TreeIntVectSet::chop(int dir, int chop_pnt)
{
  if(m_minBox.smallEnd(dir) >= chop_pnt)
	{
	  TreeIntVectSet rtn(*this);
	  clear();
	  return rtn;
	}
  if(m_minBox.bigEnd(dir) < chop_pnt)
	{
	  return TreeIntVectSet();
	}

  Box min  = m_minBox;
  Box chop = m_spanBox;

  TreeIntVectSet rtn(*this);
  
  Box chopThis = chop.chop(dir, chop_pnt);
  rtn   -= chop;
  *this -= chopThis;
  

  m_minBox = min;
  rtn.m_minBox = m_minBox.chop(dir, chop_pnt);

  return rtn;

}

/*
  void TreeIntVectSet::coarsen(int icoarse)
  {
    static  Vector<Box> boxs;
   int size;
  
    createBoxes(boxs, size);
    clear();
    for(int i=0; i<size; ++i)
      {
        Box& box = boxs[i];
        box.coarsen(icoarse);
        this->operator|=(box);
      }
    compact();
  }
*/


void TreeIntVectSet::coarsen(int icoarse)
{ 
  if(icoarse == 1) return; 
  assert(icoarse >= 1); 
  int coarsenings = icoarse/2; 
  assert(coarsenings*2 == icoarse); // check icoarse for power of 2 
  
  compact(); 
  if(m_tree.nodes == 0) return; 
  if(m_tree.nodes == &full) { m_spanBox.coarsen(icoarse); return;} 
  if(coarsenings > m_depth)  
	{  
	  clearTree(m_tree); 
	  m_tree.nodes = &full; 
	  m_spanBox.coarsen(icoarse);
	  m_minBox.coarsen(icoarse);
	  return; 
	} 
    
  // first, pure tree manipulation, turn all unit sized terminal nodes 
  // into gathered nodes.
  //      ie.
  // if depth == m_depth-1  && TreeNode.node != 0 or full  
  //     TreeNode.nodes = [full,0,0,full]  -> TreeNode.nodes = full  
  
  parents[0] = &m_tree; 
 
  index[1] = 0; 
  index[0] = 0;
  int depth = 1; 
  while(depth != 0) 
	{ 
	  TreeNode& current = parents[depth-1]->nodes[index[depth]]; 
	  if(depth == m_depth-coarsenings-1) 
		{ 
		  if(current.nodes != 0)
			{
			  clearTree(current); 
			  current.nodes = &full;
			} 
		} 
	  else if(!(current.nodes == 0 || current.nodes == &full)) 
		{ 
		  parents[depth] = &current; 
		  depth++; 
		  index[depth] = -1; 
		} 
	  nextNode(depth); 
	} 

  m_spanBox.coarsen(icoarse); 
  m_minBox.coarsen(icoarse);
  m_depth-=coarsenings; 

}

void TreeIntVectSet::expandNode(TreeNode& a_node)
{
  // Vector<int>&  indexRef = index;
  TreeNode* tmp = a_node.nodes;
  assert(tmp == 0 || tmp == &full);
  //a_node.nodes = new TreeNode[nodeSize];
  a_node.nodes = (TreeNode*)(treeNodePool.getPtr());
  //printf("alloc e: %p\n",a_node.nodes);
  for(int i=0; i<nodeSize; ++i)
    {  
      a_node.nodes[i].nodes = tmp;
    }

}


TreeIntVectSet& TreeIntVectSet::operator|=(const Box& a_box)
{
  if(a_box.isEmpty()) return *this;

  if(m_tree.nodes == 0 || m_tree.nodes == &full)
    {
      expandNode(m_tree);
    }
  m_minBox.minBox(a_box);

  // one half of the real smarts of this class, the other half being
  // the growTree function

  while(!m_spanBox.contains(a_box)) growTree();


  //Vector<int>& indexRef = index; //used for debugging
  //Vector<Box>& boxRef   = boxes; //  ..............
  index[1] = 0;
  parents[0] = &(m_tree);
  boxes[0] = m_spanBox;
  int depth = 1;

  while(depth != 0)
    {
      TreeNode* parent = parents[depth-1];
      int ind = index[depth];
      TreeNode& current = parent->nodes[ind];
      Box& parentBox  = boxes[depth-1];
      Box& currentBox = boxes[depth];
      quadrantBox(parentBox, index[depth], currentBox);
      // clear up the two most common cases quickly

      // case 0 a_box does not touch this quadrant, or this whole
      //quadrant is already marked 'full'...no change
      if(current.nodes == &full || !a_box.intersectsNotEmpty(currentBox))
	{
	  nextNode(depth);
	}
      // case 1 currentBox is completely covered by a_box.
      else if(currentBox.smallEnd() >= a_box.smallEnd() &&
	      currentBox.bigEnd()   <= a_box.bigEnd())
	{
	  clearTree(current);
	  current.nodes = &full;
	  nextNode(depth);
	}
   

      // OK, now things are more tricky, partial intersection of
      // boxes.  Need to parse deeper
      else{
	if(current.nodes == 0 || current.nodes == &full)
	  expandNode(current);
	parents[depth] = &(current);
	depth++;
	index[depth]= 0;		
      }	
    } 
  
  return *this;
  
}


TreeIntVectSet& TreeIntVectSet::operator|=(const TreeIntVectSet& set)
{
  if(set.m_tree.nodes==0) return *this;
  if(set.m_tree.nodes == &full)
    {
      this->operator|=(set.m_spanBox);
      return *this;
    }
  if(m_tree.nodes == 0)
	{
	  *this = set;
	  return *this;
	}
  if(m_tree.nodes == &full)
	{
	  Box b = m_spanBox;
	  *this = set;
	  *this |= b;
	  return *this;
	}

  const TreeNode* yourRoot;
  TreeIntVectSet tmp;
  
  if(m_depth == set.m_depth)
	{
	  yourRoot = &(set.m_tree);
	}
  else if(m_depth < set.m_depth)
	{
	  while(m_depth < set.m_depth) growTree();
	  yourRoot = &(set.m_tree);
	}
  else
	{
	  tmp = set;
	  while(tmp.m_depth < m_depth) tmp.growTree();
	  yourRoot = &(tmp.m_tree);
	}

  static Vector<const TreeIntVectSet::TreeNode*> parents_other, parents_this;
  static Vector<int> indexSet;
  if(parents.size() > parents_other.size())
    {
	  parents_other.resize(parents.size());
	  parents_this.resize(parents.size());
	  indexSet.resize(parents.size());
    }

  parents_other[0] = yourRoot;
  parents_this[0] = &m_tree;
  indexSet[1] = 0;
  int depth = 1;
  while(depth!=0) 
    {
      const TreeNode* otherParent  = parents_other[depth-1];
      TreeNode* thisParent   = (TreeNode*)parents_this[depth-1];
      const TreeNode& otherCurrent = otherParent->nodes[indexSet[depth]];
      TreeNode& thisCurrent  = thisParent->nodes[indexSet[depth]];

      if(otherCurrent.nodes == 0 || thisCurrent.nodes == &full) 
		{
		  NEXT(indexSet, depth);
		}
	  else if(otherCurrent.nodes == &full)
		{
		  clearTree(thisCurrent);
		  thisCurrent = otherCurrent;
		  // leave the rest of this node alone
		  NEXT(indexSet, depth);
		}
	  else if(thisCurrent.nodes == 0) 
		{
		  cloneNode(otherCurrent, thisCurrent);
		  NEXT(indexSet, depth);
		}
      else  // not terminal node for either one, traverse deeper
		{
		  parents_other[depth] = &(otherCurrent);
		  parents_this[depth]  = &(thisCurrent);
		  depth++;
		  indexSet[depth] = 0;
		}
    } 
  m_minBox.minBox(set.m_minBox);

  return *this;
}


//old version that didn't rely on common centering techniques.....

// TreeIntVectSet& TreeIntVectSet::operator|=(const TreeIntVectSet& set)
// {

//   if(set.m_tree.nodes==0) return *this;
//   else if(set.m_tree.nodes == &full)
//     {
//       this->operator|=(set.m_spanBox);
//       return *this;
//     }

//   //m_minBox.minBox(set.m_minBox);

//   int depth = set.m_depth+1;
//   static Vector<int> index_local;
//   static Vector<const TreeIntVectSet::TreeNode*> parents_local;
//   static Vector<Box> boxes_local;
//   if(depth > index_local.size())
//     {
//       index_local.resize(depth);
//       parents_local.resize(depth);
//       boxes_local.resize(depth);
//     }
//   index_local[0] = 0;

//   depth = 1;
//   parents_local[0] = &(set.m_tree);
//   boxes_local[0] = set.m_spanBox;
//   index_local[1] = 0;
//   while(depth != 0)
//     {
//       const TreeNode* parent =  parents_local[depth-1];
//       int ind = index_local[depth];
//       const TreeNode& current = parent->nodes[ind];
//       quadrantBox(boxes_local[depth-1], ind, boxes_local[depth]);
//       if(current.nodes == 0 || current.nodes == &full)
// 		{
// 		  if(current.nodes == 0){;}  // do nothing
// 		  else if(current.nodes == &TreeIntVectSet::full)
// 			{
// 			  this->operator|=(boxes_local[depth]);
// 			}
// 		  index_local[depth]++;
// 		  while(index_local[depth] == nodeSize)
// 			{
// 			  index_local[depth] = 0;
// 			  depth--;	      
// 			  index_local[depth]++;
// 			}
// 		}
//       else{
// 		parents_local[depth] = &(current);
// 		depth++;
// 		index_local[depth] = 0;
//       }
//     }
//   return *this;
// } 




TreeIntVectSet& TreeIntVectSet::operator-=(const TreeIntVectSet& set)
{

  if(set.m_tree.nodes==0) return *this;
  else if(set.m_tree.nodes == &full)
    {
      this->operator-=(set.m_spanBox);
      return *this;
    }
  if(m_tree.nodes == 0) return *this;

  int depth = set.m_depth+1;
  static Vector<int> index_local;
  static Vector<const TreeIntVectSet::TreeNode*> parents_local;
  static Vector<Box> boxes_local;
  if(depth > index_local.size())
    {
      index_local.resize(depth);
      parents_local.resize(depth);
      boxes_local.resize(depth);
    }
  index_local[0] = 0;
  depth = 1;
  parents_local[0] = &(set.m_tree);
  boxes_local[0] = set.m_spanBox;
  index_local[1] = 0;
  while(depth != 0)
    {
      const TreeNode* parent =  parents_local[depth-1];
      int ind = index_local[depth];
      const TreeNode& current = parent->nodes[ind];
      quadrantBox(boxes_local[depth-1], ind, boxes_local[depth]);
	  if(boxes_local[depth].intersectsNotEmpty(m_minBox))
		{
		  if(current.nodes == 0 || current.nodes == &full)
			{
			  if(current.nodes == 0){;}  // do nothing
			  else if(current.nodes == &TreeIntVectSet::full)
				{
				  this->operator-=(boxes_local[depth]);
				}
			  index_local[depth]++;
			  while(index_local[depth] == nodeSize)
				{
				  index_local[depth] = 0;
				  depth--;	      
				  index_local[depth]++;
				}
			}
		  else{
			parents_local[depth] = &(current);
			depth++;
			index_local[depth] = 0;
		  }
		}
	  else
		{
		  index_local[depth]++;
		  while(index_local[depth] == nodeSize)
			{
			  index_local[depth] = 0;
			  depth--;	      
			  index_local[depth]++;
			}
		}
	}
  return *this;
} 


TreeIntVectSet& TreeIntVectSet::operator&=(const TreeIntVectSet& set)
{
  if(set.m_tree.nodes==0 || m_tree.nodes == 0 || !m_minBox.intersects(set.m_minBox))
	{
	  clear();  
	  return *this;
	}

  if(set.m_tree.nodes==&full) { return *this&=set.m_spanBox;}
  else if(m_tree.nodes == &full)
    {
      expandNode(m_tree);
    }

  const TreeNode* yourRoot;
  TreeIntVectSet tmp;
  
  if(m_depth == set.m_depth)
	{
	  yourRoot = &(set.m_tree);
	}
  else if(m_depth < set.m_depth)
	{
	  while(m_depth < set.m_depth) growTree();
	  yourRoot = &(set.m_tree);
	}
  else
	{
	  tmp = set;
	  while(tmp.m_depth < m_depth) tmp.growTree();
	  yourRoot = &(tmp.m_tree);
	}

  static Vector<const TreeIntVectSet::TreeNode*> parents_other, parents_this;
  static Vector<int> indexSet;
  if(parents.size() > parents_other.size())
    {
	  parents_other.resize(parents.size());
	  parents_this.resize(parents.size());
	  indexSet.resize(parents.size());
    }

  parents_other[0] = yourRoot;
  parents_this[0] = &m_tree;
  indexSet[1] = 0;
  int depth = 1;
  while(depth!=0) 
    {
      const TreeNode* otherParent  = parents_other[depth-1];
      TreeNode* thisParent   = (TreeNode*)parents_this[depth-1];
      const TreeNode& otherCurrent = otherParent->nodes[indexSet[depth]];
      TreeNode& thisCurrent  = thisParent->nodes[indexSet[depth]];

      if(otherCurrent.nodes == 0 || thisCurrent.nodes == 0) 
		{
		  clearTree(thisCurrent);
		  NEXT(indexSet, depth);
		}
	  else if(otherCurrent.nodes == &full)
		{
		  // leave the rest of this node alone
		  NEXT(indexSet, depth);
		}
	  else if(thisCurrent.nodes == &full) // swap nodes
		{
		  clearTree(thisCurrent);
		  cloneNode(otherCurrent, thisCurrent);
		  NEXT(indexSet, depth);
		}
      else  // not terminal node for either one, traverse deeper
		{
		  parents_other[depth] = &(otherCurrent);
		  parents_this[depth]  = &(thisCurrent);
		  depth++;
		  indexSet[depth] = 0;
		}
    } 
  m_minBox &= set.m_minBox;

  return *this;
}

/*
TreeIntVectSet& TreeIntVectSet::operator&=(const TreeIntVectSet& set)
{

  if(set.m_tree.nodes==0 || m_tree.nodes == 0 || !m_minBox.intersects(set.m_minBox))
	{
	  clear();  
	  return *this;
	}
  if(set.m_tree.nodes==&full) { return *this&=set.m_spanBox;}
  else if(m_tree.nodes == &full)
    {
      expandNode(m_tree);
    }
  
  // step one, trim by be minbox;
  *this &= set.m_minBox;

  // now, subtract the complement of the
  int depth = set.m_depth+1;
  static Vector<int> index_local;
  static Vector<const TreeIntVectSet::TreeNode*> parents_local;
  static Vector<Box> boxes_local;
  if(depth > index_local.size())
    {
      index_local.resize(depth);
      parents_local.resize(depth);
      boxes_local.resize(depth);
    }
  index_local[0] = 0;

  depth = 1;
  parents_local[0] = &(set.m_tree);
  boxes_local[0] = set.m_spanBox;
  index_local[1] = 0;
  while(depth != 0)
    {
      const TreeNode* parent =  parents_local[depth-1];
      int ind = index_local[depth];
      const TreeNode& current = parent->nodes[ind];
      quadrantBox(boxes_local[depth-1], ind, boxes_local[depth]);
      if(current.nodes == 0 || current.nodes == &full)
	{
	  if(current.nodes == 0)
	    {
	      *this -= boxes_local[depth];
	    } 
	  else if(current.nodes == &TreeIntVectSet::full)
	    {
	      //do nothing.
	    }
	  index_local[depth]++;
	  while(index_local[depth] == nodeSize)
	    {
	      index_local[depth] = 0;
	      depth--;	      
	      index_local[depth]++;
	    }
	}
      else{
	parents_local[depth] = &(current);
	depth++;
	index_local[depth] = 0;
      }
    }
  return *this;
} 

*/


// first pass to help matters a little, next pass can exploit
// the common centering aspect of the tmp and *this tree.  bvs.
void TreeIntVectSet::nestingRegion(int radius, const Box& domain)
{
  assert(radius>0);
  TreeIntVectSet tmp(*this);
  {
	IntVect lo = m_minBox.smallEnd();
	IntVect hi = m_minBox.bigEnd();
	for(int i=0; i<SpaceDim; ++i)
	  {
		if(lo[i] != domain.smallEnd()[i]) lo[i] += radius;
		if(hi[i] != domain.bigEnd()[i])   hi[i] -= radius;
	  }
	Box shrink(lo, hi);
	*this &= shrink;
  }

  static IntVect lower(-radius * IntVect::Unit), upper(IntVect::Unit * radius);
 
  int depth  = m_depth+1;
  static Vector<int> index_tmp;
  static Vector<const TreeIntVectSet::TreeNode*> parents_tmp;
  static Vector<Box> boxes_tmp;
  if(depth > index_tmp.size())
    {
      index_tmp  .resize(depth);
      parents_tmp.resize(depth);
      boxes_tmp  .resize(depth);
    }
  index_tmp[0] = 0;

  depth = 1;
  parents_tmp[0] = &(tmp.m_tree);
  boxes_tmp[0] = m_spanBox;
  index_tmp[1] = 0;
  Box clobberBox;
  IntVect& lo = (IntVect&)(clobberBox.smallEnd());
  IntVect& hi = (IntVect&)(clobberBox.bigEnd());
  while(depth != 0)
    {
      const TreeNode* parent =  parents_tmp[depth-1];
      int ind = index_tmp[depth];
	  Box&  currentBox = boxes_tmp[depth];
      const TreeNode& current = parent->nodes[ind];
      quadrantBox(boxes_tmp[depth-1], ind, currentBox);
      if(current.nodes == 0 || current.nodes == &full)
		{
		  if(current.nodes == 0)
			{
			  lo = currentBox.smallEnd();
			  hi = currentBox.bigEnd();
			  lo.max(domain.smallEnd());
			  hi.min(domain.bigEnd());
			  if(hi>=lo)
				{
				  hi+=upper;
				  lo+=lower;				  
				  clobberBox.computeBoxLenNotEmpty();
				  *this -= clobberBox;
				} 
			}
		  else if(current.nodes == &TreeIntVectSet::full)
			{
			  //do nothing.
			}
		  index_tmp[depth]++;
		  while(index_tmp[depth] == nodeSize)
			{
			  index_tmp[depth] = 0;
			  depth--;	      
			  index_tmp[depth]++;
			}
		}
      else{
		parents_tmp[depth] = &(current);
		depth++;
		index_tmp[depth] = 0;
      }
	}
}

bool TreeIntVectSet::contains(const IntVect& iv) const
{
  if(m_tree.nodes == 0 || !m_minBox.contains(iv)) return false;
  if(m_tree.nodes == &full) return true;

  //Vector<int>& indexRef = index; //used for debugging
  //Vector<Box>& boxRef   = boxes; //  ..............

  int depth = 1;
  parents[0] = (TreeNode*)&(m_tree);
  boxes[0] = m_spanBox;
  index[1] = 0;
  while(depth != 0)
    {
      const TreeNode* parent =  parents[depth-1];
      int ind = index[depth];
      const TreeNode& current = parent->nodes[ind];
      quadrantBox(boxes[depth-1], ind, boxes[depth]);
      if(!boxes[depth].contains(iv))
	{
	  index[depth]++;
	  assert(index[depth] != nodeSize);
	}
      else
	{
	  if(current.nodes == 0) return false;
	  if(current.nodes == &full) return true;
	  parents[depth] = (TreeNode*)&(current);
	  depth++;
	  index[depth] = 0;
	}
    }
  return false;
} 

TreeIntVectSet& TreeIntVectSet::operator&=(const Box& a_box)
{
  // one half of the real smarts of this class, the other half being
  // the growTree function

  if(m_tree.nodes == 0) return *this;
  if(a_box.contains(m_minBox)) return *this;
  if(m_tree.nodes == &full) expandNode(m_tree);

  //Vector<int>& indexRef = index; //used for debugging
  //Vector<Box>& boxRef   = boxes; //  ..............
  index[0] = 0;
  index[1] = 0;
  parents[0] = &(m_tree);
  boxes[0] = m_spanBox;
  int depth = 1;
  while(depth != 0)
    {
      TreeNode* parent = parents[depth-1];
      int ind = index[depth];
      TreeNode& current = parent->nodes[ind];
      Box& parentBox  = boxes[depth-1];
      Box& currentBox = boxes[depth];
      quadrantBox(parentBox, index[depth], currentBox);
      if(current.nodes == 0){
		nextNode(depth);
      }
      else if(current.nodes == &full)
		{
		  if(a_box.contains(currentBox))
			{
			  nextNode(depth);
			}
		  else if(!a_box.intersectsNotEmpty(currentBox))
			{
			  clearTree(current);
			  current.nodes = 0;
			  nextNode(depth);
			}
		  else
			{
			  expandNode(current);
			  parents[depth] = &(current);
			  depth++;
			  index[depth]= 0;
			}	
		}
      // case 1 currentBox is completely covered by a_box.
      else{
		parents[depth] = &(current);
		depth++;
		index[depth]= 0;		
      }	
    } 
  
  return *this; 
}


TreeIntVectSet& TreeIntVectSet::operator-=(const Box& a_box)
{
 

  if(m_tree.nodes == 0 || (a_box & m_minBox).isEmpty()) return *this;
  if(a_box.contains(m_minBox))
    {
      clear();
      return *this;
    }
  if(m_tree.nodes == &full)
    {
      expandNode(m_tree);
    }
	  

  //Vector<int>& indexRef = index; //used for debugging
  //Vector<Box>& boxRef   = boxes; //  ..............
  index[0] = 0;
  index[1] = 0;
  parents[0] = &(m_tree);
  boxes[0] = m_spanBox;
  int depth = 1;
  while(depth != 0)
    {
      TreeNode* parent = parents[depth-1];
      int ind = index[depth];
      TreeNode& current = parent->nodes[ind];
      Box& parentBox  = boxes[depth-1];
      Box& currentBox = boxes[depth];
      quadrantBox(parentBox, index[depth], currentBox);
      if(a_box.contains(currentBox))
	{
	  clearTree(current);
	  current.nodes = 0;
	  nextNode(depth);
	}
      else if(current.nodes == 0){
	nextNode(depth);
      }
      else if(current.nodes == &full)
	{
	  if((depth == m_depth-1) || !a_box.intersectsNotEmpty(currentBox))
	    {
	      nextNode(depth);
	    }
	  else
	    {
	      expandNode(current);
	      parents[depth] = &(current);
	      depth++;
	      index[depth]= 0;
	    }	
	}
      else{
	parents[depth] = &(current);
	depth++;
	index[depth]= 0;		
      }	
    } 
  compact();
  return *this; 
}

void TreeIntVectSet::growTree()
{
  // do the grow operation
  
  // two cases. top node is terminal, or it has nodeSize leaves to parse out

  if(m_tree.nodes == 0 || m_tree.nodes == &full)
    {
      expandNode(m_tree);
    }
  else
    {
      for(int i=0; i<nodeSize; ++i)
	{
	  TreeNode* tmp = m_tree.nodes[i].nodes;
	  if(tmp != 0)
	    {
	      int ind = oppositeQuadrant(i);
	      //m_tree.nodes[i].nodes = new TreeNode[nodeSize];
		  TreeNode* newPtr = (TreeNode*)(treeNodePool.getPtr());
	      m_tree.nodes[i].nodes = newPtr;
		  for(int j=0; j< nodeSize; ++j) newPtr[j].nodes = 0;
	      //printf("alloc: %p\n", m_tree.nodes[i].nodes );
	      //printf("tmp: %p\n", tmp);
	      newPtr[ind].nodes = tmp;
	    }
	}

    }
  
  if(m_spanBox.isEmpty()) 
    m_spanBox = Box(-1*IntVect::Unit, IntVect::Zero);
  else
    m_spanBox.grow(m_spanBox.size(0)/2);
 

  m_depth++;
  int m = m_depth+1;
  if(m > index.size())
    {
      index.resize(m);
      parents.resize(m);
      boxes.resize(m);
    }
}


// Basically, a BoxIterator function without the crap of
// a BoxIterator.
bool TreeIntVectSet::nextIntVect(const Box& box, IntVect& iv)
{

  iv[0]++;
  if(iv[0] > box.bigEnd(0))
    {
#if (CH_SPACEDIM >= 2)
      iv[0] = box.smallEnd(0);
      iv[1]++;
#else
      return false;
#endif
    }
  else
    {
      return true;
    }

#if (CH_SPACEDIM >= 2)
  if(iv[1] > box.bigEnd(1))
    {
#if (CH_SPACEDIM == 3)
      iv[1] = box.smallEnd(1);
      iv[2]++;
#else
      return false;
#endif 
    }
  else
    {
      return true;
    }
#endif

#if (CH_SPACEDIM == 3)
  if(iv[2] > box.bigEnd(2))
    {
      return false;
    }
#endif

  assert(iv<=box.bigEnd());
  assert(iv>=box.smallEnd());
  return true;
}

void TreeIntVectSet::quadrantBox(const Box& inputBox, 
				 int quadrant, 
				 Box& outputQuadrant)
{
  int halfSize = inputBox.size(0)/2;
  int q   = nodeSize/2;
  //IntVect low, hi;
  int* low = (int*)(outputQuadrant.loVect());
  int* hi  = (int*)(outputQuadrant.hiVect());
  if(quadrant < q)
    {
      low[0] = inputBox.smallEnd(0);
      hi[0]   = inputBox.smallEnd(0)+halfSize-1;
    }
  else
    {
      low[0] = inputBox.smallEnd(0)+halfSize;
      hi[0]   = inputBox.bigEnd(0);
      quadrant-=q;
    }

#if (CH_SPACEDIM >= 2)
  q/=2;
  if(quadrant < q)
    {
      low[1] = inputBox.smallEnd(1);
      hi[1]   = inputBox.smallEnd(1)+halfSize-1;
    }
  else
    {
      low[1] = inputBox.smallEnd(1)+halfSize;
      hi[1]   = inputBox.bigEnd(1);
      quadrant-=q;
    }
#endif

#if (CH_SPACEDIM >= 3)
  q/=2;
  if(quadrant < q)
    {
      low[2] = inputBox.smallEnd(2);
      hi[2]   = inputBox.smallEnd(2)+halfSize-1;
    }
  else
    {
      low[2] = inputBox.smallEnd(2)+halfSize;
      hi[2]   = inputBox.bigEnd(2);
    }
#endif
  assert(!outputQuadrant.isEmpty());
  outputQuadrant.computeBoxLenNotEmpty();
}

inline
void tab(ostream& os, int tab)
{
  for(; tab>=0 ;tab--) os << " ";
}

int TreeIntVectSet::numPts() const
{
  if(isEmpty()) return 0;
  if(m_tree.nodes == &full) return m_spanBox.numPts();
  
  int rtn= 0;
  int depth = 1;
  parents[0] = (TreeNode*)&m_tree;
  boxes[0] = m_spanBox;
  index[1] = 0;
  while(depth != 0)
    {
      const TreeNode* parent =  parents[depth-1];
      int ind = index[depth];
      const TreeNode& current = parent->nodes[ind];
      quadrantBox(boxes[depth-1], ind, boxes[depth]);
      if(current.nodes == 0 || current.nodes == &full)
	{
	  if(current.nodes == 0){;}  // do nothing
	  else if(current.nodes == &full)
	    {
	      rtn += boxes[depth].numPts();
	    }
	  nextNode(depth);
	}
      else{
	parents[depth] = (TreeNode*)&(current);
	depth++;
	index[depth] = 0;
      }
    }
  return rtn;
}  
Vector<Box> TreeIntVectSet::createBoxes() const
{
  static Vector<Box> buffer(200), rtn;
  int size;
  createBoxes(buffer, size);
  rtn.resize(size, Box());
  for(int i=0; i<size; ++i) rtn[i] = buffer[i];
  return rtn;
}

void TreeIntVectSet::createBoxes(Vector<Box>& rtn, int& size) const
{
  size = 0;
  if(m_tree.nodes == 0) return;
  if(m_tree.nodes == &full)
    {
	  if(rtn.size() < 1) rtn.resize(1, m_minBox);
      else               rtn[0] = m_minBox;
	  size = 1;
      return;
    }
  int depth = 1;
  parents[0] = (TreeNode*)&m_tree;
  boxes[0] = m_spanBox;
  index[1] = 0;
  while(depth != 0)
    {
      const TreeNode* parent =  parents[depth-1];
      int ind = index[depth];
      const TreeNode& current = parent->nodes[ind];
      quadrantBox(boxes[depth-1], ind, boxes[depth]);
      if(current.nodes == 0 || current.nodes == &full)
	{
	  if(current.nodes == 0){;}  // do nothing
	  else if(current.nodes == &full)
	    {
		  if(rtn.size() > size) 
			rtn[size] = boxes[depth];
		  else
			rtn.push_back(boxes[depth]);
		  ++size;
	    }
	  nextNode(depth);
	}
      else{
	parents[depth] = (TreeNode*)&(current);
	depth++;
	index[depth] = 0;
      }
    }
} 

bool TreeIntVectSet::contains(const Box& box) const
{
  if(m_tree.nodes == 0) return false;
  if(!m_minBox.contains(box)) return false;
  if(m_tree.nodes == &full)
    { 
	  return true;
    }
  int depth = 1;
  parents[0] = (TreeNode*)&m_tree;
  boxes[0] = m_spanBox;
  index[1] = 0;
  while(depth != 0)
    {
      const TreeNode* parent =  parents[depth-1];
      int ind = index[depth];
      const TreeNode& current = parent->nodes[ind];
      quadrantBox(boxes[depth-1], ind, boxes[depth]);
	  if(!box.intersects(boxes[depth]))
		nextNode(depth);
      else if(current.nodes == 0 || current.nodes == &full)
		{
		  if(current.nodes == 0)
			{
			  // OK, boxes[depth] is NOT in IntVectSet, but it does
			  // intersect 'box', hence we do not 'contain' this box.
			  return false;
			}
		  else if(current.nodes == &full)
			{
			  //do nothing
			}
		  nextNode(depth);
		}
      else{
		parents[depth] = (TreeNode*)&(current);
		depth++;
		index[depth] = 0;
      }
    }
  return true;
} 
void dumpTree(const TreeIntVectSet* set)
{
  if(set == NULL) return;
  if(set->m_tree.nodes == 0) return;
  if(set->m_tree.nodes == &TreeIntVectSet::full) 
    {
      cout << set->m_spanBox.smallEnd()
	   <<"..."<<set->m_spanBox.bigEnd()<<"\n";
      return;
    }
  int depth = set->m_depth+1;
  static Vector<int> index;
  static Vector<const TreeIntVectSet::TreeNode*> parents;
  static Vector<Box> boxes;
  if(depth > index.size())
    {
      index.resize(depth);
      parents.resize(depth);
      boxes.resize(depth);
    }
  depth = 1;
  parents[0] = &(set->m_tree);
  boxes[0] = set->m_spanBox;
  index[0] = 0;
  index[1] = 0;
  while(depth != 0)
    {
      const TreeIntVectSet::TreeNode* parent =  parents[depth-1];
      int ind = index[depth];
      const TreeIntVectSet::TreeNode& current = parent->nodes[ind];
      TreeIntVectSet::quadrantBox(boxes[depth-1], ind, boxes[depth]);
      if(current.nodes == 0 || current.nodes == &TreeIntVectSet::full)
	{
	  if(current.nodes == 0)
	    {
	      cout<<depth;
	      tab(cout, depth);
	      cout <<"0\n";
	    }
	  else if(current.nodes == &TreeIntVectSet::full)
	    {
	      cout<<depth;
	      tab(cout, depth);
	      cout << boxes[depth].smallEnd()
		   <<"..."<<boxes[depth].bigEnd()<<"\n";
	    }
	   
	  index[depth]++;
	  while(index[depth] == TreeIntVectSet::nodeSize)
	    {
	      index[depth] = 0;
	      depth--;
	      index[depth]++;
	    }
	}
      else{
	cout<<depth;
	tab(cout, depth);
	cout <<"+\n";
	parents[depth] = &(current);
	depth++;
	index[depth] = 0;
      }
    }
} 


int TreeIntVectSet::oppositeQuadrant(int ind)
{
  assert(ind < nodeSize);
  assert(ind >= 0);
  return nodeSize+~ind;
}

#include "Tuple.H"


void TreeIntVectSet::compact() const
{
  if(m_tree.nodes == 0 || m_tree.nodes == &full) return;
  int depth = 1;
  static Vector<Tuple<TreeNode*, 8> > flags;
  flags.resize(index.size());
  parents[0] = (TreeNode*)&(m_tree);
  index[1] = 0;
 
  while(depth != 0)
    {
      TreeNode* parent =  parents[depth-1];
      int ind = index[depth];
      //Vector<int>& indexRef = index;  //debug aids
      //Vector<TreeNode*>& parentRef = parents; //debug aids
      TreeNode& current = parent->nodes[ind];
      flags[depth][ind] = current.nodes;
      if(current.nodes == 0 || current.nodes == &full)
	{
	  index[depth]++;
	  while(index[depth] == nodeSize)
	    {
	      int i=0;
	      while(i< nodeSize-1 && flags[depth][i] == flags[depth][i+1]) i++;
	      if(i == nodeSize-1)
		{
		  TreeNode* parent = parents[depth-1];
		  clearTree(*parent);
		  parent->nodes = flags[depth][0];
		  flags[depth-1][index[depth-1]] = flags[depth][0] ;
		}
	      index[depth] = 0;
	      depth--;
	      index[depth]++;
	    }
	}
      else{
	parents[depth] = &(current);
	depth++;
	index[depth] = 0;
      }
    }
  
}


//======================================================================


void TreeIntVectSetIterator::begin()
{
  if(m_ivs->m_tree.nodes == 0)
    {
      m_depth = -1;
      return;
    }
  boxes[0] = m_ivs->m_spanBox;
  TreeIntVectSet::quadrantBox(boxes[0], 0, boxes[1]);
  nodes[0] = (TreeIntVectSet::TreeNode*)&(m_ivs->m_tree); 
  m_depth = 1;
  index[1] = -1;
  if( m_ivs->m_tree.nodes == &TreeIntVectSet::full)
    {
      m_current = boxes[0].smallEnd();
      m_depth = 0;
      return;
    }
  else{
    findNextNode();
  }
  assert(m_ivs->minBox().contains(m_current));
}

void TreeIntVectSetIterator::findNext()
{
  if(TreeIntVectSet::nextIntVect(boxes[m_depth], m_current))
    return;
  else
    {
      findNextNode();
    }
}

void TreeIntVectSetIterator::findNextNode()
{    
  static TreeIntVectSet::TreeNode* full = &TreeIntVectSet::full;
  index[m_depth]++;
  while(index[m_depth] == TreeIntVectSet::nodeSize)
    {
      m_depth--;
      index[m_depth]++;
    }
  while(m_depth > 0)
    {
      const TreeIntVectSet::TreeNode* parent =  nodes[m_depth-1];
      assert(index[m_depth] >= 0);
      const TreeIntVectSet::TreeNode& current = parent->nodes[index[m_depth]];
      TreeIntVectSet::quadrantBox(boxes[m_depth-1], index[m_depth], boxes[m_depth]);
      if(current.nodes == full)
	{
	  m_current = boxes[m_depth].smallEnd();
	  return;
	}
      else if(current.nodes == 0)
	{
	  index[m_depth]++;
	  while(index[m_depth] == TreeIntVectSet::nodeSize)
	    {
	      m_depth--;
	      index[m_depth]++;
	    }
	}
      else{
	nodes[m_depth] = &(current);
	m_depth++;
	index[m_depth] = 0;
      }
    }
  m_depth = -1;
}

Vector<Box> TreeIntVectSet::boxes(2);
TreeIntVectSet::TreeNode TreeIntVectSet::full;
Vector<int> TreeIntVectSet::index(2);
Vector<TreeIntVectSet::TreeNode*> TreeIntVectSet::parents(2);
Pool TreeIntVectSet::treeNodePool(sizeof(TreeNode[nodeSize]));

