/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
// DTGraves, Fri, Dec 3, 1999

#ifndef _UGIO_
#define _UGIO_
#include "LevelData.H"
#include "REAL.H"
#include "Vector.H"
#include "FArrayBox.H"
#include "IntVectSet.H"
#include <string>
using std::string;
using std::istream;

#ifdef HDF5
///
/**
   Writes a single level in HDF5 format.  Only available if the
   preprocessor macro HDF5 is defined at compilation.
   This is blocking.
   {\bf Arguments:}\\
   filename  :  file to output to.\\
   a_Grid : grids \\
   a_Data :  data \\
   a_Names:  names of variables.\\
   a_domain :  domain of level.\\

*/
void
WriteUGHDF5(const string& filename,                             
            const DisjointBoxLayout& a_Grids, 
            const LevelData<FArrayBox>& a_Data,
            const Box& a_domain);

///
/**
   Writes a single level in HDF5 format.  Only available if the
   preprocessor macro HDF5 is defined at compilation.
   This is the hdf5 handle version and is not blocking.

   {\bf Arguments:}\\
   handle  :  handle to output to.\\
   a_Grid : grids \\
   a_Data :  data \\
   a_Names:  names of variables.\\
   a_domain :  domain of level.\\

*/
void
WriteUGHDF5(HDF5Handle& handle,
            const DisjointBoxLayout& a_grids, 
            const LevelData<FArrayBox>& a_data,
            const Box& a_domain);


///
/**
   Reads a  level of data in HDF5 format.  Only available if the
   preprocessor macro HDF5 is defined at compilation.  Returns 0 on success.
   This function is blocking
   {\bf Arguments:}\\
   filename  :  file to input from.\\
   a_Grids : grids \\
   a_Data :  data \\
   a_domain :  domain \\
   (ith entry is refinement ratio between levels i and i + 1).\\
   a_numLevels :  number of levels to read.\\

   {\bf Returns:} \\
   status code with values:\\
   0: success \\
   -1: bogus number of levels \\
   -2: bogus number of components \\
   -3: error in readlevel \\
   -4: file open failed \\
*/
int
ReadUGHDF5(const string& filename,                             
           DisjointBoxLayout& a_grids, 
           LevelData<FArrayBox>& a_Data,
           Box& a_domain,                               
           const IntVect& a_ghostVector = IntVect::TheZeroVector());

///
/**
   This is the handle version of ReadUGHDF5.
   This is not blocking.   
   Reads a  level of data in HDF5 format.  Only available if the
   preprocessor macro HDF5 is defined at compilation.  Returns 0 on success.

   {\bf Arguments:}\\
   handle  :  hdf5 handle to input from.\\
   a_Grids : grids \\
   a_Data :  data \\
   a_domain :  domain \\
   (ith entry is refinement ratio between levels i and i + 1).\\
   a_numLevels :  number of levels to read.\\

   {\bf Returns:} \\
   status code with values:\\
   0: success \\
   -1: bogus number of levels \\
   -2: bogus number of components \\
   -3: error in readlevel \\
   -4: file open failed \\
*/
int
ReadUGHDF5(HDF5Handle& handle,
           DisjointBoxLayout& a_grids, 
           LevelData<FArrayBox>& a_data,
           Box& a_domain,                               
           const IntVect& a_ghostVector);
#endif

#endif
