/* _______              __
  / ___/ /  ___  __ _  / /  ___
 / /__/ _ \/ _ \/  ' \/ _ \/ _ \
 \___/_//_/\___/_/_/_/_.__/\___/ 
*/
//
// This software is copyright (C) by the Lawrence Berkeley
// National Laboratory.  Permission is granted to reproduce
// this software for non-commercial purposes provided that
// this notice is left intact.
// 
// It is acknowledged that the U.S. Government has rights to
// this software under Contract DE-AC03-765F00098 between
// the U.S.  Department of Energy and the University of
// California.
//
// This software is provided as a professional and academic
// contribution for joint exchange. Thus it is experimental,
// is provided ``as is'', with no warranties of any kind
// whatsoever, no support, no promise of updates, or printed
// documentation. By using this software, you acknowledge
// that the Lawrence Berkeley National Laboratory and
// Regents of the University of California shall have no
// liability with respect to the infringement of other
// copyrights by any part of this software.
//
// DTGraves, Fri, Dec 3, 1999

#include <fstream>
#include <string>
using std::fstream;
using std::string;
#include <cstdio>
#include <cmath>
#include "HDF5.H"
#include "UGIO.H"
#include "SPMD.H"

#ifdef HDF5

void
WriteUGHDF5(const string& filename,                             
            const DisjointBoxLayout& a_grids, 
            const LevelData<FArrayBox>& a_data,
            const Box& a_domain)
{
  HDF5Handle handle(filename.c_str(),  HDF5Handle::CREATE);

  WriteUGHDF5(handle, a_grids, a_data, a_domain);

#ifdef MPI  
  MPI_Barrier(Chombo_MPI::comm);
#endif   

  handle.close();
}

void
WriteUGHDF5(HDF5Handle& handle,
            const DisjointBoxLayout& a_grids, 
            const LevelData<FArrayBox>& a_data,
            const Box& a_domain)
{
  HDF5HeaderData header;
  int nComp = a_data.nComp();

  int numlevels = 1;
  string filedescriptor("VanillaAMRFileType");
  header.m_string ["filetype"]      = filedescriptor;
  header.m_int ["num_levels"]       =  numlevels;
  header.m_int ["num_components"]    = nComp;

  for(int ivar = 0; ivar < nComp; ivar++)
    {
      char labelChSt[80];
      sprintf(labelChSt, "component_%d", ivar);
      string label(labelChSt);
      header.m_string[label] = label;
    }
  header.writeToFile(handle);

  Real dtLevel = 1.0;
  Real dxLevel = 1.0;
  Real time = 1.0;
  int refLevel = 1;
  int ilev = 0;

  int eek = writeLevel(handle, ilev, a_data, 
                       dxLevel, dtLevel,  time,
                       a_domain, refLevel);
  if(eek != 0)
    MayDay::Error("WriteUGHDF5: Error in writeLevel");
}

int
ReadUGHDF5(const string& filename,                             
           DisjointBoxLayout& a_grids, 
           LevelData<FArrayBox>& a_data,
           Box& a_domain,                               
           const IntVect& a_ghostVector)
{
  HDF5Handle handle;
  int err = handle.open(filename.c_str(),  HDF5Handle::OPEN_RDONLY);
  if( err < 0) return -4;

  int eekflag = ReadUGHDF5(handle, a_grids, a_data, a_domain, a_ghostVector);

#ifdef MPI  
  MPI_Barrier(Chombo_MPI::comm);
#endif   

  handle.close();
  return (eekflag);
}

int
ReadUGHDF5(HDF5Handle& handle,
           DisjointBoxLayout& a_grids, 
           LevelData<FArrayBox>& a_data,
           Box& a_domain,                               
           const IntVect& a_ghostVector)
{
  HDF5HeaderData header;
  header.readFromFile(handle);

  int numLevels = header.m_int["num_levels"];
  if(numLevels != 1)
    {
      MayDay::Warning("Tried to read multilevel data using ReadUGHDF5");
      return -5;
    }

  int nComp = header.m_int["num_components"];
  if(nComp <= 0)
  {
    MayDay::Warning("ReadUGHDF5: Bogus number of Components");
    return (-2);
  }

  
  int ilev = 0;
  Real dtLevel;
  Real dxLevel;
  Real time;
  int refLevel =1;
  int eek = readLevel(handle, ilev, a_data, 
                      dxLevel, dtLevel, time,
                      a_domain, refLevel, nComp, a_ghostVector, true);
  if(eek != 0)
    {
      MayDay::Warning("ReadUGHDF5: readLevel failed");
      return (-3);
    }
  
  a_grids = a_data.getBoxes();
  
  return (0);
}
#endif
