// -*- Mode: C++; Modified: "Tue 08 Feb 2000 17:09:19 by dave"; -*- 

// 
// This software is copyright (C) by the Lawrence Berkeley National
// Laboratory.  Permission is granted to reproduce this software for
// non-commercial purposes provided that this notice is left intact.
//  
// It is acknowledged that the U.S. Government has rights to this
// software under Contract DE-AC03-765F00098 between the U.S.  Department
// of Energy and the University of California.
//  
// This software is provided as a professional and academic contribution
// for joint exchange. Thus it is experimental, is provided ``as is'',
// with no warranties of any kind whatsoever, no support, no promise of
// updates, or printed documentation. By using this software, you
// acknowledge that the Lawrence Berkeley National Laboratory and Regents
// of the University of California shall have no liability with respect
// to the infringement of other copyrights by any part of this software.
//  

// 
//  The simplest of all possible vector classes
//  dtg
//  ANAG 10-28-97
// 

#ifndef _SIMP_VECTOR_H_
#define _SIMP_VECTOR_H_

#include <cmath>
#include <cstdlib>
#include <cassert>
#include <vector>


using std::vector;
template <class T>
/// one dimensional dynamic array
/**
   Vector is a resizable one-dimensional array with constant-time
   random access and range checking.  The template type T must have a
   default constructor, a copy constructor, and an assignment
   operator.

   Vector<T> is publically derived from std::vector<T>.  Read the
   friendly STL manual.  
*/
class Vector: public vector<T>
{
public:
  /// 
  /** 
      Default constructor. 
      Creates a Vector of zero length with null data.
  */
  inline
  Vector():vector<T>()
  {
  }

  ///
  /**
     Copy constructor.
  */
  inline
  Vector(const Vector<T>& invec):vector<T>(invec)
  {
  }

  ///
  /** 
      Constructs a Vector with given number of elements.\\

      {\bf Arguments:} \\
      size (not modified):  number of elements of Vector to construct.\\
      {\bf This:} \\
      -------The object is modified----------

  */
  inline
  Vector(/*long*/ unsigned int isize)
  {
    resize(isize);
  }

  /// 
  /** 
      Constructs a Vector with given number of elements and constant value.\\

      {\bf Arguments:} \\
      size (not modified):  number of elements of Vector to construct.\\
      value (not modified):  value to set every element to.\\
      {\bf This:} \\
      -------The object is modified----------
      
  */
  inline
  Vector(/*long*/ unsigned int isize, const T& value) : vector<T>(isize, value)
  {}

  /// 
  /** 
      Returns a modifiable lvalue reference to the value of the given
      element in this Vector.  It is an error if n < 0 or n >= this->size(). \\

      {\bf Arguments: }  \\
      n (not modified) index of desired element.\\
      {\bf Returns:} \\
      modifiable reference to value in Vector at index n.\\
      {\bf This:} \\
      ----- 
      This object is modified if the returned reference is assigned a new value
      ----- 
  */
  inline
  T& operator[] (/*long*/ unsigned int n) 
  { 
    assert(n < size());
    //vector<T>* svec = static_cast<vector<T>* >(this);
    //return  svec->operator[](n);
    return  (static_cast<vector<T>* >(this))->operator[](n);
  } 
  
  /// 
  /** 
      Returns a constant reference to the given element in this
      Vector.\\

      {\bf Arguments: }  \\
      n (not modified) index of desired element.\\
      {\bf Returns:} \\
      constant reference to value in Vector at index n.\\
      {\bf This:} \\
      This object is not modified.  */
  inline
  const T& operator[] (/*long*/ unsigned int n) const
  { 
    assert(n < size());
    //const vector<T>* svec = static_cast<const vector<T>* >(this);
    //return  svec->operator[](n);
    return  (static_cast<const vector<T>* >(this))->operator[](n);
  } 
  
  ///
  /**
     Modifies this Vector by appending the elements of the argument
     Vector.  The new Vector will have a size of this->size() +
     invec.size() (where this Vector is considered before the append is
     performed).  The first element of invec will have index
     this->size(), the second element will have index this->size()+1,
     etc.\\

     {\bf Arguments:} \\
     invec (not modified):  Vector whose elements to append to this Vector.\\
     {\bf This:} \\
     -------The object is modified---------- */
  inline
  void  append(const Vector<T>& invec)
  {
    for (int ivec = 0; ivec <  invec.size(); ivec++)
      {
        push_back(invec[ivec]);
      }
  }

  inline
  void resize(/*long*/ unsigned int isize)
  {
    vector<T>* svec = static_cast<vector<T>* >(this);
	unsigned int l= size();
    svec->resize(isize);
    for(; l<isize; ++l)
      svec->operator[](l) = T();
  }

  inline
  void resize(/*long*/ unsigned int isize, const T& value)
  {
    vector<T>* svec = static_cast<vector<T>* >(this);
    svec->resize(isize, value);
  }
};


#endif

