/*
 * Copyright (c) 2011 trialox.org (trialox AG, Switzerland).
 *
 */
package test.memoryleak;

import com.hp.hpl.jena.datatypes.TypeMapper;
import com.hp.hpl.jena.graph.Graph;
import com.hp.hpl.jena.graph.Node;
import com.hp.hpl.jena.graph.Triple;
import com.hp.hpl.jena.query.Dataset;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.tdb.TDBFactory;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.UUID;
import org.openjena.atlas.lib.Base64;

/**
 *
 * @author hasan
 */
public class RetrieveBase64Binary {

	private static final String XML_LITERAL_TYPE = "http://www.w3.org/1999/02/22-rdf-syntax-ns#XMLLiteral";

	private static final String XSD_BASE64BINARY_TYPE = "http://www.w3.org/2001/XMLSchema#base64Binary";

	private static final String GRAPH_NAME = "http://localhost/test.graph";

	public static void main(String... args) throws UnsupportedEncodingException {
		org.openjena.riot.SysRIOT.wireIntoJena();
		Node.cache(false);
		int count = 0;
		if (args.length > 0) {
			try {
				count = Integer.parseInt(args[0]);
			} catch (NumberFormatException e) {
			}
		}
		int typedLiteralType = 1;
		if (args.length > 1) {
			try {
				typedLiteralType = Integer.parseInt(args[1]);
			} catch (NumberFormatException e) {
			}
		}
		System.out.print("Number of triples to be generated: ");
		System.out.println(count);
		if (count > 0) {
			System.out.print("Type of TypedLiteral chosen: ");
			System.out.println(typedLiteralType == 1 ? XML_LITERAL_TYPE : XSD_BASE64BINARY_TYPE);
		}

		File mGraphsDir = new File("tdb-data/mgraph/");
		File tcDir = getTcDir(mGraphsDir, GRAPH_NAME);
		if (!tcDir.exists()) {
			tcDir.mkdirs();
			File optimizationIndicator = new File(tcDir, "fixed.opt");
			try {
				optimizationIndicator.createNewFile();
			} catch (IOException ex) {
				throw new RuntimeException(ex);
			}
		}

		Dataset dataset = TDBFactory.createDataset(tcDir.getAbsolutePath());
		Model model = dataset.getDefaultModel();
		Graph jenaGraph = model.getGraph();

		if (count > 0) {
			System.out.println("Generating triples...");
			generateTriples(jenaGraph, count, typedLiteralType);
			System.out.println("Triples generated");
		}

		final ExtendedIterator jenaIter = jenaGraph.find(null, null, null);
		while (jenaIter.hasNext()) {
			Triple triple = (Triple) jenaIter.next();
		}

		jenaGraph.close();
		dataset.close();

		System.out.println("Test finished.");
	}

	private static File getTcDir(File directory, String name) {
		try {
			String subDirName = URLEncoder.encode(name, "utf-8");
			return new File(directory, subDirName);
		} catch (UnsupportedEncodingException ex) {
			throw new RuntimeException("utf-8 not supported", ex);
		}
	}

	private static void generateTriples(Graph jenaGraph, int count, int type) throws UnsupportedEncodingException {
		Node subject;
		Node predicate;
		Node object;
		Random r = new Random();
		List<Node> nodeList = new ArrayList<Node>();
		List<Node> bNodeList = new ArrayList<Node>();
		List<Node> predicateList = new ArrayList<Node>();
		for (int i = 0; i < count; i++) {
			subject = getNonLiteral(nodeList, bNodeList, r);
			if (r.nextInt(2) == 0 && predicateList.size() > 0) {
				predicate = predicateList.get(r.nextInt(predicateList.size()));
			} else {
				predicate = Node.createURI("http://example.org/" + UUID.randomUUID().toString());
				predicateList.add(predicate);
			}
			switch (r.nextInt(6)) {
				case 0:
					object = getNonLiteral(nodeList, bNodeList, r);
					break;
				case 1:
					object = Node.createLiteral(UUID.randomUUID().toString());
					break;
				case 2:
					object = Node.createLiteral("<p>"+UUID.randomUUID().toString()+"</p>", null,
							TypeMapper.getInstance().getSafeTypeByName(XML_LITERAL_TYPE));
				default:
					if (type == 1) {
						object = Node.createLiteral("<p>"+getLargeString(1048576)+"</p>", null,
								TypeMapper.getInstance().getSafeTypeByName(XML_LITERAL_TYPE));
					} else {
						object = Node.createLiteral(Base64.encodeBytes(getLargeString(1048576).getBytes("UTF-8")), null,
								TypeMapper.getInstance().getSafeTypeByName(XSD_BASE64BINARY_TYPE));
					}
			}
			jenaGraph.add(new Triple(subject, predicate, object));
		}
	}

	private static Node getNonLiteral(List<Node> nodeList, List<Node> bNodeList, Random r) {
		if (r.nextInt(2) == 0) {
			return getBNode(bNodeList, r);
		} else {
			return getNode(nodeList, r);
		}
	}

	private static Node getBNode(List<Node> bNodeList, Random r) {
		if (r.nextInt(2) == 0 && bNodeList.size() > 0) {
			return bNodeList.get(r.nextInt(bNodeList.size()));
		} else {
			Node bNode = Node.createAnon();
			bNodeList.add(bNode);
			return bNode;
		}
	}

	private static Node getNode(List<Node> nodeList, Random r) {
		if (r.nextInt(2) == 0 && nodeList.size() > 0) {
			return nodeList.get(r.nextInt(nodeList.size()));
		} else {
			Node node = Node.createURI("http://example.org/" + UUID.randomUUID().toString());
			nodeList.add(node);
			return node;
		}
	}

	private static String getLargeString(int approxLength) {
		StringBuilder sb = new StringBuilder(approxLength);
		while (sb.length() < approxLength) {
			sb.append(UUID.randomUUID().toString());
		}
		return sb.toString();
	}
}
