/*+++++++++++++++++
  connect.c - read/write replacements with timeouts
  hoenicka_markus@compuserve.com 2-10-00
  $Id: connect.c,v 1.4 2000/06/07 06:12:17 markus Exp $ 
  +++++++++++++++++*/

/* This file is inspired considerably by a similar file in Wget (Copyright (C) 1995, 1996, 1997 Free Software Foundation, Inc.)

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <unistd.h>

#include <errno.h>
#include <string.h>

#include "connect.h"

extern int n_refdb_timeout; /* timeout in seconds for read/write on sockets */

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  select_fd(): Wait for file descriptor FD to be readable, MAXTIME
  being the timeout in seconds.  If WRITEP is non-zero, checks for FD
  being writable instead.

  int select_fd returns 1 if FD is accessible, 0 for timeout
  and -1 for error in select().

  int fd the file descriptor of a socket

  int maxtime the time in seconds that the function will wait before
      a timeout occurs

  int writep if 1, checks for fd being writable. if 0, checks for fd
      being readable

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int select_fd (int fd, int maxtime, int writep)
{
  fd_set fds, exceptfds;
  struct timeval timeout;

  FD_ZERO (&fds);
  FD_SET (fd, &fds);
  FD_ZERO (&exceptfds);
  FD_SET (fd, &exceptfds);
  timeout.tv_sec = maxtime;
  timeout.tv_usec = 0;
  /* HPUX reportedly warns here.  What is the correct incantation?  */
  return select (fd + 1, writep ? NULL : &fds, writep ? &fds : NULL,
		 &exceptfds, &timeout);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  iread(): Read at most LEN bytes from FD, storing them to BUF.  This
  is virtually the same as read(), but takes care of EINTR braindamage
  and uses select() to timeout the stale connections (a connection is
  stale if more than n_timeout time is spent in select() or read()).  

  int iread the number of bytes read from fd, or -1 if timeout

  char *buf a pointer to a character buffer which receives the data

  int len the number of bytes that iread will attempt to read from fd

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int iread (int fd, char *buf, int len) {
  int res;
  int n_byte_read = 0;

  while (len > 0) { /* read until we have all or until timeout occurs */
    do {
      if (n_refdb_timeout) {
	do {
	  res = select_fd (fd, n_refdb_timeout, 0);
	} while (res == -1 && errno == EINTR);
	if (res <= 0) {
	  /* Set errno to ETIMEDOUT on timeout.  */
	  if (res == 0)
	    /* #### Potentially evil!  */
	    errno = ETIMEDOUT;
	  return -1;
	}
      }
      res = read(fd, buf, len); /* try to read data */
    } while (res == -1 && errno == EINTR);
    if (res <= 0)
      break;
    n_byte_read += res;
    buf += res;
    len -= res;
  }
/*    printf("%d\n", n_byte_read); */
  return n_byte_read;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  tread(): Read at most LEN bytes from FD, storing them to BUF.  This
  is virtually the same as iread(), but it checks after each success-
  ful read() whether a string is complete (i.e. whether a '\0' was
  received). In this case, the function returns immediately, instead
  of timing out, even if less byte than requested were received.

  int tread the number of bytes read from fd, or -1 if timeout

  char *buf a pointer to a character buffer which receives the data

  int len the number of bytes that iread will attempt to read from fd

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int tread (int fd, char *buf, int len) {
  int res;
  int n_byte_read = 0;

  while (len > 0) { /* read until we have all, a complete string, or timeout */
    do {
      if (n_refdb_timeout) {
	do {
	  res = select_fd (fd, n_refdb_timeout, 0);
/*  	  printf("tread:res=%d\n", res); */
/*  	  perror("tread"); */
	} while (res == -1 && errno == EINTR);
	if (res <= 0) {
	  /* Set errno to ETIMEDOUT on timeout.  */
	  if (res == 0)
	    /* #### Potentially evil!  */
	    errno = ETIMEDOUT;
	  return -1;
	}
      }
      res = read(fd, buf, len); /* read some data */
/*  	  printf("tread0:fd:%d res=%d\n", fd, res); */
/*  	  perror("tread0"); */
      if (res > 0) { /* see whether we've got a complete string */
	if (buf[res-1] == '\0') { /* complete string received */
	  n_byte_read += res;
/*  	  printf("tread1:n_byte_read=%d\n", n_byte_read); */
/*  	  perror("tread1"); */
	  return n_byte_read; /* get back w/o timeout */
	}
      }
    } while (res == -1 && errno == EINTR);
    if (res <= 0)
      break;
    n_byte_read += res;
    buf += res;
    len -= res;
  }
/*    printf("tread2:n_byte_read=%d\n", n_byte_read); */
/*    perror("tread2"); */
  return n_byte_read;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  iwrite(): Write LEN bytes from BUF to FD.  This is similar to
  iread().  It makes sure that all of BUF is actually
  written to FD, so callers needn't bother with checking that the
  return value equals to LEN.  Instead, you should simply check
  for -1.

  int iwrite the number of bytes actually written to fd, or -1 if
             timeout

  char *buf a pointer to a character buffer which holds the data

  int len the number of bytes that iwrite will attempt to write to fd

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int iwrite (int fd, const char *buf, int len) {
  int res = 0;
  int n_byte_written = 0;

  /* `write' may write less than LEN bytes, thus the outward loop
     keeps trying it until all was written, or an error occurred.  The
     inner loop is reserved for the usual EINTR f*kage, and the
     innermost loop deals with the same during select().  */
  while (len > 0) {
    do {
      if (n_refdb_timeout) {
	do {
	  res = select_fd (fd, n_refdb_timeout, 1);
	} while (res == -1 && errno == EINTR);
	if (res <= 0) {
	  /* Set errno to ETIMEDOUT on timeout.  */
	  if (res == 0)
	    /* #### Potentially evil!  */
	    errno = ETIMEDOUT;
	  return -1;
	}
      }
      res = write (fd, buf, len); /* write some data */
/*        printf("iwrite0:%d\n", res); */
      n_byte_written += res;
    } while (res == -1 && errno == EINTR);
    if (res <= 0)
      break;
    buf += res;
    len -= res;
  }
/*    printf("iwrite1:%d\n", n_byte_written); */
  return n_byte_written;
}
