/* pref.c read, write, and manipulate preferences */
/* hoenicka_markus@compuserve.com 2-8-00 */
/* $Id: pref.c,v 1.3 2000/02/15 08:04:18 markus Exp markus $ */
/* Todo: first check site config file /etc/???/init_file, then ~/init_file */

#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <stdlib.h>

#include "pref.h"

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  read_prefs(): reads preferences from an init file                    
                The init file format is simple: Each line contains a   
                pair of variable name and variable value, separated by
	        whitespace. A # outcomments a line or a value token.
	        A variable name may have up to 16 characters, the
		value may have up to 255 characters (_POSIX_PATH_MAX)

  int read_prefs returns 0 if successful, 1 if error

  Prefs* prefs ptr to an array of Prefs structures
 
  char *init_file ptr to a string with the name of the init file.
                  This is first looked up in HOME, then in PWD

  int numOpts maximum number of options to retrieve

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int read_prefs(Prefs* prefs, char* init_file, int numOpts) {
  FILE *fp;
  char line [PREFS_BUF_LEN]; /* acommodates a variable name of max.
				     16 chars, a tab, a POSIX full path,
				     and a \0 */
  char *name_token;
  char *value_token;
  char *home; /* value of the HOME environment variable */
  char *refdblib; /* value of the REFDBLIB environment variable */
  int i;
  char theinitfile[_POSIX_NAME_MAX+1];
  char have_global = 0;

  home = getenv("HOME");
  refdblib = getenv("REFDBLIB");

  line[0] = '\0';

  /* first try to read the settings in $REFDBLIB/.init */
  if (refdblib != NULL) { /* if REFDBLIB is set, try reading this first */
    strcpy(theinitfile, refdblib);
    strcat(theinitfile, "/");
    strcat(theinitfile, init_file);
    fp = fopen(theinitfile, "r");
    if (fp != NULL) {    
      have_global = 1;
      while (fgets(line, PREFS_BUF_LEN, fp) != NULL) {
	/* discard comment lines right away*/
	if (line[0] == '#') {
	  continue;
	}
    
	line[PREFS_BUF_LEN-1] = '\0';
	name_token = strtok(line, " \t\r\n");
	if (name_token == NULL || name_token[0] == '#') {
	  continue;
	}

	/* check also for # in value token, so we can outcomment values to get
	   the default */
	value_token = strtok(NULL, " \t\r\n");
	if (value_token == NULL || value_token[0] == '#') {
	  continue;
	}

	for (i = 0; i < numOpts; i++) {
	  if (strcmp(prefs[i].varname, name_token) == 0) {
	    strcpy(prefs[i].varvalue, value_token);
	    prefs[i].varvalue[PREFS_BUF_LEN-1] = '\0';
	    break;
	  }
	}
      }    
      fclose(fp);
    }
  }

  /* now try to read the settings in $HOME/.init */
  if (home == NULL) {    /* if HOME is not set, try PWD */
    fp = fopen(init_file, "r");
    if (fp == NULL) {
      return ((have_global) ? 0:1);
    }
  }
  else {
    strcpy(theinitfile, home);
    strcat(theinitfile, "/");
    strcat(theinitfile, init_file);
    fp = fopen(theinitfile, "r");
    if (fp == NULL) {    /* if not present in HOME, try PWD */
      fp = fopen(init_file, "r");
      if (fp == NULL) {
	return ((have_global) ? 0:1);
      }
    }
  }

  line[0] = '\0';

  while (fgets(line, PREFS_BUF_LEN, fp) != NULL) {
    /* discard comment lines right away*/
    if (line[0] == '#') {
      continue;
    }
    
    line[PREFS_BUF_LEN-1] = '\0';
    name_token = strtok(line, " \t\r\n");
    if (name_token == NULL || name_token[0] == '#') {
      continue;
    }

    /* check also for # in value token, so we can outcomment values to get
       the default */
    value_token = strtok(NULL, " \t\r\n");
    if (value_token == NULL || value_token[0] == '#') {
      continue;
    }

    for (i = 0; i < numOpts; i++) {
      if (strcmp(prefs[i].varname, name_token) == 0) {
	strcpy(prefs[i].varvalue, value_token);
	prefs[i].varvalue[PREFS_BUF_LEN-1] = '\0';
	break;
      }
    }
  }    
  fclose(fp);

  return 0;
}

