/*+++++++++++++++++
  refdb-admin - the refdb administration console application
  hoenicka_markus@compuserve.com 2-10-00
  $Id: refdb-admin.c,v 1.22 2000/09/22 04:30:07 markus Exp markus $ 
  +++++++++++++++++*/

/* ToDo: implement the getuser command */
/* ToDo: implement a fixdb command to ensure the integrity of the database ?? */
/* ToDo: if no password is specified in the init file or the command line, ask for it at startup */

/* general includes */
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <signal.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <readline/readline.h>
#include <readline/history.h>

#include <unistd.h> 
#include <limits.h>
#include <mysql/mysql.h>  /* needed to define HOSTNAME_LENGTH and USERNAME_LENGTH */
#include <getopt.h>

/* these defines appeared in MySQL 3.22 */
#ifdef MYSQL321
#define HOSTNAME_LENGTH 60
#define USERNAME_LENGTH 16
#endif

/* our own stuff */
#include "refdb.h"  /* common stuff for all refdb applications*/
#include "pref.h" /* for init file */
#include "strfncs.h" /* for is functions */
#include "refdb-admin.h" /* stuff specific to this file */
#include "readln.h"  /* readline-related  stuff */
#include "page.h" /* pager functions */
#include "refdb-client.h" /* stuff common to all clients */
#include "connect.h" /* modified read/write socket functions */
#include "tokenize.h" /* tokenizes command-lines */
#include "readris.h"

/*+ the commands array contains the user commands, the functions called, and
  short explanatory messages +*/
COMMAND commands[] = {
  { "help", com_help, "Display this text" },
  { "?", com_help, "Synonym for `help'" },
  { "quit", com_quit, "Quit using refdb-admin" },
  { "adduser", com_adduser, "Add or remove users" },
  { "createdb", com_createdb, "Create a new database" },
  { "deletedb", com_deletedb, "Delete a database" },
  { "confserv", com_confserv, "Configure the application server" },
  { "listdb", com_listdb, "List databases" },
  { "getuser", com_getuser, "Get a list of users" },
  { "viewstat", com_viewstat, "View statistics" },
  { "verbose", com_verbose, "Toggle verbose mode" },
  { (char *)NULL, (Function *)NULL, (char *)NULL }
};

/* Globals */

/*+ When non-zero, this global means the user is done using this program. +*/
int n_done;

/*+ this array will hold the user preferences +*/
Prefs prefs[7] = {
  {"serverip", ""},
  {"port", ""},
  {"verbose", ""},
  {"pager", ""},
  {"username", ""},
  {"passwd", ""},
  {"timeout", ""}
};

/* these are the configurable variables with the compile-time defaults */
char server_ip[PREFS_BUF_LEN] = "127.0.0.1"; /*+ default IP address of refdbs +*/
char port_address[PREFS_BUF_LEN] = "9734"; /*+ default port address of refdbs +*/
char the_pager[PREFS_BUF_LEN] = "stdout"; /*+ default "pager" (stdout) +*/
char username[USERNAME_LENGTH + 1] = ""; /*+ default username (emtpy) +*/
char passwd[PREFS_BUF_LEN] = ""; /*+ default password (empty) +*/
char refdb_timeout[PREFS_BUF_LEN] = "5"; /*+ 5 seconds default timeout +*/
int n_refdb_timeout;
int n_verbose = 0; /*+ do we want logorrhoeic output? +*/

int n_broken_pipe; /*+ 1 indicates that we attempted to write to a broken pipe +*/
int n_log_dest = 1; /* destination of log output */
int n_log_level = 0; /* level of log information that will be printed */

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  The one and only main function
  
  int main returns 0 if successful, 1 if error
  
  int argc number of arguments

  char** argv ptr to array of strings with the command line arguments
  
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int main (int argc, char** argv)
{
  char *line, *s;
  int n_opt, i;
  int n_readinit = 1; /* if 1, read config file. If 0, skip config file */

  struct sigaction act;

  /* initialize signal handler */
  n_broken_pipe = 0;
  act.sa_handler = pipehandler;
  sigemptyset(&act.sa_mask);
  act.sa_flags = 0;
  sigaction(SIGPIPE, &act, 0);

  /* initialize the array of preference values */
  prefs[0].varvalue = server_ip;
  prefs[1].varvalue = port_address;
  prefs[3].varvalue = the_pager;
  prefs[4].varvalue = username;
  prefs[5].varvalue = passwd;
  prefs[6].varvalue = refdb_timeout;

  /* a slimy hack to detect the -q option before we run getopt */
  for (i = 0; i < argc; i++) {
    if (argv[i][0] == '-' && argv[i][1] == 'q') {
      n_readinit = 0;
      break;
    }
  }

  if (n_readinit) {
    /* read config file settings */
    read_prefs(prefs, ".refdb-admin-init", 6);
  }

  /* read command line settings. These may override the config file settings */
  while ((n_opt = getopt(argc, argv, "c:hi:p:qT:u:vVw:")) != -1) {
    switch (n_opt) {
    case 'c':
      strcpy(the_pager, optarg);
      break;
    case 'h':
      printf("Usage: refdb-admin [-c pager] [-h] [-i address] [-p port] [-q] [-T time] [-v] [-V]\nOptions: -c command to run a pager\n         -h prints this help\n         -i set server IP address to address\n         -p set server port to port\n         -q ignore init-file\n         -T set timeout to time seconds        -v show version information\n         -V switch to verbose mode\n");
      exit (0);
      break;
    case ':':
      printf("Usage: refdb-admin [-c pager] [-h] [-i address] [-p port] [-q] [-T time] [-v] [-V]\nOptions: -c command to run a pager\n         -h prints this help\n         -i set server IP address to address\n         -p set server port to port\n         -q ignore init-file\n         -T set timeout to time seconds        -v show version information\n         -V switch to verbose mode\n");
      exit (1);
      break;
    case 'i':
      if (is_ip(optarg)) {
	strcpy(server_ip, optarg);
      }
      else {
	fprintf(stderr, "option -i needs a value: %s is no valid IP address\n", optarg);
	if (n_verbose) { 
	  fprintf(stderr, "A valid IP address consists of four tokens separated by periods (.). Each token has a value between 0 and 255. A valid IP address is e.g. the localhost address '127.0.0.1'.\n");
	}
	exit (1);
      }
      break;
    case 'p':
      if (is_port(optarg)) {
	strcpy(port_address, optarg);
      }
      else {
	fprintf(stderr, "option -p needs a value: %s is no valid port\n", optarg);
	if (n_verbose) {
	  fprintf(stderr, "Port addresses below 1024 are reserved for system use. refdb should use a port higher than 1024. The server and all clients must use the same port.\n");
	}
	exit (1);
      }
      break;
    case 'q':
      n_readinit = 0;
      break;
    case 'T':
      strcpy(refdb_timeout, optarg);
      break;
    case 'u':
      strncpy(username, optarg, USERNAME_LENGTH);
      username[USERNAME_LENGTH] = '\0'; /* terminate if optarg string was too long */
      break;
    case 'v':
      printf("refdb-admin $Id: refdb-admin.c,v 1.22 2000/09/22 04:30:07 markus Exp markus $ hoenicka_markus@compuserve.com\nYou may redistribute and modify this software under the terms of the GNU public license.\n");
      exit (0);
      break;
    case 'V':
      n_verbose = 1;
      break;
    case 'w':
      strcpy(passwd, optarg);
      break;
    case '?':
      fprintf(stderr, "unknown option %c: use refdb-admin -h to display usage\n", optopt);
      break;
    }
  }

/*    printf("username: %s; password: %s\n", username, passwd); */

  n_refdb_timeout = atoi(refdb_timeout);

  initialize_readline ();	/* Bind our completer. */

  /* Loop reading and executing lines until the user quits. */
  for ( ; n_done == 0; )
    {
      line = readline ("refdb_admin: ");

      if (!line)
        break;

      /* Remove leading and trailing whitespace from the line.
         Then, if there is anything left, add it to the history list
         and execute it. */
      s = stripwhite (line, 0, 0);

      if (*s)
        {
          add_history (s);
          execute_line (s, commands);
        }

      free (line);
    }
  exit (0);
}

/* **************************************************************** */
/*                                                                  */
/*                   refdb-admin Commands                           */
/*                                                                  */
/* **************************************************************** */


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  com_createdb(): create a new database

  int com_createdb  0 if successful, 1 on error

  char *arg the name of the new database. This must be a name which
            is a valid filename on the operating system that the
            database server runs on. 

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int com_createdb (char* arg)
{
  return (0);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  com_deletedb(): delete a database

  int com_deletedb 0 if successful, 1 if error 

  char *arg the name of the database which is to be deleted

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int com_deletedb (char* arg)
{
  return (0);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  com_confserv(): configure application server

  int com_confserv 0 if successful, 1 if error

  char *arg command argument. Currently supported by refdbs are stop,
            ping, serverip value, timeout value, logfile value,
            logdest value, loglevel value

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int com_confserv (char* arg)
{
  int n_sockfd; /* file descriptor of the socket */
  char outbuffer[COMMAND_INBUF_LEN] = "confserv ";
  char inbuffer[OUTBUF_LEN];
  int numbyte;
  int n_read_done = 0;
  FILE *pagerfp;

  if (strncmp(arg, "-h", 2) == 0) {
    printf("Configures the application server\nSyntax: confserv [-h] command\nOptions: -h           prints this mini-help\nSupported commands are:\n  stop            stops the application server\n  ping            tests whether the application server is alive\n  serverip value  set the serverip variable to value\n  timeout value   set the timeout variable to value\n  logfile value   set the logfile variable to value\n  logdest value   set the logdest variable to value\n  loglevel value  set the loglevel variable to value\n");
    return 0;
  }

  if (connect_to_server(&n_sockfd, server_ip, port_address) != 0) {
    return 1;
  }
  strcat(outbuffer, arg);
  numbyte = iwrite(n_sockfd, outbuffer, strlen(outbuffer)+1);
  if (numbyte == -1) {
    fprintf(stderr, "could not write to refdbs. Stop\n");
    close(n_sockfd);
    return (1);
  }

  pagerfp = openpager(the_pager);
  do {
    numbyte = tread(n_sockfd, inbuffer, OUTBUF_LEN);
    if (numbyte == -1) {
      fprintf(stderr, "could not read from refdbs. Stop\n");
      close(n_sockfd);
      closepager(pagerfp);
      n_broken_pipe = 0;
      return (1);
    }

    if (inbuffer[numbyte-1] == '\0') { /* if transmission ends */
      n_read_done++;
    }
    /* write numbyte chars to output, unless this is the last chunk: we do not
       want to write the terminating \0 */
    if (!n_broken_pipe) {
      fwrite(inbuffer, sizeof(char), (n_read_done) ? numbyte-1 : numbyte, pagerfp);
    }
    /*  printf("%s", inbuffer); */
  } while (!n_read_done);
  /*  printf("\n"); */
  n_broken_pipe = 0;
  closepager(pagerfp);
  close(n_sockfd);

  /* due to the odd implementation of the confserv command on the server side (the command is handled by a child and queued to the parent, so any changes take effect only when the *next* command is accepted. this is ok for all commands except for stop which users expect to be immediate) we have to send a second command to really let the server stop */
  if (strncmp(arg, "stop", 4) == 0) {
    if (connect_to_server(&n_sockfd, server_ip, port_address) != 0) {
      return 1;
    }
    strcpy(outbuffer, "confserv boom"); /* the actual command is irrelevant */
    numbyte = iwrite(n_sockfd, outbuffer, strlen(outbuffer)+1);
    if (numbyte == -1) {
      fprintf(stderr, "could not write to refdbs. Stop\n");
      close(n_sockfd);
      return (1);
    }
  }
  return (0);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  com_viewstat(): view statistics

  int com_viewstat 0 if successful, 1 if error

  char *arg command argument, currently ignored by refdbs

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int com_viewstat (char* arg)
{
  return (0);
}

/* Return non-zero if ARG is a valid argument for CALLER, else print
   an error message and return zero. */
int valid_argument (char *caller, char *arg)
{
  if (!arg || !*arg)
    {
      fprintf (stderr, "%s: Argument required.\n", caller);
      return (0);
    }

  return (1);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  com_adduser(): add or remove user

  int com_adduser 0 if successful, 1 if error 

  char *arg specifies the host, the database, and the usernames

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int com_adduser (char* arg)
{
  return (0);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  com_getuser(): list all users that have some entries in their
                  personal interest lists

  int com_getuser 0 if successful, 1 if error 

  char *arg specifies the host, the database, and the usernames

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int com_getuser (char* arg)
{
  return (0);
}
