/*+++++++++++++++++
  refdb-client.c - functions common to all client applications
  hoenicka_markus@compuserve.com 3-7-00
  $Id: refdb-client.c,v 1.2 2000/06/20 04:38:04 markus Exp $
  +++++++++++++++++*/

/* socket related includes */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/un.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include "refdb.h"
#include "page.h"
#include "readln.h"  /* for COMMAND definition */
#include "refdb-client.h"

extern int n_verbose;
extern int n_done;
extern int n_broken_pipe;
extern COMMAND commands[]; 
extern char server_ip[];
extern char port_address[];
extern char the_pager[];
extern char username[];
extern char passwd[];

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  connect_to_server(): establish a connection to the application server
  
  int connect_to_server 0 if successful, not zero if failure

  int *n_sockfd pointer to the variable which receives the socket
                file descriptor if successful

  char *server_ip ip address of server

  char *port_address port address of server

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int connect_to_server (int* n_sockfd, char* server_ip, char* port_address)
{
  int n_len; /* length of the address structure */
  struct sockaddr_in address; /* the server address */
  int n_result; /* connection result */

  *n_sockfd = socket(AF_INET, SOCK_STREAM, 0);
  address.sin_family = AF_INET;
  address.sin_addr.s_addr = inet_addr(server_ip);
  address.sin_port = htons(atoi(port_address));
  n_len = sizeof(address);
  
  n_result = connect(*n_sockfd, (struct sockaddr *)&address, n_len);
  
  if (n_result != 0) {
    perror("refdb: could not establish server connection");
    if (n_verbose) {
      printf("(1) Check the settings for the server IP address and the port. You can set these either on the command line or in your refdb-admin-init file\n(2)The refdb server may be stopped or crashed or otherwise unwilling to process the request.\n(3) The machine running the server may be down or overloaded.\n");
    }
  }

  return n_result;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  com_verbose(): toggle verbose mode

  int com_verbose returns always 0

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int com_verbose (char *arg)
{
  if (strncmp(arg, "-h", 2) == 0) {
    printf("Toggles the verbose mode on and off. If on, the error messages tend to be somewhat more explicit.\nSyntax: verbose [-h]\nOptions: -h           prints this mini-help\n");
    return 0;
  }

  n_verbose = !n_verbose;
  if (n_verbose) {
    printf("Switched verbose mode ON\n");
  }
  else {
    printf("Switched verbose mode OFF\n");
  }
  return (0);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  com_quit(): The user wishes to quit using this program.

  int com_quit returns always 0

  char *arg currently not used

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int com_quit (char *arg)
{
  if (strncmp(arg, "-h", 2) == 0) {
    printf("Closes the session and exits.\nSyntax: quit [-h]\nOptions: -h           prints this mini-help\n");
    return 0;
  }

  n_done = 1;
  return (0);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  com_help(): Print out help for ARG, or for all of the commands if
              ARG is not present.

  void com_help returns always 0

  char *arg name of a command, or empty string. In the latter case
            help for all commands will be displayed

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int com_help (char *arg)
{
  register int i;
  int n_printed = 0;

  printf("This is a command overview. To get specific information about a command, run this command with the -h option.\n");

  for (i = 0; commands[i].name; i++)
    {
      if (!*arg || (strcmp (arg, commands[i].name) == 0))
        {
	  if (strlen(commands[i].name) >= 8) {
	    printf ("%s\t\t%s.\n", commands[i].name, commands[i].doc);
	  }
	  else {
	    printf ("%s\t\t\t%s.\n", commands[i].name, commands[i].doc);
	  }
          n_printed++;
        }
    }

  if (!n_printed)
    {
      printf ("No commands match `%s'.  Possibilties are:\n", arg);

      for (i = 0; commands[i].name; i++)
        {
          /* Print in six columns. */
          if (n_printed == 6)
            {
              n_printed = 0;
              printf ("\n");
            }

          printf ("%s\t", commands[i].name);
          n_printed++;
        }

      if (n_printed)
        printf ("\n");
    }
  return (0);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  com_listdb(): list databases
 
  int com_listdb 0 if successful, 1 if error

  char *arg search pattern for the database names. This must be a
            valid SQL regular expression. If this string is empty,
            the search will return all existing databases

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int com_listdb (char* arg)
{
  int n_sockfd; /* file descriptor of the socket */
  char outbuffer[COMMAND_INBUF_LEN] = "listdb ";
  char inbuffer[OUTBUF_LEN];
  int numbyte;
  int n_read_done = 0;
  FILE *pagerfp;

  if (strncmp(arg, "-h", 2) == 0) {
    printf("Lists the names of the available databases. If an argument is given as a SQL regular expression, only the matching databases are shown.\nSyntax: listdb [-h] [db]\nOptions: -h           prints this mini-help\n         All other arguments are interpreted as an SQL regular expression.\n");
    return 0;
  }

  if (connect_to_server(&n_sockfd, server_ip, port_address) != 0) {
    return 1;
  }
  strcat(outbuffer, arg);
  strcat(outbuffer, " -u ");
  strcat(outbuffer, username);
  if (strlen(passwd) > 0) {
    strcat(outbuffer, " -w ");
    strcat(outbuffer, passwd);
  }
  numbyte = iwrite(n_sockfd, outbuffer, strlen(outbuffer)+1);
  if (numbyte == -1) {
    printf("could not write to refdbs. Stop\n");
    close(n_sockfd);
    return (1);
  }

  /* openpager is guaranteed to return a valid pipe - and be it stdout */
  pagerfp = openpager(the_pager);

  do {
    numbyte = tread(n_sockfd, inbuffer, OUTBUF_LEN);
    if (numbyte == -1) {
      printf("could not read from refdbs. Stop\n");
      close(n_sockfd);
      closepager(pagerfp);
      return (1);
    }

    if (inbuffer[numbyte-1] == '\0') { /* if transmission ends */
      n_read_done++;
    }
    /* write numbyte chars to output, unless this is the last chunk: we do not
       want to write the terminating \0 */
    fwrite(inbuffer, sizeof(char), (n_read_done) ? numbyte-1 : numbyte, pagerfp);
    /*  printf("%s", inbuffer); */
  } while (!n_read_done);

  closepager(pagerfp);
  close(n_sockfd);
  return (0);
}

/* Function which tells you that you can't do this. */
/*  void too_dangerous (char *caller) */
/*  { */
/*    fprintf (stderr, */
/*             "%s: not yet implemented\n", */
/*             caller); */
/*  } */

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  pipehandler(): handler for the SIGPIPE signal. Sets the global
                 variable n_broken_pipe to non-zero. This condition
                 can be used by a function writing to a pipe to abort
                 further write attempts. Any function that uses
                 n_broken_pipe should reset it to zero when it is done
                 dealing with the broken pipe.

  void pipehandler

  int sig the received signal

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void pipehandler(int sig) {
  n_broken_pipe = 1;
}

