/* refdbs.c: refdb application server */
/* hoenicka_markus@compuserve.com 2-10-00 */
/* $Id: refdbs.c,v 1.49 2000/10/31 05:11:25 markus Exp $ */

/* ToDo: implement table locking */
/* ToDo: confserv stop is not immediate, but needs another command afterwards - modify detection of run=0 */
/* ToDo: stick tons of LOG_PRINT lines into the code */
/* ToDo: linked list for socket descriptors looks ok, but does not have much effect as long as this app is not multithreaded. Test thoroughly again as soon as multithreading is implemented. Update: With multithreading still pending (and maybe never coming) and the new fork() way of doing things the linked list does not do much anyway. The overhead is small, so probably its best to just leave it there for the time being */
/* Todo: implement signal handling */
/* ToDo: Implement a database filename "shadow" function: prefix all database names with e.g. refdb, but make this transparent for the clients */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <stdio.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <string.h>
#include <limits.h>
#include <stdlib.h>
#include <syslog.h> /* priority level definitions */
#include <time.h>
#include <signal.h>
#include <fcntl.h>
#include <mysql/mysql.h>

#include "getopt.h"

#include "pref.h"
#include "strfncs.h"
#include "refdb.h"
#include "refdbs.h"
#include "tokenize.h"
#include "connect.h"
#include "readris.h"
#include "writeris.h"
#include "linklist.h"
#include "risdb.h"

Prefs prefs[6] = {
  {"serverip", ""},
  {"timeout", ""},
  {"port", ""},
  {"logfile", ""},
  {"logdest", ""},
  {"loglevel", ""}
};

/* these definitions appear in MySQL 3.22 */
#ifdef MYSQL321
#define HOSTNAME_LENGTH 60
#define USERNAME_LENGTH 16
#endif

char outomem[] = "refdbs: out of memory";
char connerr[] = "Could not connect to database server. Stop.";
char argerr[] = "Missing argument";
char newln[] = "\n";
char positive[] = "POS";
char negative[] = "NEG";
char unknown_command[] = "Unknown command. Stop.";
char server_ip[(int)PREFS_BUF_LEN] = "127.0.0.1"; /* the IP address of the database server */
char username[USERNAME_LENGTH] = ""; /* the database username. Do we need this? */
char passwd[PREFS_BUF_LEN] = ""; /* database password. Do we need this? */
char log_file[PREFS_BUF_LEN] = "/var/log/refdbs.log"; /* custom log file */
char log_dest[PREFS_BUF_LEN] = "1"; /* log destination. 0=print on stderr, 1=use syslog, 2=use custom logfile */
char log_level[PREFS_BUF_LEN] = "6"; /* the maximum priority that actually gets sent out - priorities are from 0 (only highest importance) to 7 (send all log info). -1 means nothing gets logged */
char port_address[PREFS_BUF_LEN] = "9734"; /* default port */
char refdb_timeout[PREFS_BUF_LEN] = "5"; /* 5 seconds default timeout */
char the_fifo[_POSIX_PATH_MAX]; /* full path of fifo for child->parent msg*/
int n_refdb_timeout; /* timeout in seconds */
int n_verbose = 0; /* do we want logorrhoeic output? */
int run = 1; /* main will exit when this is set to 0 */
int n_log_level = 6; /* numeric version of log_level */
int n_log_dest = 1; /* numeric version of log_dest */
int n_reopen_log = 0; /* if 1, the log file will be reopened */
int fd_fifo = -1; /* file descriptor of the fifo (child->parent communication) */
int parent_pid; /* process id of parent process */
FILE* fp_log_file = NULL; /* a FILE pointer to a custom log file */

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  the one and only main function

  int main

  int argc number of arguments

  char** argv ptr to array of ptrs to argument strings

  returns 0 if successful, 1 if an error occurred
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int main(int argc, char** argv) {
  int server_sockfd, client_sockfd; /* sockets used for client/server dialog */
  int server_len, client_len;
  int i, n_opt; /* flags, result, */
  int n_pipe_buf = 512; /* maximum size of atomic write to fifo. 512 is the posix value. if the OS defines a larger value, this will be used below */
  struct sockaddr_in server_address;
  struct sockaddr_in client_address;
  fd_set readfds, testfds;
  int readinit = 1;
  char *msg_buffer; /* buffer for log messages */
  struct olili first_olili; /* ordered linked list for socket fds */
  char* fifo_buffer;

  /* initialize array of init file variables */
  prefs[0].varvalue = server_ip;
  prefs[1].varvalue = refdb_timeout;
  prefs[2].varvalue = port_address;
  prefs[3].varvalue = log_file;
  prefs[4].varvalue = log_dest;
  prefs[5].varvalue = log_level;
 
  /* a slimy hack to detect the -q option before we run getopt */
  for (i = 0; i < argc; i++) {
    if (argv[i][0] == '-' && argv[i][1] == 'q') {
      readinit = 0;
      break;
    }
  }

  if (readinit) { /* -q was not used */
    /* read config file settings */
    read_prefs(prefs, ".refdbs-init", 6);
  }

  /* read command line settings. These may override the config file settings */
  while ((n_opt = getopt(argc, argv, "d:f:l:hi:p:t:qvVw:")) != -1) {
    switch (n_opt) {
    case 'd':
      strncpy(log_dest, optarg, PREFS_BUF_LEN);
      log_dest[PREFS_BUF_LEN-1] = '\0';
      break;
    case 'f':
      strncpy(log_file, optarg, PREFS_BUF_LEN);
      log_file[PREFS_BUF_LEN-1] = '\0';
      break;
    case 'l':
      strncpy(log_level, optarg, PREFS_BUF_LEN);
      log_level[PREFS_BUF_LEN-1] = '\0';
      break;
    case 'h':
      printf("Usage: refdbs [-d dest] [-f file] [-l level] [-h] [-i address] [-p port] [-q] [-t time] [-u name] [-v] [-V] [-w passwd]\nOptions: -d set log destination to dest (0-2)\n         -f use file as log-file (full path)\n         -l set the log level to level (0<=level<=7)\n         -h prints this help\n         -i set server IP address to address\n         -p set port to port\n         -q ignore init-file         -t set timeout to time\n         -u set username to name\n         -v show version information\n         -V switch to verbose mode\n         -w set password to passwd\n");
      exit (0);
      break;
    case ':':
      printf("Usage: refdbs [-d dest] [-f file] [-l level] [-h] [-i address] [-p port] [-q] [-t time] [-u name] [-v] [-V] [-w passwd]\nOptions: -d set log destination to dest (0-2)\n         -f use file as log-file (full path)\n         -l set the log level to level (0<=level<=7)\n         -h prints this help\n         -i set server IP address to address\n         -p set port to port\n         -q ignore init-file         -t set timeout to time\n         -u set username to name\n         -v show version information\n         -V switch to verbose mode\n         -w set password to passwd\n");
      exit (1);
      break;
    case 'i':
      if (is_ip(optarg)) {
	strcpy(server_ip, optarg);
      }
      else {
	printf("option -i needs a value: %s is no valid IP address\n", optarg);
	if (n_verbose) { 
	  printf("A valid IP address consists of four tokens separated by periods (.). Each token has a value between 0 and 255. A valid IP address is e.g. the localhost address '127.0.0.1'.\n");
	}
	exit (1);
      }
      break;
    case 'p':
      strcpy(port_address, optarg);
      break;
    case 'q':
      readinit = 0;
      break;
    case 't':
      strcpy(refdb_timeout, optarg);
      break;
    case 'u':
      strcpy(username, optarg);
      break;
    case 'v':
      printf("refdbs $Id: refdbs.c,v 1.49 2000/10/31 05:11:25 markus Exp $ hoenicka_markus@compuserve.com\nYou may redistribute and modify this software under the terms of the GNU public license.\n");
      exit (0);
      break;
    case 'V':
      n_verbose = 1;
      break;
    case 'w':
      strcpy(passwd, optarg);
      break;
    case '?':
      printf("unknown option %c: use refdbs -h to display usage\n", optopt);
      break;
    }
  }
  
  /* translate some command line settings into numeric values */
  n_refdb_timeout = atoi(refdb_timeout);
  n_log_level = atoi(log_level);
  n_log_dest = atoi(log_dest);

  parent_pid = getpid();

  if (n_log_dest == 2) { /* use custom log file */
    if ((fp_log_file = fopen(log_file, "ab")) == NULL) {
      n_log_dest = 1; /* fall back to syslog */
      LOG_PRINT(LOG_WARNING, "could not open custom log file");
    }
  }
/*    printf("level %d; dest %d, file %s\n", n_log_level, n_log_dest, log_file); */

  LOG_PRINT(LOG_INFO, "application server started");

  /* get us some buffer for log messages */
  msg_buffer = (char*)malloc(MSG_BUF_SIZE);
  if (msg_buffer == NULL) {
    LOG_PRINT(LOG_CRIT, "Out of memory");
    exit(1);
  }

  /* add pid to fifo name. This way, multiple copies of refdbs can run */
  sprintf(the_fifo, "%s%d", FIFO_NAME, parent_pid);
  sprintf(msg_buffer, "use %s as fifo", the_fifo);
  LOG_PRINT(LOG_DEBUG, msg_buffer);

  /* prepare a fifo for child->parent communication */
#ifndef CYGWIN  /* cygwin (1.1.4) does not implement mkfifo/named pipes */
  /* create fifo if necessary and open for non-blocking read */
  if (access(the_fifo, F_OK) == -1) {
    if (mkfifo(the_fifo, 0777) != 0) {
      LOG_PRINT(LOG_WARNING, "could not create fifo");
    }
    else {
      fd_fifo = open(the_fifo, O_RDONLY | O_NONBLOCK);
    }
  }
  else {
    fd_fifo = open(the_fifo, O_RDONLY | O_NONBLOCK);
  }

  if (fd_fifo == -1) {
    LOG_PRINT(LOG_WARNING, "could not open fifo");
  }
#else
  if (access(the_fifo, F_OK) != -1) {
    unlink(the_fifo); /* most likely crash leftover */
  }
#endif /* CYGWIN */

  /* get a fifo buffer for reading data */
#ifdef PIPE_BUF
  n_pipe_buf = PIPE_BUF; /* this may be larger than _POSIX_PIPE_BUF */
#endif

  fifo_buffer = (char*)malloc(n_pipe_buf);
  if (fifo_buffer == NULL) {
    free(msg_buffer);
    LOG_PRINT(LOG_CRIT, "Out of memory");
    exit(1);
  }
  
  /* set up socket */
  unlink("server_socket");
  server_sockfd = socket(AF_INET, SOCK_STREAM, 0);

  server_address.sin_family = AF_INET;
  server_address.sin_addr.s_addr = htonl(INADDR_ANY);
  server_address.sin_port = htons((unsigned short int)atoi(port_address));
  server_len = sizeof(server_address);
  bind(server_sockfd, (struct sockaddr *)&server_address, server_len);

  listen(server_sockfd, 10);
  
  signal(SIGCHLD, SIG_IGN); /* don't wait til child completes */

  FD_ZERO(&readfds);
  FD_SET(server_sockfd, &readfds);

  /* initialize linked list for socket descriptors */
  first_olili.fd = server_sockfd;
  first_olili.ptr_next = NULL;

  while(run) {
    char **inargv; /* tokens of the client request */
    char *inbuffer;
    char *returnmsg;
    char current_db[_POSIX_PATH_MAX] = "";
    char set_owner[PREFS_BUF_LEN] = "";
    char format_string[MAX_FMT_LEN] = "";
    char sort_string[MAX_FMT_LEN] = "";
    char user_host[HOSTNAME_LENGTH] = "";
    size_t msg_size;
    int fd;
    int n_privatelist; /* if 1, search only personal interest list */
    int n_personal; /* add only personal information */
    int n_keep_id; /* keep ID of existing references in U5 */
    int numbyte, numbyte1; /* number of bytes read */
    int n_read_done;
    int n_malloc_failed;
    int n_cmdlinerror = 0;
    int n_inbuf_size;
    int n_remove;
    int result;
    int inargc; /* number of tokens of the client request */
    int inargcmax; /* maximum number of tokens */
    int n_success; /* counter for commands with multiple arguments */
    int ref_format = 0; /* format for output */
    int n_all = 0; /* getjournal: return all synonyms if 1 */
    struct ADDRESULT addresult; /* will hold result of addref */
    struct DELRESULT delresult; /* will hold result of deleteref */

    testfds = readfds;

    sprintf(msg_buffer, "server waiting n_max_fd=%d", max_olili(&first_olili));
    LOG_PRINT(LOG_INFO, msg_buffer);

    result = select(FD_SETSIZE, &testfds, (fd_set *)0, (fd_set *)0,
		    (struct timeval *)0);

    if (result < 1) {
      LOG_PRINT(LOG_CRIT, "select failed");
      free(msg_buffer);
      free(fifo_buffer);
      exit(1);
    }

    /* loop over all file descriptors */
    for(fd = 0; fd <= max_olili(&first_olili); fd++) {
#ifdef CYGWIN
      if (access(the_fifo, F_OK) != -1) {
	if ((fd_fifo = open(the_fifo, O_RDWR)) == -1) {
	  /* pseudo-fifo is still open, give child a chance to finish */
	  sleep(1);
	  if ((fd_fifo = open(the_fifo, O_RDWR)) != -1) {
	    if ((numbyte = read(fd_fifo, fifo_buffer, n_pipe_buf)) > 0) {
	      if (fifo_buffer[numbyte] != '\0') {
		sleep(1);
		if ((numbyte1 = read(fd_fifo, (void*)fifo_buffer+numbyte, n_pipe_buf)) > 0) {
		  if (fifo_buffer[numbyte+numbyte1] != '\0') {
		    LOG_PRINT(LOG_WARNING, "incomplete message - confserv aborted");
		  }
		  else {
		    confserv(fifo_buffer);
		  }
		}
		else {
		  LOG_PRINT(LOG_WARNING, "incomplete message - confserv aborted");
		}
	      }
	      else {
		confserv(fifo_buffer);
	      }
	      close(fd_fifo);
	      unlink(the_fifo);
	    }
	  }
	  /* else: forget about it this time */
	}
	else { /* open fifo ok */
	  if ((numbyte = read(fd_fifo, fifo_buffer, n_pipe_buf)) > 0) {
	    if (fifo_buffer[numbyte-1] != '\0') {
	      sleep(1);
	      if ((numbyte1 = read(fd_fifo, (void*)fifo_buffer+numbyte, n_pipe_buf)) > 0) {
		if (fifo_buffer[numbyte+numbyte1-1] != '\0') {
		  LOG_PRINT(LOG_WARNING, "incomplete message - confserv aborted");
		}
		else {
		  confserv(fifo_buffer);
		}
	      }
	      else {
		LOG_PRINT(LOG_WARNING, "incomplete message - confserv aborted");
	      }
	    }
	    else {
	      confserv(fifo_buffer);
	    }
	    close(fd_fifo);
	    unlink(the_fifo);
	  }
	}
      }
      /* else: nothing to do */

#else
      if ((numbyte = read(fd_fifo, (void*)fifo_buffer, n_pipe_buf)) > 0) {
	/* confserv() changed some value */
	if (fifo_buffer[numbyte-1] != '\0') { /* if message incomplete */
/*  	  printf(fifo_buffer); */
	  sleep(1); /* give child a chance to complete the message */
	  if ((numbyte1 = read(fd_fifo, (void*)fifo_buffer + numbyte, n_pipe_buf)) > 0) {
	    /* don't use a loop to try reading indefinitely as this would hang the server if the client crashed during writing to the fifo */
	    if (fifo_buffer[numbyte+numbyte1-1] != '\0') {
	      LOG_PRINT(LOG_WARNING, "incomplete message - confserv aborted");
	    }
	    else {
	      confserv(fifo_buffer);
	    }
	  }
	  else {
	    LOG_PRINT(LOG_WARNING, "incomplete message - confserv aborted");
	  }
	}
	else {
	  confserv(fifo_buffer);
	}
      }
      /* else: no message pending */
#endif /* CYGWIN */

      if (!run) {
	break;
      }
      /* if some activity was detected on one of the file descriptors */
      if (FD_ISSET(fd, &testfds)) {
	/* this must be a client knocking on the door */
	if (fd == server_sockfd) {

	  client_sockfd = accept(server_sockfd,
				 (struct sockaddr *)&client_address,
				 &client_len);
	  if (client_sockfd == -1) {  /* accept failed */
	    LOG_PRINT(LOG_WARNING, "accept failed");
	    continue; /* try another fd. Quitting here seems a bit harsh,
                         but might be more honest */
	  }
	  FD_SET(client_sockfd, &readfds);
	  sprintf(msg_buffer, "adding client on fd %d", client_sockfd);
	  LOG_PRINT(LOG_INFO, msg_buffer);
	  insert_olili(&first_olili, client_sockfd);
	}

	/* this is a real request from an existing client */
	else {
	  /* write log data to disk. otherwise the child still has
	     all previous messages in its buffer and repeats them */
	  if (fp_log_file != NULL) {
	    fflush(fp_log_file);
	  }

	  if (fork() == 0) { /* we must be child, handle request */
	    n_read_done = 0;
	    n_inbuf_size = COMMAND_INBUF_LEN;
	    n_malloc_failed = 0;

	    /* get us some buffer for log messages */
	    msg_buffer = (char*)malloc(MSG_BUF_SIZE);
	    if (msg_buffer == NULL) {
	      write(fd, outomem, strlen(outomem)+1);
	      LOG_PRINT(LOG_CRIT, "Out of memory");
	      exit(1);
	    }

	    inbuffer = (char*)malloc((size_t)COMMAND_INBUF_LEN*sizeof(char));
	    if (inbuffer == NULL) { /* malloc failed */
	      LOG_PRINT(LOG_CRIT, outomem);
	      write(fd, outomem, strlen(outomem)+1);
	      free(msg_buffer);
	      exit(1);
	    }
	      
	    msg_size = 256;
	    returnmsg = (char*)malloc(msg_size);
	    if (returnmsg == NULL) { /* malloc failed */
	      write(fd, outomem, strlen(outomem)+1);
	      LOG_PRINT(LOG_CRIT, outomem);
	      free(inbuffer);
	      free(msg_buffer);
	      exit(1);
	    }
	    returnmsg[0] = '\0';
	      
	    numbyte = tread(fd, inbuffer, COMMAND_INBUF_LEN-1);
	    if (inbuffer[numbyte-1] != '\0') { /* if transmission incomplete */
	      LOG_PRINT(LOG_CRIT, outomem);
	      iwrite(fd, outomem, strlen(outomem)+1);
	      free(inbuffer);
	      free(returnmsg);
	      free(msg_buffer);
	      exit(1);
	    }

	    sprintf(msg_buffer, "serving client on fd %d", fd);
	    LOG_PRINT(LOG_INFO, msg_buffer);
	    LOG_PRINT(LOG_DEBUG, inbuffer);

	    /* parse the client request. first we cut the request
	       into pieces with strtok(), then we use getopt() to interpret.
	       This code works only with GNU getopt(), not with BSD getopt() */

	    /* get a buffer to hold the tokens. Start with 10 tokens,
	       increase in steps of 10 as needed */
	    inargc = 0;
	    inargcmax = 10;
	    inargv = (char**)malloc((size_t)inargcmax*sizeof(char*));
	    if (inargv == NULL) {
	      LOG_PRINT(LOG_CRIT, outomem);
	      iwrite(fd, outomem, strlen(outomem)+1);
	      free(inbuffer);
	      free(returnmsg);
	      free(msg_buffer);
	      exit(1);
	    }

	    result = cmdln_tokenize(&inargc, &inargv, inargcmax, inbuffer);

	    if (result == 1) { /* memory error */
	      LOG_PRINT(LOG_CRIT, outomem);
	      iwrite(fd, outomem, strlen(outomem)+1);
	      free(inbuffer);
	      free(inargv);
	      free(returnmsg);
	      free(msg_buffer);
	      exit(1);
	    }
	    else if (result == 2) { /* empty command line */
	      iwrite(fd, "enter a command", 16);
	      free(inbuffer);
	      free(inargv);
	      free(returnmsg);
	      free(msg_buffer);
	      exit(1);
	    }

	    /* now we have the tokens nicely arranged in inargc */
	    /*  	  for (i = 0; i < inargc; i++) { */
	    /*  	    printf("inargv[%d]: %s\n", i, inargv[i]); */
	    /*  	  } */

	    /* reset optional arguments */
	    set_owner[0] = '\0';
	    n_privatelist = 0;
	    n_remove = 0;
	    n_personal = 1;
	    n_keep_id = 0;
	    n_all = 0;

	    /* get options */
	    optind = 0;

	    while ((n_opt = getopt(inargc, inargv, "ad:H:kpPrs:S:t:u:U:w:")) != -1) {
	      switch(n_opt) {
	      case 'a':
		n_all = 1;
		break;
	      case 'd':
		/*  	      printf("-d %s\n", optarg); */
		strcpy(current_db, optarg);
		break;
	      case 'H':
		/*  	      printf("-H %s\n", optarg); */
		strcpy(user_host, optarg);
		break;
	      case 'k':
		n_keep_id = 1;
		break;
	      case 'p':
		n_personal = 2;
		break;
	      case 'P':
		/*  	      printf("-P\n"); */
		n_privatelist = 1;
		break;
	      case 'r':
		/*  	      printf("-r\n"); */
		n_remove = 1;
		break;
	      case 's':
		/*  	      printf("-s %s\n", optarg); */
		strncpy(format_string, optarg, MAX_FMT_LEN-1);
		format_string[MAX_FMT_LEN-1]='\0'; /* if string was truncated */
		break;
	      case 'S':
		/*  	      printf("-S %s\n", optarg); */
		strncpy(sort_string, optarg, MAX_FMT_LEN-1);
		sort_string[MAX_FMT_LEN-1]='\0'; /* if string was truncated */
		break;
	      case 't':
		/*  	      printf("-t %s\n", optarg); */
		if (strcmp(optarg, "html") == 0) {
		  ref_format = 4;
		}
		else if (strcmp(optarg, "ris") == 0) {
		  ref_format = 1;
		}
		else if (strcmp(optarg, "db31") == 0) {
		  ref_format = 2;
		}
		else if (strcmp(optarg, "bibtex") == 0) {
		  ref_format = 3;
		}
		else { /* default is screen */
		  ref_format = 0;
		}
		break;
	      case 'u':
		/*  	      printf("-u %s\n", optarg); */
		strcpy(username, optarg);
		break;
	      case 'U':
		/*  	      printf("-U %s\n", optarg); */
		strcpy(set_owner, optarg);
		break;
	      case 'w':
		/*  	      printf("-w %s\n", optarg); */
		strcpy(passwd, optarg);
		break;
	      case ':':
		/*  	      printf("missing option\n"); */
		iwrite(fd, "missing option", 15);
		free(inbuffer);
		free(inargv);
		free(returnmsg);
		free(msg_buffer);
		n_cmdlinerror = 1;
		break;
	      case '?':
		/*  	      printf("unknown option\n"); */
		iwrite(fd, "unknown option", 15);
		free(inbuffer);
		free(inargv);
		free(returnmsg);
		free(msg_buffer);
		n_cmdlinerror = 1;
		break;
	      }
	    }
	    
	    /* get arguments */
	    /*  	  for (i = optind; i < inargc; i++) { */
	    /*  	    printf("argument %s\n", inargv[i]); */
	    /*  	  } */

	    if (n_cmdlinerror) {
	      LOG_PRINT(LOG_DEBUG, "command line error");
	      exit(1);
	    }

	    /* test for commands (still in inargv[0]) */

	    /*********************************************** viewstat ****/
	    if (strcmp(inargv[0], "viewstat") == 0) {
	      viewstat(fd, username, passwd, server_ip);
	    }

	    /*********************************************** listdb ****/
	    else if (strcmp(inargv[0], "listdb") == 0) {
	      listdb(fd, (optind < inargc) ? inargv[optind] : NULL, username, passwd, server_ip, 0);
	    }

	    /*********************************************** createdb ****/
	    else if (strcmp(inargv[0], "createdb") == 0) {

	      /* test whether we have a filename argument */
	      if (optind == inargc) {
		iwrite(fd, argerr, strlen(argerr));
		iwrite(fd, newln, strlen(newln));
	      }

	      n_success = 0;

	      /* loop over all filename arguments */
	      for (; optind < inargc; optind++) {
		n_success += createdb(fd, inargv[optind], username, passwd, server_ip);
	      }

	      /* summary for client */
	      sprintf(returnmsg, "%d databases created", n_success);
	    }

	    /*********************************************** deletedb ****/
	    else if (strcmp(inargv[0], "deletedb") == 0) {

	      /* test whether we have a filename argument */
	      if (optind == inargc) {
		iwrite(fd, argerr, strlen(argerr));
		iwrite(fd, newln, strlen(newln));
	      }

	      n_success = 0;

	      /* loop over all filename arguments */
	      for (; optind < inargc; optind++) {
		n_success += deletedb(fd, inargv[optind], username, passwd, server_ip);
	      }

	      /* summary for client */
	      sprintf(returnmsg, "%d databases deleted", n_success);
	    }

	    /*********************************************** confserv ****/
	    else if (strcmp(inargv[0], "confserv") == 0) {
	      child_confserv(fd, optind, inargc, inargv);
	    }

	    /*********************************************** selectdb ****/
	    else if (strcmp(inargv[0], "selectdb") == 0) {
	      listdb(fd, (optind < inargc) ? inargv[optind] : NULL, username, passwd, server_ip, 1);
	    }

	    /*********************************************** addref ****/
	    else if (strcmp(inargv[0], "addref") == 0) {
	      result = addref(fd, (optind < inargc) ? inargv[optind] : NULL, username, set_owner, passwd, current_db, &addresult, 0, n_keep_id);

	      /* let the client know what happened */
	      sprintf(returnmsg, "%d dataset(s) added, %d failed\n", addresult.success, addresult.failure);
	    }

	    /*********************************************** updateref ****/
	    else if (strcmp(inargv[0], "updateref") == 0) {
	      result = addref(fd, (optind < inargc) ? inargv[optind] : NULL, username, set_owner, passwd, current_db, &addresult, n_personal, n_keep_id);

	      /* let the client know what happened */
	      sprintf(returnmsg, "%d dataset(s) updated, %d added, %d failed\n", addresult.updated, addresult.success, addresult.failure);
	    }

	    /*********************************************** deleteref ****/
	    else if (strcmp(inargv[0], "deleteref") == 0) {

	      /* test whether we have a size argument */
	      if (optind == inargc) {
		iwrite(fd, argerr, strlen(argerr));
		iwrite(fd, newln, strlen(newln));
	      }

	      n_success = deleteref(fd, inargv[optind], username, passwd, current_db, &delresult);

	      /* let the client know what happened */
	      if (n_success) {
		sprintf(returnmsg, "%d datasets deleted", delresult.success);
	      }
	      else {
		sprintf(returnmsg, "%d datasets successfully deleted, stopped after error", delresult.success);
	      }
	    }

	    /*********************************************** pickref ****/
	    else if (strcmp(inargv[0], "pickref") == 0) {

	      /* test whether we have a size argument */
	      if (optind == inargc) {
		iwrite(fd, argerr, strlen(argerr));
		iwrite(fd, newln, strlen(newln));
	      }

	      n_success = pickref(fd, inargv[optind], username, passwd, current_db, n_remove, &delresult);

	      /* let the client know what happened */
	      if (n_remove) {
		if (n_success) {
		  sprintf(returnmsg, "%d datasets removed from personal interest list, %d ignored", delresult.success, delresult.failure);
		}
		else {
		  sprintf(returnmsg, "%d datasets removed from personal interest list, stopped after error", delresult.success);
		}
	      }
	      else {
		if (n_success) {
		  sprintf(returnmsg, "%d datasets added to personal interest list, %d ignored", delresult.success, delresult.failure);
		}
		else {
		  sprintf(returnmsg, "%d datasets added to personal interest list, stopped after error", delresult.success);
		}
	      }
	    }

	    /*********************************************** getref ****/
	    else if (strcmp(inargv[0], "getref") == 0) {
	      getref(fd, (optind < inargc) ? inargv[optind] : NULL, username, passwd, current_db, format_string, sort_string, ref_format, n_privatelist);
	    }

	    /*********************************************** getau *****/
	    else if (strcmp(inargv[0], "getau") == 0) {
	      getfoo(fd, (optind < inargc) ? inargv[optind] : NULL, username, passwd, current_db, server_ip, 0, 0);
	    }

	    /*********************************************** geted *****/
	    else if (strcmp(inargv[0], "geted") == 0) {
	      getfoo(fd, (optind < inargc) ? inargv[optind] : NULL, username, passwd, current_db, server_ip, 6, 0);
	    }

	    /*********************************************** getas *****/
	    else if (strcmp(inargv[0], "getas") == 0) {
	      getfoo(fd, (optind < inargc) ? inargv[optind] : NULL, username, passwd, current_db, server_ip, 7, 0);
	    }

	    /*********************************************** getkw *****/
	    else if (strcmp(inargv[0], "getkw") == 0) {
	      getfoo(fd, (optind < inargc) ? inargv[optind] : NULL, username, passwd, current_db, server_ip, 1, 0);
	    }

	    /*********************************************** getjo *****/
	    else if (strcmp(inargv[0], "getjo") == 0) {
	      getfoo(fd, (optind < inargc) ? inargv[optind] : NULL, username, passwd, current_db, server_ip, 2, n_all);
	    }

	    /*********************************************** getjf *****/
	    else if (strcmp(inargv[0], "getjf") == 0) {
	      getfoo(fd, (optind < inargc) ? inargv[optind] : NULL, username, passwd, current_db, server_ip, 3, n_all);
	    }

	    /*********************************************** getj1 *****/
	    else if (strcmp(inargv[0], "getj1") == 0) {
	      getfoo(fd, (optind < inargc) ? inargv[optind] : NULL, username, passwd, current_db, server_ip, 4, n_all);
	    }

	    /*********************************************** getj2 *****/
	    else if (strcmp(inargv[0], "getj2") == 0) {
	      getfoo(fd, (optind < inargc) ? inargv[optind] : NULL, username, passwd, current_db, server_ip, 5, n_all);
	    }

	    /*********************************************** adduser ****/
	    else if (strcmp(inargv[0], "adduser") == 0) {
	      adduser(fd, optind, inargc, inargv, username, passwd, current_db, user_host, server_ip, n_remove);
	    }

	    /*********************************************** unknown ****/
	    else {
	      iwrite(fd, unknown_command, strlen(unknown_command));
	    }

	    /* finish dialog with client */
/*  	    iwrite(fd, returnmsg, strlen(returnmsg)+1); */ /* include \0 */

	    free(inargv);
	    free(inbuffer);
	    free(returnmsg);
	    free(msg_buffer);	    
	    close(fd);
	    FD_CLR(fd, &readfds);
	    sprintf(msg_buffer, "child finished client on fd %d", fd);
	    LOG_PRINT(LOG_INFO, msg_buffer);
	    /* write log data to disk */
	    if (fp_log_file != NULL) {
	      fflush(fp_log_file);
	    }
	    if (fd_fifo != -1) {
	      close(fd_fifo);
	    }

	    /* decrease n_max_fd to minimize loops */
/*  	    delete_olili(&first_olili, client_sockfd); */
	    exit (0);
	  }
	  else { /* we're parent, close client connection */
	    close(fd);
	    FD_CLR(fd, &readfds);
	    sprintf(msg_buffer, "parent removing client on fd %d", fd);
	    LOG_PRINT(LOG_INFO, msg_buffer);
	    /* write log data to disk */
/*  	    if (fp_log_file != NULL) { */
/*  	      fflush(fp_log_file); */
/*  	    } */
	    /* decrease n_max_fd to minimize loops */
	    delete_olili(&first_olili, client_sockfd);
	  }
	} /* end: if (fd == server_sockfd) */
      } /* end: if (FD_ISSET(fd, &testfds)) */
    } /* end: for() */
  } /* end: while(run) */
  free(msg_buffer);
  free(fifo_buffer);
  if (fp_log_file != NULL) {
    fclose(fp_log_file); /* not strictly necessary here, but it's nicer */
  }
  if (fd_fifo != -1) {
    close(fd_fifo);
  }
  exit (0);
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  addref(): implements the client command addref

  int addref returns 0 if failed, 1 if successful

  int fd file descriptor of socket

  char *argument command argument (or NULL if none)

  char *username the mysql username of the user typing on the client

  char *set_owner the mysql username of the owner of new/updated
                      references if different from *username

  char *passwd the mysql password

  char *current_db the currently selected database

  int n_requested_bufsize the requested number of bytes for first dataset

  struct ADDRESULT* addresult this structure will be filled in with the number
                      of (un-)successfully added/updated references

  int replace_ref if 1, ref will be updated according to ID field; if 0
         ref will be added without looking at an ID field; if 2, only the
         personal settings (N1, AV, RP) will be updated according to the
         ID field

  int n_keep_id if 1, any existing reference ID will be saved in U5

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int addref(int fd, char *argument, char *username, char* set_owner, char *passwd, char *current_db, struct ADDRESULT* ptr_addresult, int replace_ref, int n_keep_id) {
  return 1;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  deleteref(): implements the client command deleteref

  int deleteref returns 0 if failed, 1 if successful

  int fd file descriptor of socket

  int optind current index for *inargv array

  int inargc number of arguments in *inargv array

  char** inargv array of arguments

  char *username the mysql username

  char *passwd the mysql password

  char *current_db the currently selected database

  struct DELRESULT* ptr_delresult structure to hold number of successful and
                              failed deleterefs

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/* migration to mstr* not necessary */
int deleteref(int fd, char *bufsize, char *username, char *passwd, char *current_db, struct DELRESULT* ptr_delresult) {
  return 1;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  pickref(): implements the client command pickref

  int pickref returns 0 if failed, 1 if successful

  int fd file descriptor of socket

  char *bufsize pointer to a string holding the buffer size required
                to read the client request

  char *username the mysql username

  char *passwd the mysql password

  char *current_db the currently selected database
  
  int n_remove if 1, remove from personal interest list. if 0, add to list

  struct DELRESULT* ptr_delresult structure to hold number of successful and
                              failed pickrefs

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/* Done migrating to mstr* */
int pickref(int fd, char *bufsize, char *username, char *passwd, char *current_db, int n_remove, struct DELRESULT* ptr_delresult) {
  return 1;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  getref(): implements the client command getref

  void getref

  int fd file descriptor of socket

  char *argument command argument (or NULL if none)
                 the argument is a search expression

  char *username the database username

  char *passwd the database password

  char *current_db the currently selected database

  char *format_string ptr to buffer holding the output format information

  char *sort_string ptr to buffer holding the sorting information

  int ref_format the output format 0 = simple screen rendering,
        1 = RIS format, 2 = DocBook format, 3 = BibTeX format,
        4 = HTML

  int n_privatelist if set to 1, limit search to user's private list
                    if set to 0, search all

  This function has no return value

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
/* Done migrating to mstr* */
void getref(int fd, char *argument, char *username, char *passwd, char *current_db, char *format_string, char *sort_string, int ref_format, int n_privatelist) {
  return;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  log_print(): writes a log message

  void log_print

  int priority the priority level of the log message as in syslog.h

  char* string a string containing the message to be logged

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void log_print(int priority, char* string) {
  /* we must have this fn in refdbs.c because FILE* fp_log_file
     cannot be declared extern (??) */
  time_t the_time;
  char timestring[256];
  FILE* new_fp_log_file;

  if (n_log_dest == 0) { /* output on stderr */
    fprintf(stderr, "%s\n", string);
  }
  else if (n_log_dest == 1) { /* output via syslog */
    syslog(priority, "%s pid=%d", string, getpid());
  }
  else { /* output in user-defined logfile */
    if (n_reopen_log) { /* refdb-admin changed the log filename */
      n_reopen_log = 0; /* reset switch */
      if ((new_fp_log_file = fopen(log_file, "ab")) == NULL) {
	n_log_dest = 1; /* fall back to syslog */
	fclose(fp_log_file);
	LOG_PRINT(LOG_WARNING, "could not open custom log file");
      }
      else {
	fclose(fp_log_file);
	fp_log_file = new_fp_log_file;
	LOG_PRINT(LOG_INFO, "switched log file");
      }
    }
    time(&the_time);
    strftime(timestring, 256, "%a %b %d %H:%M:%S %Y", gmtime(&the_time));
    fprintf(fp_log_file, "pid=%d:%s:%s\n", getpid(), timestring, string);
  }
}








