/*++++
  strfncs.c provides some string and array helpers
  hoenicka_markus@compuserve.com 2-8-00
  $Id: strfncs.c,v 1.6 2000/07/29 04:02:30 markus Exp $
  ++++++*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "strfncs.h"


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  is_ip(): checks whether string contains a valid IP address
           This function does not verify that the given IP address
	   exists or is accessible, but rather checks whether the
	   string is sane.

  int is_ip

  char *address ptr to the string to check

  returns 0 if invalid, 1 if valid

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int is_ip(char *address) {
  char *token[4];
  char ip_address[16];
  int i, n_value;

  /* refuse if string is too long */
  if (strlen(address) > 15) {
    return 0;
  }

  /* make a local copy as strtok modifies its argument */
  strncpy(ip_address, address, 16);
  ip_address[15] = '\0';

  /* now look for four tokens separated by "." */
  token[0] = strtok(ip_address, ".");
  if (token[0] == NULL) {
    return 0;
  }

  token[1] = strtok(NULL, ".");
  if (token[1] == NULL) {
    return 0;
  }

  token[2] = strtok(NULL, ".");
  if (token[2] == NULL) {
    return 0;
  }

  token[3] = strtok(NULL, "\r\n");
  if (token[3] == NULL) {
    return 0;
  }

  /* see whether the tokens are in the allowed numerical range */
  for (i = 0; i < 4; i++) {
    n_value = atoi(token[i]);
    if (n_value < 0 || n_value > 255) {
      return 0;
    }
  }
  return 1;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  is_port(): checks whether string contains a valid port
             This fn simply ensures that the port is outside the
             range that the system uses, i.e. > 1024

  int is_port

  char *address ptr to the string to check

  returns 0 if invalid, 1 if valid

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int is_port(char *port) {
  if (atoi(port) < 1024) {
    return 0;
  }
  else {
    return 1;
  }
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  stripwhite(): strips whitespace from the start and end of STRING

  char* stripwhite

  char* string address of string to convert

  int mode 0 = strips start and end, 1 = start only, 2 = end only

  int type 0 = only space and tab 1 = space, tab, cr, lf

  Returns ptr to the modified string

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
char *stripwhite (char *string, int mode, int type) {
  /* this function was modified from a readline sample program. The
     style is cryptic indeed, but it works. */
  register char *s, *t;

  s = string;

  if (type) {
    if (mode != 2) {
      for (; full_whitespace (*s); s++)
	;
      
      if (*s == 0)
	return (s);
    }

    if (mode != 1) {
      t = s + strlen (s) - 1;
      while (t > s && full_whitespace (*t))
	t--;
      *++t = '\0';
    }
  }
  else {
    if (mode != 2) {
      for (; whitespace (*s); s++)
	;
      
      if (*s == 0)
	return (s);
    }

    if (mode != 1) {
      t = s + strlen (s) - 1;
      while (t > s && whitespace (*t))
	t--;
      *++t = '\0';
    }
  }
  return s;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  strup(): converts a string in place to uppercase
  
  char* strup() returns a pointer to the modified string

  char* string pointer to the string to be converted

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
char* strup(char* string) {
  char* chr;

  chr = string; /* don't modify string, we need it as a return value */

  /* loop until we find \0 */
  while (*chr) {
    *chr = (char)toupper((int)*chr); /* now convert */
    chr++;
  }
  return string;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  compare_ptr(): compares two pointers to strings

  int compare_ptr returns -1 if the first argument is smaller than the
                  second; 0 if both are equal; 1 if the first is larger

  void* ptr_one the first char* to compare

  void* ptr_two the second char* to compare

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int compare_ptr(void* ptr_one, void* ptr_two) {
  char** firstptr;
  char** secondptr;

  firstptr = (char**)ptr_one;
  secondptr = (char**)ptr_two;

  /* handle cases where at least one of the ptrs is a NULL ptr */
  if (*firstptr == NULL) {
    if (*secondptr == NULL) {
      return 0;
    }
    else {
      return 1;
    }
  }
  else if (*secondptr == NULL) {
    return -1;
  }

  /* all ptrs are non-NULL now */
  if (*firstptr < *secondptr) {
    return -1;
  }
  else if (*firstptr == *secondptr) {
    return 0;
  }
  else {
    return 1;
  }
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  mstrcat(): a modified version of strcat which operates on destination
             strings obtained with malloc(); the length of the buffer
             is dynamically increased if necessary.

  char* mstrcat returns a pointer to the destination string or NULL
        if the realloc failed. As the buffer holding the destination
        string may have been reallocated, it is mandatory to use
        *ONLY* this returned pointer after the function call and
        *NEVER* the old pointer to destination

  char* destination the buffer obtained with malloc() to which the 
        source string will be appended. The calling function is
        responsible to free this buffer after use

  char* source the string that will be appended to destination

  size_t* dest_len points to a variable that contains the current size
       of the buffer that destination points to. Will be modified if
       a realloc() is necessary to increase the buffer size

  size_t offset a positive value n will concatenate the string at the 
      position omega-n of the destination string. Set offset to zero
      or less than zero to get the standard strcat behaviour.

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
char* mstrcat(char* destination, char* source, size_t* dest_len, size_t offset) {
  char *new_dest;
  size_t len;
  size_t source_len;
  size_t destination_len;

  /* take the short way out if destination doesn't exist */
  if (destination == NULL || source == NULL) {
    return NULL;
  }

  source_len = strlen(source);
  destination_len = strlen(destination);

  /* do not accept negative values which wouldn't make much sense */
  offset = (offset < 0) ? 0 : offset;

  /* don't allow offsets longer than the destination length */
  offset = (offset > destination_len) ? destination_len : offset;

  /* we need the sum of both string lengths plus one for the \0 minus
     the offset if there is one */
  len = destination_len + source_len + 1 - offset;

  /* reallocate buffer if it is too small */
  if (len > *dest_len) {
    len = (len - *dest_len < realloc_chunk) ? *dest_len + realloc_chunk : len;
    if ((new_dest = (char*)realloc(destination, len)) == NULL) {
      return NULL;
    }
    else {
      destination = new_dest;
    }
    /* adjust the length variable */
    *dest_len = len;
  }
  
  if (!offset) { /* this is the standard strcat behaviour */
    strcat(destination, source);
  }
  else { /* this will append the string at position omega minus offset */
    strcpy(&destination[strlen(destination)-offset], source);
  }

  /* return the new pointer to the buffer */
  return destination;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  mstrcpy(): a modified version of strcpy which operates on destination
             strings obtained with malloc(); the length of the buffer
             is dynamically increased if necessary.

  char* mstrcpy returns a pointer to the destination string or NULL
        if the realloc failed. As the buffer holding the destination
        string may have been reallocated, it is mandatory to use
        *ONLY* this returned pointer after the function call and
        *NEVER* the old pointer to destination

  char* destination the buffer obtained with malloc() to which the 
        source string will be copied

  char* source the string that will be copied to destination

  size_t* dest_len points to a variable that contains the current size
       of the buffer that destination points to. Will be modified if
       a realloc() is necessary to increase the buffer size

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
char* mstrcpy(char* destination, char* source, size_t* dest_len) {
  char *new_dest;
  size_t len;

  /* take the short way out if destination doesn't exist */
  if (destination == NULL) {
    return NULL;
  }

  /* we need the length of the string plus one for the \0 */
  len = strlen(source)+1;

  /* reallocate the buffer if it is too small */
  if (*dest_len < len) {
    len = (len - *dest_len < realloc_chunk) ? *dest_len + realloc_chunk : len;
    if ((new_dest = (char*)realloc(destination, len)) == NULL) {
      return NULL;
    }
    else {
      destination = new_dest;
    }
    /* adjust the length variable */
    *dest_len = len;
  }

  /* now copy the string*/
  strcpy(destination, source);

  /* return the new pointer to the buffer */
  return destination;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  mstrncpy(): a modified version of strncpy which operates on destination
             strings obtained with malloc(); the length of the buffer
             is dynamically increased if necessary.

  char* mstrncpy returns a pointer to the destination string or NULL
        if the realloc failed. As the buffer holding the destination
        string may have been reallocated, it is mandatory to use
        *ONLY* this returned pointer after the function call and
        *NEVER* the old pointer to destination

  char* destination the buffer obtained with malloc() to which the 
        source string will be copied

  char* source the string that will be copied to destination

  size_t n the number of characters to copy

  size_t* dest_len points to a variable that contains the current size
       of the buffer that destination points to. Will be modified if
       a realloc() is necessary to increase the buffer size

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
char* mstrncpy(char* destination, char* source, size_t n, size_t* dest_len) {
  char *new_dest;
  size_t len;

  /* take the short way out if destination doesn't exist */
  if (destination == NULL || source == NULL) {
    return NULL;
  }

  /* we need the length of the string plus one for the \0 */
  len = n+1;

  /* reallocate the buffer if it is too small */
  if (*dest_len < len) {
    len = (len - *dest_len < realloc_chunk) ? *dest_len + realloc_chunk : len;
    if ((new_dest = (char*)realloc(destination, len)) == NULL) {
      return NULL;
    }
    else {
      destination = new_dest;
    }
    /* adjust the length variable */
    *dest_len = len;
  }

  /* now copy the string*/
  strncpy(destination, source, n);

  /* return the new pointer to the buffer */
  return destination;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  mstrdup() creates a copy of a string in a malloc()'ed buffer

  char* mstrdup returns a pointer to the copy of the string. Returns
                NULL in the case of a failure (out of memory)

  char* buffer  pointer to the string to be copied

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
char* mstrdup(char *buffer) {
  char *new_buffer;

  if ((new_buffer = malloc(strlen(buffer)+1)) == NULL) {
    return NULL;
  }
  else {
    strcpy(new_buffer, buffer);
    return new_buffer;
  }
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  free_all() frees memory, taking the adresses out of a
             NULL-terminated array

  void free_all

  void*** ptr_mem pointer to an array of pointers which point to
             pointer variables which hold valid addresses of memory 
             allocated with malloc/calloc/realloc or NULL. This
             somewhat complicated arrangement allows to reallocate
             the memory without changing the values in the array.
             If memory is freed before free_all() is called, the
             value of the pointer to the freed memory must be set
             to NULL, which will not harm free().

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void free_all(void*** ptr_mem) {
  int i = 0;

  while (ptr_mem[i] != NULL) {
/*      printf("free %d: %d\n", i, *(ptr_mem[i])); */
    free(*(ptr_mem[i++]));
  }
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  count_the_flowers() counts the number of occurrences of letter in
                      buffer

  size_t count_the_flowers returns the number of occurrences of letter
                      in buffer. It will return zero if letter is not
                      in buffer.

  char* buffer pointer to the string to scan

  char letter the letter to locate

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
size_t count_the_flowers(char *buffer, char letter) {
  size_t numletters = 0;
  int i;

  for (i = 0; i < strlen(buffer); i++) {
    if (buffer[i] == letter) {
      numletters++;
    }
  }
  return numletters;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  insert_string() inserts the string insert in front of buffer. The
                  calling function is responsible to allocate enough
                  space for buffer so the insertion can be safely
                  executed.

  void insert_string has no return value

  char* buffer pointer to the string that will be extended

  char* insert pointer to the string that will be inserted

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void insert_string(char *buffer, char *insert) {
  int len_insert;

  len_insert = strlen(insert);

  memmove(buffer+len_insert, buffer+1, strlen(buffer));
  memcpy(buffer, insert, len_insert);
}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  sgml_entitize() replaces special characters with entities. The
                  replacement is done according to an array of
                  char/string pairs that has to be provided by the
                  calling function.

  char* insert_string returns a pointer to the converted string. This
                  may be different from the original value of the
		  passed pointer, so after calling this function only
		  the returned value should be used. The function also
		  modifies the value of the passed pointer to buffer.
		  If the function is successful, the return value and
		  the current value of the passed pointer will be
                  identical. The return value is NULL if an error
                  occurred. In this case, the current value of the
		  passed pointer can still be used to access the 
		  converted string as it was before the error.

  char** buffer pointer to a pointer to the string that will be
                  entitized

  size_t* buffer_len pointer to the length of the allocated buffer.
                  This value will be modified if the buffer needs to
                  be reallocated.

  struct charent* myents pointer to an array of character/string pairs
                  which will be used for the replacements. The last
                  element of the array must have a '0' (zero) as the
		  character value as this is taken as a termination
		  signal.

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
char* sgml_entitize(char** buffer, size_t* buffer_len, struct charent* myents) {
  int numents = 0;
  int i = 0;
  char *new_buffer;
  char *token, *the_end;

  while (myents[i].character != 0) {
  
    numents = count_the_flowers(*buffer, myents[i].character);
/*      printf("%d\n", numents); */
    
    if (numents > 0) {
      if ((new_buffer = realloc(*buffer, strlen(*buffer) + ((strlen(myents[i].entity))*(numents+1)))) == NULL) {
	return NULL;
      }
      else {
	*buffer = new_buffer;
      }
      
      token = strchr(*buffer, myents[i].character);
      the_end = &((*buffer)[strlen(*buffer)-1]);

      while (token != NULL) {
	insert_string(token, myents[i].entity);
	token = (token + 1 > the_end + 5) ? NULL:strchr(token+1, myents[i].character);
      }
    }
    i++;
  }
  return *buffer;
}














