/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Ant" and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.tools.ant.taskdefs.optional.junit;

import junit.framework.Test;
import junit.framework.TestSuite;
import junit.framework.TestCase;

import java.io.File;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;

/**
 * A JUnit test with a configurable "suite".
 * The contents of the suite are determined by reading classnames out of a file.
 * The location of this file is found by looking at the system property
 * "org.apache.tools.ant.taskdefs.optional.junit.BATCHFILE".
 *
 * @author <a href="http://madbean.com/">Matt Quail</a>
 *
 * @version $Revision $
 *
 * @since Ant 1.6FIXME
 *
 * @see JUnitTask
 */
public class BatchSuite {

    /**
     * This method is called by the JUnit reflection mechanism to
     * return the tests to run
     * @return the test found by looking in the "batch" file
     */
    public static Test suite() {
        TestSuite suite = new TestSuite();

        String batchFilename = System.getProperty("org.apache.tools.ant.taskdefs.optional.junit.BATCHFILE");
        if (batchFilename == null) {
            suite.addTest(warning("could not find BATCHFILE property"));
        } else {
            File batchFile = new File(batchFilename);
            addBatchToSuite(batchFile, suite);
        }
        return suite;
    }

    /**
     * Parses a file for test class names
     * @param batchFile the file to parse
     * @param suite tests will be added to this suite
     */
    private static void addBatchToSuite(File batchFile, TestSuite suite) {
        try {
            BufferedReader in = new BufferedReader(new FileReader(batchFile));
            try {
                String testname;
                while ((testname = in.readLine()) != null) {
                    try {
                        Test test = JUnitTestRunner.extractTestFromClassname(testname, null);
                        suite.addTest(test);
                    } catch (Exception e) {
                        suite.addTest(warning("could not load class " + testname + " : " + e.getMessage()));
                    }
                }
            } finally {
                in.close();
            }
        } catch (IOException e) {
            suite.addTest(warning("error reading file " + batchFile.getPath() + " : " + e.getMessage()));
        }
    }

    /**
     * @return a test which will fail and log a warning message.
     */
    private static Test warning(final String message) {
        return new TestCase("warning") {
            protected void runTest() {
                fail(message);
            }
        };
    }
}
