/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Ant" and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.tools.ant.taskdefs;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Target;
import org.apache.tools.ant.Task;

import java.io.PrintWriter;
import java.io.StringWriter;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.SortedSet;
import java.util.TreeSet;


/**
 * Displays information about the current project, listing the project description,
 * defined targets, and the default target.
 * <p>The primary purpose of this class is to
 * provide information about a build file when Ant is invoked with the
 * <code>-projecthelp</code> argument -- in that case,
 * {@link org.apache.tools.ant.Main} adds, configures,
 * and executes the task automagically. However, you may also choose to explicitly
 * invoke this task from a target:</p>
 * <pre>
 * &lt;target name="usage"&gt;
 *   &lt;projecthelp/&gt;
 * &lt;/target&gt;
 * </pre>
 * <p>When invoked this way, the output of <code>ant usage</code> will contain exactly
 * the same information as that of <code>ant -projecthelp</code>. You may, however,
 * display only specific information about the build file, such as only listing the
 * project description:</p>
 * <pre>
 * &lt;target name="usage"&gt;
 *   &lt;projecthelp includeMainTargets="false" includeDefaultTarget="false"/&gt;
 * &lt;/target&gt;
 * </pre>
 * <p>For additional configuration attributes, refer to the documentation of the setter
 * methods (included herein), or to this task's Ant manual page. This task does not
 * support any nested elements.</p>
 *
 * @author <a href="mailto:f.g.haas@gmx.net">Florian G. Haas</a>
 * @see org.apache.tools.ant.Main
 */
public class ProjectHelp extends Task {
    /** The default maximum line width of the output. */
    private static final int DEFAULT_LINE_WIDTH = Integer.MAX_VALUE;

    /** The line separator character used on the system. */
    private static String lineSep = System.getProperty("line.separator");

    /** Will the output include a line stating which target is the default? */
    private boolean includeDefaultTarget = true;

    /** Will the output include the project description, if available? */
    private boolean includeDescription = true;

    /** Will the output include information about "internal" targets (the ones that
     * whose name starts with <code>-</code> (hyphen)? */
    private boolean includeInternalTargets = false;

    /** Will the output include information about "main" targets (the ones that
     * come with a <code>description</code> attribute)? */
    private boolean includeMainTargets = true;

    /** Will the output include information about "sub" targets (the ones that
     * don't have a <code>description</code> attribute)? */
    private boolean includeSubTargets = false;

    /** A set representing the "internal" targets configured for the project (sorted
     * by target name). */
    private SortedSet internalTargets = new TreeSet(new TargetComparator());

    /** The total maximum width of the output. */
    private int lineWidth = DEFAULT_LINE_WIDTH;

    /** A set representing the "main" targets configured for the project (sorted
     * by target name). */
    private SortedSet mainTargets = new TreeSet(new TargetComparator());

    /** The longest name of any target configured in the project. */
    private int maxTargetNameLength;

    /** A set representing the "sub" targets configured for the project (sorted
     * by target name). */
    private SortedSet subTargets = new TreeSet(new TargetComparator());

    /**
     * Executes the task. Depending on configuration, prints the project description,
     * "main" targets, "sub" targets, and the default target.
     *
     * @throws BuildException if something goes wrong while doing the above tasks.
     *                        This should never happen.
     */
    public void execute() throws BuildException {
        try {
            if (includeDescription) {
                printDescription();
            }

            if (includeMainTargets) {
                printMainTargets();
            }

            if (includeSubTargets) {
                printSubTargets();
            }

            if (includeInternalTargets) {
                printInternalTargets();
            }

            if (includeDefaultTarget) {
                printDefaultTarget();
            }
        } catch (Exception e) {
            throw new BuildException(e);
        }
    }

    /**
     * Initializes the task. Scans through the targets configured in the project,
     * sorts them by name, and determines the maximum target name length.
     */
    public void init() {
        Enumeration targets = getProject().getTargets().elements();
        Target currentTarget;
        String targetName;

        while (targets.hasMoreElements()) {
            currentTarget = (Target) targets.nextElement();
            targetName = currentTarget.getName();

            if (targetName.equals("")) {
                continue;
            } else if (targetName.startsWith("-")) {
                // target name starts with "-", is an "internal" target
                internalTargets.add(currentTarget);
            } else if (currentTarget.getDescription() == null) {
                // target doesn't have a description, is a "sub" target
                subTargets.add(currentTarget);
            } else {
                // target has a description, is a "main" target
                mainTargets.add(currentTarget);

                if (targetName.length() > maxTargetNameLength) {
                    // update maximum target name length (used later for padding)
                    maxTargetNameLength = targetName.length();
                }
            }
        }
    }

    /**
     * Prints the default target. The default target is the target referenced in the
     * <code>default</code> attribute defined on the build file's
     * <code>&lt;project&gt;</code> element.
     */
    private void printDefaultTarget() {
        String defaultTarget = getProject().getDefaultTarget();

        if ((defaultTarget != null) && !"".equals(defaultTarget)) {
            // shouldn't need to check but...
            log("Default target: " + defaultTarget + lineSep);
        }
    }

    /**
     * Prints the project description. The project description is the content of the
     * <code>&lt;description&gt;</code> element, a child of <code>&lt;project&gt;</code>.
     * If no project description is specified for the current project, this method
     * does nothing.
     */
    private void printDescription() {
        String description = getProject().getDescription();

        if (description != null) {
            log(description);
            log(lineSep);
        }
    }

    /**
     * Prints the "internal" targets in the current project.
     * "Internal" targets are those whose name starts with "-", and hence can't be invoked
     * from outside Ant. */
    private void printInternalTargets() {
        printTargets(internalTargets, "Internal targets:");
    }

    /**
     * Prints the "main" targets in the current project. "Main" targets are those which
     * carry a <code>description</code> attribute on their <code>&lt;target&gt;</code>
     * element. */
    private void printMainTargets() {
        printTargets(mainTargets, "Main targets:");
    }

    /**
     * Logs information about a single target. The information contains the target name
     * and, if available, the target description.
     *
     * @param target the target to be described
     * @param totalCols the total number of columns available for output. The output text
     *        is formatted so as not to exceed that maximum text width.
     */
    private void printSingleTarget(Target target) {
        String name = target.getName();
        String desc = target.getDescription();

        StringWriter stringWriter = new StringWriter();
        PrintWriter printWriter = new PrintWriter(stringWriter);

        // margin to left of target name (spaces)
        int leftMargin = 2;

        // margin to left of target description (spaces)
        int descLeftMargin = 2;

        // determine text width available for description
        int descWidth = this.lineWidth
            - (descLeftMargin + this.maxTargetNameLength + leftMargin);

        // determine hanging indent for multiline descriptions
        int descIndent = this.lineWidth - descWidth;

        // target name uses left margin
        printWriter.print(space(leftMargin));

        // target name is padded to the max target name length
        printWriter.print(pad(name, this.maxTargetNameLength));

        // description, if available, is printed in a formatted manner
        if (desc == null) {
            printWriter.println();
        } else {
            printWriter.print(space(descLeftMargin));
            printWriter.println(lineWrap(desc, this.lineWidth, descIndent));
        }

        log(stringWriter.toString());
    }

    /**
     * Prints the "sub" (or "other") targets in the current project.
     * "Sub" targets are those whose <code>&lt;target&gt;</code> element does not carry
     * a <code>description</code> attribute. */
    private void printSubTargets() {
        printTargets(subTargets, "Other targets:");
    }

    /**
     * Logs a formatted list of target names with an optional description.
     *
     * @param targets The sorted set of {@link Target}s to be described.
     *              Must not be <code>null</code>.
     * @param heading The heading to display.
     *                Should not be <code>null</code>.
     */
    private void printTargets(SortedSet targets, String heading) {
        if (!targets.isEmpty()) {
            log(heading);

            Iterator iterator = targets.iterator();

            while (iterator.hasNext()) {
                Target target = (Target) iterator.next();
                printSingleTarget(target);
            }
        }

        log(lineSep);
    }

    /**
     * Configures the task to include information about the project's default target
     * in the logged output.
     *
     * @param includeDefaultTarget if <code>true</code>, the logged output will
     * contain a line specifying the target configured as the project default.
     * If <code>false</code>, that information is omitted.
     */
    public void setIncludeDefaultTarget(boolean includeDefaultTarget) {
        this.includeDefaultTarget = includeDefaultTarget;
    }

    /**
     * Configures the task to include the project description in the logged output.
     *
     * @param includeDescription if <code>true</code>, the logged output will include
     * the project's description, if any is provided. If <code>false</code>,
     * that information is omitted.
     */
    public void setIncludeDescription(boolean includeDescription) {
        this.includeDescription = includeDescription;
    }

    /**
     * Configures the task to include "internal" targets in the logged output.
     *
     * @param includeInternalTargets if <code>true</code>, the logged output
     * will include the name and, if available, the description of every
     * target whose name starts with <code>-</code> (hyphen), and which
     * can hence be only invoked only from within Ant.
     * If <code>false</code>, that information is omitted.
     */
    public void setIncludeInternalTargets(boolean includeInternalTargets) {
        this.includeInternalTargets = includeInternalTargets;
    }

    /**
     * Configures the task to include "main" targets in the logged output.
     *
     * @param includeMainTargets if <code>true</code>, the logged output will include
     * the name of every target for which a description was provided, followed
     * by that description itself, formatted in such a way so as not to exceed
     * the maximum configured line width. If <code>false</code>,
     * that information is omitted.
     */
    public void setIncludeMainTargets(boolean includeMainTargets) {
        this.includeMainTargets = includeMainTargets;
    }

    /**
     * Configures the task to include "sub" targets in the logged output.
     *
     * @param includeSubTargets if <code>true</code>, the logged output will include
     * the name of every target for which no description was provided.
     * If <code>false</code>, that information is omitted.
     */
    public void setIncludeSubTargets(boolean includeSubTargets) {
        this.includeSubTargets = includeSubTargets;
    }

    /**
     * Configures the maximum line width of the logged output.
     *
     * @param lineWidth sets the maximum line width. The description of targets
     * will be line wrapped in such a way that this total maximum line width is
     * not exceeded.
     */
    public void setLineWidth(int lineWidth) {
        this.lineWidth = lineWidth;
    }

    /**
     * Inserts newlines into a string at appropriate places so the string never exceeds
     * a given maximum line width. The newlines are inserted at the space character
     * nearest the maximum line width.
     *
     * @param original the original text. If the length of this text is
     * <code>&lt;= totalLineWidth</code>, it is returned unchanged.
     * @param totalLineWidth the maximum line width, in characters.
     * @param indent a "hanging indent". If <code>&gt; 0</code>, after a newline is inserted
     * into the string, a corresponding number of spaces is inserted into the string as well.
     * It is still guaranteed, however, that the total line width (text plus indent) of the
     * returned string is never <code>&gt; totalLineWidth</code>.
     * @return the original string, with newlines and indentation inserted as described
     * above.
     * @throws IllegalArgumentException if <code>indent &lt; 0</code>.
     */
    private static String lineWrap(String original, int totalLineWidth,
        int indent) {
        StringBuffer buffer = new StringBuffer(original);

        if (indent < 0) {
            throw new IllegalArgumentException("\"indent\" must not be < 0.");
        }

        int indentedLineTextWidth = totalLineWidth - indent;

        int lineStart = 0;
        int lineEnd = totalLineWidth - indent;

        while ((lineStart + indentedLineTextWidth) < buffer.length()) {
            lineEnd = buffer.toString().lastIndexOf(" ",
                    lineStart + indentedLineTextWidth);

            if ((lineEnd == -1) || (lineEnd < lineStart)) {
                // we haven't found a space at which to line wrap, try the
                // next possible opportunity
                lineStart++;

                continue;
            }

            // replace the space character with newline
            buffer.replace(lineEnd, lineEnd + 1, lineSep);

            // insert the spaces needed for the hanging indent
            buffer.insert(lineEnd + lineSep.length(), space(indent));

            // skip forward
            lineStart = lineEnd + lineSep.length() + indent;
        }

        return buffer.toString();
    }

    /**
     * Helper mthod that pads a line of text, i.e. appends whitespace to the text until it
     * reaches the specified length.
     *
     * @param text the original text to be padded
     * @param length the requested length to which the text will be padded
     * @return the padded text. If <code>text.length() &gt;= length</code>,
     *        <code>text</code> is returned unchanged.
     */
    private static String pad(String text, int length) {
        String ret;
        if (text.length() >= length) {
            ret = text;
        } else {
            ret = text + space(length - text.length());
        }
        return ret;
    }

    /**
     * Helper method that creates a {@link String} of the specified length, containing
     * only space characters.
     *
     * @param length the desired string length
     * @return a {@link String} of length <code>length</code>, consisting only of space
     * characters.
     */
    private static String space(int length) {
        char[] spaces = new char[length];
        Arrays.fill(spaces, ' ');

        return String.valueOf(spaces);
    }

    /**
     * A {@link Comparator} for {@link Target}s. Allows a collection
     * of {@link Target}s to be lexically sorted in a {@link SortedSet}, based on their
     * target names. */
    private class TargetComparator implements Comparator {
        /**
         * Compares one target to another by target name.
         *
         * @param object an object. Must be a {@link Target}.
         * @param anotherObject the object to compare to. Must also
         *        be a {@link Target}.
         * @return <code>0</code> if both targets have the same name
         *         (shouldn't happen for two {@link Target}s in the same
         *         {@link Project}, <code>&lt;1</code> if the name of the
         *          target <code>object</code> is less than that of the
         *          target <code>anotherObject</code>, <code>gt;1</code>
         *          otherwise.
         * @throws ClassCastException if either of the two objects can't
         *         be converted to {@link Target}.*/
        public int compare(Object object, Object anotherObject) {
            Target target = (Target) object;
            Target anotherTarget = (Target) anotherObject;

            return target.toString().compareTo(anotherTarget.toString());
        }
    }
}
