/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "Ant" and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
package org.apache.tools.ant.taskdefs;

import org.apache.tools.ant.BuildEvent;
import org.apache.tools.ant.BuildFileTest;
import org.apache.tools.ant.BuildListener;
import org.apache.tools.ant.Project;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;


/**
 * A JUnit test case for the {@link ProjectHelp} task. Compares the output of the
 * {@link ProjectHelp} task to a template file found in the test resource directory.
 *
 * <p>After major changes to the {@link ProjectHelp} task, it may be necessary to
 * redefine the template. You can do so by invoking this test case with the
 * system property <code>org.apache.tools.ant.taskdefs.ProjectHelpTest.recreateTemplate</code>
 * set to <code>true</code>. <strong>Do so only after you are sure that you know
 * exactly what you are doing</strong>.
 *
 * @author <a href="mailto:f.g.haas@gmx.net">Florian G. Haas</a>
 */
public class ProjectHelpTest extends BuildFileTest {
    /** The encoding used for reading and writing files. */
    public static final String ENCODING = "ISO-8859-1";

    /** The directory where to find test resources. */
    private static final File TESTDIR = new File("src/etc/testcases/taskdefs");

    /** The build file configuring the test. */
    private static final File BUILDFILE = new File(TESTDIR, "projecthelp.xml");

    /** The output file to which the {@link ProjectHelpTestListener} will write
     * its output. */
    private static File outFile = new File(TESTDIR, "projecthelp.out");

    /** The template file against which the output will be compared. */
    private static File templateFile = new File(outFile.getAbsolutePath() +
            ".template");

    /** Flag to recreate the template file, rather than use it for comparison.
     * <strong>CAUTION:</strong> Setting this to <code>true</code> <em>redefines</em>
     * the template to compare against. You should normally never do this. */
    private static boolean recreateTemplate = false;

    /** Static initializer. Deletes and re-creates the output file in case
     * it already exists. */
    static {
        String recreateProp = System.getProperty(ProjectHelpTest.class.getName() +
                ".recreateTemplate");

        if (recreateProp != null) {
            recreateTemplate = Boolean.valueOf(recreateProp).booleanValue();
        }

        if (recreateTemplate) {
            String warning = "About to recreate template file " +
                templateFile.getAbsolutePath() +
                ". I really hope you know what your're doing.";
            System.out.println(warning);
            outFile = templateFile;
        }

        try {
            if (outFile.exists()) {
                outFile.delete();
                outFile.createNewFile();
            }
        } catch (IOException e) {
            String msg = "Failed to create new file " +
                outFile.getAbsolutePath() + ": " + e.getMessage();
            ;
            throw new RuntimeException(msg);
        }
    }

    /** The output stream to which the output is logged. */
    private OutputStream out;

    /**
     * Creates the test.
     */
    public ProjectHelpTest(String name) throws Exception {
        super(name);
    }

    /**
     * Sets up the test. Configures {@link ProjectHelpTestListener} as the build
     * listener, sets up the output stream and input streams.
     */
    public void setUp() throws Exception {
        configureProject(BUILDFILE.getAbsolutePath(), Project.MSG_DEBUG);

        out = new FileOutputStream(outFile.getAbsolutePath(), true);

        BuildListener listener = new ProjectHelpTestListener(Project.MSG_INFO,
                out);
        getProject().addBuildListener(listener);
    }

    /**
     * Tears down the test. Closes the output stream used by
     * {@link ProjectHelpTestListener}.
     */
    public void tearDown() throws Exception {
        out.close();
        super.tearDown();
    }

    /**
     * Executes the <code>testDefault</code> target defined in the build file,
     * and compares its output to the result template.
     */
    public void testDefault() throws Exception {
        executeTarget("testDefault");
        checkAgainstTemplate("testDefault");
    }

    /**
     * Executes the <code>testNoSubTargets</code> target defined in the build file,
     * and compares its output to the result template.
     */
    public void testNoSubTargets() throws Exception {
        executeTarget("testNoSubTargets");
        checkAgainstTemplate("testNoSubTargets");
    }

    /**
     * Executes the <code>testSubTargetsOnly</code> target defined in the build file,
     * and compares its output to the result template.
     */
    public void testSubTargetsOnly() throws Exception {
        executeTarget("testSubTargetsOnly");
        checkAgainstTemplate("testSubTargetsOnly");
    }

    /**
     * Executes the <code>testDefaultTargetOnly</code> target defined in the build file,
     * and compares its output to the result template.
     */
    public void testDefaultTargetOnly() throws Exception {
        executeTarget("testDefaultTargetOnly");
        checkAgainstTemplate("testDefaultTargetOnly");
    }

    /**
     * Executes the <code>testShowInternal</code> target defined in the build file,
     * and compares its output to the result template.
     */
    public void testShowInternal() throws Exception {
        executeTarget("testShowInternal");
        checkAgainstTemplate("testShowInternal");
    }

    /**
     * Executes the <code>testSmallLineWidth</code> target defined in the build file,
     * and compares its output to the result template.
     */
    public void testSmallLineWidth() throws Exception {
        executeTarget("testSmallLineWidth");
        checkAgainstTemplate("testSmallLineWidth");
    }

    /**
     * Executes the <code>testBigLineWidth</code> target defined in the build file,
     * and compares its output to the result template.
     */
    public void testBigLineWidth() throws Exception {
        executeTarget("testBigLineWidth");
        checkAgainstTemplate("testBigLineWidth");
    }

    /**
     * Executes the <code>testShowEverything</code> target defined in the build file,
     * and compares its output to the result template.
     */
    public void testShowEverything() throws Exception {
        executeTarget("testShowEverything");
        checkAgainstTemplate("testShowEverything");
    }

    /**
     * Checks the output produced by invoking a specific target against
     * the result template. If {@link #recreateTemplate} is <code>true</code>,
     * this method does nothing.
     *
     * @param targetName the name of the target having been executed.
     * @throws IOException if the output file, or the template file,
     * can't be read.
     */
    private void checkAgainstTemplate(String targetName)
        throws IOException {
        if (!recreateTemplate) {
            String templateLine = "";
            String resultLine = "";

            // set up the reader for the result file
            InputStream resultIn = new FileInputStream(outFile);
            LineNumberReader resultReader = new LineNumberReader(new InputStreamReader(
                        resultIn, ENCODING));

            // set up the reader for the template file
            InputStream templateIn = new FileInputStream(templateFile);
            LineNumberReader templateReader = new LineNumberReader(new InputStreamReader(
                        templateIn, ENCODING));

            // find the section that applies to the test case
            while (!templateLine.equals("------------- BEGIN Target \"" +
                        targetName + "\" -------------")) {
                templateLine = templateReader.readLine();
            }

            while (!resultLine.equals("------------- BEGIN Target \"" +
                        targetName + "\" -------------")) {
                resultLine = resultReader.readLine();
            }

            // now, compare the lines in that section one by one
            while (!templateLine.equals("------------- END Target \"" +
                        targetName + "\" -------------")) {
                int templateLineNumber = templateReader.getLineNumber();
                int resultLineNumber = resultReader.getLineNumber();

                String contentMsg = "Content of line # " + resultLineNumber +
                    " in " + outFile.getAbsolutePath() +
                    " doesn't match corresponding line # " +
                    templateLineNumber + " in " +
                    templateFile.getAbsolutePath() + ".";
                assertEquals(contentMsg, templateLine, resultLine);

                // read next line from both streams
                templateLine = templateReader.readLine();
                resultLine = resultReader.readLine();
            }

            // clean up
            resultReader.close();
            templateReader.close();
        }
    }

    /**
     * Our own personal build listener. Writes to an output file
     * (test result) which can then be compared to a template.
     */
    private class ProjectHelpTestListener implements BuildListener {
        /** The character encoding to be used for writing the output. */
        private static final String ENCODING = ProjectHelpTest.ENCODING;

        /** The logging threshold. */
        private int logLevel;

        /** The {@link PrintWriter} which prints to the output file. */
        private PrintWriter writer;

        /**
         * Constructs a test listener which will ignore log events
         * above the given level.
         *
         * @param logLevel the logging threshold.
         * @param stream the output stream to write to.
         */
        public ProjectHelpTestListener(int logLevel, OutputStream stream)
            throws UnsupportedEncodingException {
            this(logLevel, new OutputStreamWriter(stream, ENCODING));
        }

        /**
         * This constructor is protected on purpose. For test reliability,
         * must ensure that writer encoding is {@link #ENCODING}.
         *
         * @param logLevel the logging threshold.
         * @param stream the writer to be used for output.
         */
        protected ProjectHelpTestListener(int logLevel, Writer writer) {
            this.logLevel = logLevel;
            this.writer = new PrintWriter(writer, true);
        }

        /** Empty. */
        public void buildStarted(BuildEvent event) {
        }

        /** Empty. */
        public void buildFinished(BuildEvent event) {
        }

        /** Logs a message about starting to execute a target. */
        public void targetStarted(BuildEvent event) {
            String msg = "------------- BEGIN Target \"" +
                event.getTarget().getName() + "\" -------------";
            writer.println(msg);
        }

        /** Logs a message about having finished to execute a target. */
        public void targetFinished(BuildEvent event) {
            String msg = "------------- END Target \"" +
                event.getTarget().getName() + "\" -------------";
            writer.println(msg);
            writer.println();
        }

        /** Empty. */
        public void taskStarted(BuildEvent event) {
        }

        /** Empty. */
        public void taskFinished(BuildEvent event) {
        }

        /**
         * Fired whenever a message is logged.
         *
         * @see BuildEvent#getMessage()
         * @see BuildEvent#getPriority()
         */
        public void messageLogged(BuildEvent event) {
            if (event.getPriority() <= logLevel) {
                writer.println(event.getMessage());
            }
        }
    }
}
