/*
 * Copyright  2000-2004 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package org.apache.tools.ant.util;

import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.net.URL;
import org.apache.tools.ant.BuildException;

/**
 * This class encapsulates methods to create URLs.
 * @since Ant 1.7
 */

public final class URLUtils {
    private static interface URLHelper {
        File createFile(String fileOrURL);
        URL createURL(File file);
        URL createURL(String fileOrURL);
        boolean isAbsolute(String fileOrURL);
        String normalize(String fileOrURL);
        String resolve(String fileOrURL, File dir);
    }

    private static class URLHelper12 implements URLHelper {

        public File createFile(String fileOrURL) {
            try {
                if (fileOrURL.charAt(0) == '/') {
                    return new File(fileOrURL);
                }
                URL url = new URL(fileOrURL);
                if ("file".equals(url.getProtocol())) {
                    return new File(url.getFile());
                }
                throw new BuildException(fileOrURL + " is not a file URL");
            } catch (BuildException e) {
                throw e;
            } catch (Exception e) {
                return new File(fileOrURL);
            }
        }
        public URL createURL(File f) {
            try {
                return f.toURL();
            } catch (Exception e) {
                throw new BuildException(e.getMessage(), e);
            }
        }
        public URL createURL(String fileOrURL) {
            try {
                if (fileOrURL.charAt(0) == '/') {
                    return createURL(new File(fileOrURL));
                }
                return new URL(fileOrURL);
            } catch (Exception e) {
                return createURL(new File(fileOrURL));
            }
        }
        public boolean isAbsolute(String fileOrURL) {
            try {
                if (fileOrURL.charAt(0) == '/') {
                    return true;
                }
                URL x = new URL(fileOrURL);
                return (x.getProtocol() != null);
            } catch (Exception e) {
                return new File(fileOrURL).isAbsolute();
            }
        }
        public String normalize(String fileOrURL) {
            return fileOrURL;
        }
        public String resolve(String fileOrURL, File dir) {
            try {
                if (fileOrURL.charAt(0) == '/') {
                    try {
                        return new File(fileOrURL)
                            .getCanonicalFile()
                            .toURL()
                            .toExternalForm();
                    } catch (Exception e1) {
                        throw new BuildException(e1);
                    }
                }

                URL url = new URL(fileOrURL);
                if (url.getProtocol() == null) {
                    return new File(dir, fileOrURL)
                        .getCanonicalFile()
                        .toURL()
                        .toExternalForm();
                }
                return url.toExternalForm();
            } catch (Exception e) {
                File f = new File(fileOrURL);
                if (!f.isAbsolute()) {
                    f = new File(dir, fileOrURL);
                }
                try {
                    return f.getCanonicalFile().toURL().toExternalForm();
                } catch (Exception e1) {
                    throw new BuildException(e1);
                }
            }
        }
    }

    private static class URLHelper14 implements URLHelper {

        private final Class clazz;
        private final Constructor constructor;
        private final Constructor fileConstructor;
        private final Method fileToURI;
        private final Method getSchema;
        private final Method isAbsolute;
        private final Method normalize;
        private final Method resolve;
        private final Method toURL;

        URLHelper14() throws Exception {
            clazz = Class.forName("java.net.URI");
            constructor = clazz.getConstructor(new Class[] {String.class });
            normalize = clazz.getMethod("normalize", null);
            isAbsolute = clazz.getMethod("isAbsolute", null);
            getSchema = clazz.getMethod("getSchema", null);
            toURL = clazz.getMethod("toURL", null);
            resolve = clazz.getMethod("resolve", new Class[] {clazz });
            fileConstructor = File.class.getConstructor(new Class[] {clazz });
            fileToURI = File.class.getMethod("toURI", null);
        }
        private File createFile(Object uri) throws Exception {
            return (File) fileConstructor.newInstance(new Object[] {uri });
        }

        public File createFile(String fileOrURL) {
            try {
                if (fileOrURL.charAt(0) == '/') {
                    return new File(fileOrURL);
                }
                Object uri = createURI(fileOrURL);
                uri = normalize(uri);
                if (isAbsolute(uri)) {
                    String schema = getSchema(uri);
                    if ("file".equals(schema)) {
                        return createFile(uri);
                    }
                    throw new BuildException(fileOrURL + " is not a file URL");
                }
                return new File(uri.toString());
            } catch (BuildException e) {
                throw e;
            } catch (Exception e) {
                return new File(fileOrURL);
            }
        }

        private Object createURI(String fileOrURL) throws Exception {
            return constructor.newInstance(new Object[] {fileOrURL });
        }
        public URL createURL(File f) {
            try {
                Object uri = toURI(f);
                uri = normalize(uri);
                return toURL(uri);
            } catch (Exception e) {
                throw new BuildException(e.getMessage(), e);
            }
        }
        public URL createURL(String fileOrURL) {
            try {
                if (fileOrURL.charAt(0) == '/') {
                    return createURL(new File(fileOrURL));
                }
                Object uri = createURI(fileOrURL);
                uri = normalize(uri);
                return toURL(uri);
            } catch (Exception e) {
                return createURL(new File(fileOrURL));
            }
        }
        private String getSchema(Object uri) throws Exception {
            return (String) getSchema.invoke(uri, null);
        }
        private boolean isAbsolute(Object uri) throws Exception {
            return ((Boolean) isAbsolute.invoke(uri, null)).booleanValue();
        }
        public boolean isAbsolute(String fileOrURL) {
            try {
                if (fileOrURL.charAt(0) == '/') {
                    return true;
                }
                Object uri = createURI(fileOrURL);
                return isAbsolute(uri);
            } catch (Exception e) {
                return new File(fileOrURL).isAbsolute();
            }
        }
        private Object normalize(Object uri) throws Exception {
            return normalize.invoke(uri, null);
        }
        public String normalize(String fileOrURL) {
            try {
                if (fileOrURL.charAt(0) == '/') {
                    try {
                        return normalize(toURI(new File(fileOrURL))).toString();
                    } catch (Exception e1) {
                        throw new BuildException(e1);
                    }
                }
                return normalize(createURI(fileOrURL)).toString();
            } catch (Exception e) {
                try {
                    return normalize(toURI(new File(fileOrURL))).toString();
                } catch (Exception e1) {
                    throw new BuildException(e1);
                }
            }
        }
        private Object resolve(Object root, Object relative) throws Exception {
            return resolve.invoke(root, new Object[] {relative });
        }
        public String resolve(String fileOrURL, File dir) {
            try {
                if (fileOrURL.charAt(0) == '/') {
                    try {
                        return normalize(toURI(new File(fileOrURL))).toString();
                    } catch (Exception e1) {
                        throw new BuildException(e1);
                    }
                }
                Object uri = createURI(fileOrURL);
                uri = normalize(uri);
                if (!isAbsolute(uri)) {
                    Object dirUri = toURI(dir);
                    uri = resolve(dirUri, uri);
                }
                return uri.toString();
            } catch (Exception e) {
                File f = new File(fileOrURL);
                if (!f.isAbsolute()) {
                    f = new File(dir, fileOrURL);
                }
                try {
                    return normalize(toURI(f)).toString();
                } catch (Exception e1) {
                    throw new BuildException(e1);
                }
            }
        }
        private Object toURI(File file) throws Exception {
            return fileToURI.invoke(file, null);
        }
        private URL toURL(Object uri) throws Exception {
            return (URL) toURL.invoke(uri, null);
        }
    }
    private static URLHelper urlHelper;
    static {
        URLHelper x = null;
        try {
            x = new URLHelper14();
        } catch (Throwable e) {
            x = new URLHelper12();
        }
        urlHelper = x;
    }
    /**
     * creates a file from a absolute or relative file or url.
     * @param fileOrURL absolute or relative file or url
     * @return a file
     */
    public static File createFile(String fileOrURL) {
        return urlHelper.createFile(transformFileSep(fileOrURL));
    }
    /**
     * creates a URL from a absolute or relative file or url
     * @param fileOrURL absolute or relative file or url
     * @return an URL
     */
    public static URL createURL(String fileOrURL) {
        return urlHelper.createURL(transformFileSep(fileOrURL));
    }
    /**
     * indicates whether the denoted fileOrURL is absolute or relative.
     * @param fileOrURL absolute or relative file or url
     * @return true if <code>fileOrURL</code> denotes a absolute file or url
     *         , false if it is relative
     */
    public static boolean isAbsolute(String fileOrURL) {
        return urlHelper.isAbsolute(transformFileSep(fileOrURL));
    }
    /**
     * normalizes an absolute or relative file or url
     * @param fileOrURL absolute or relative file or url
     * @return normalized file or url
     */
    public static String normalize(String fileOrURL) {
        return urlHelper.normalize(transformFileSep(fileOrURL));
    }
    /**
     * resolves an absolute or relative file or url with respect to a basedirectory.
     * @param fileOrURL absolute or relative file or url
     * @param dir base directory
     * @return resolved url
     */
    public static String resolve(String fileOrURL, File dir) {
        return urlHelper.resolve(transformFileSep(fileOrURL), dir);
    }
    /**
     * transforms the fileseparators to url format ('/')
     * @param fileOrURL absolute or relative file or url
     * @return fileOrURL with '/'as fileSeparator
     */
    public static String transformFileSep(String fileOrURL) {
        return fileOrURL.replace('\\', '/');
    }
    private URLUtils() {
    }

}
