/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "apr.h"
#include "apr_atomic.h"
#include "apr_thread_mutex.h"

#include "apr_private.h"

#include <stdlib.h>


/* This note is by Colin Hirsch <share-apache@think42.com>

   Changes:
   - Added atomic primitives for Darwin/Mac OS X; these require
     Mac OS 10.4 or higher, and therefore a change to the build
     system to detect these cases and define a symbol similar to
     how SOLARIS2 is defined.
   - Removed 'volatile' from all prototypes; memory objects are
     not volatile: accesses are. Note that for apr_atomic_casptr
     the volatile was completely bogus since it declared the
     pointee of the pointee volatile; in all other cases the
     volatile is not, in general, sufficient, and leads to code
     pessimisation in the no-thread case.
   - Similarly 'const void *cmp' for the third argument of
     apr_atomic_casptr is bogus since it again declares the
     pointee const.
   - Added apr_atomic_read32 for i386 and x86-64.
   - Changed operating-system dependent include files to be
     included only when needed.
   - Added #if !defined to all specific implementations.
   - Added apr_atomic_casptr for Solaris 10 implementation.
   - Moved !APR_HAS_THREADS implementations to top of file,
     "out of the way"...
   - Replaced the additional USE_GENERIC_ATOMICS define
     by the more aptly named FORBID_INLINE_ASSEMBLER and
     changed to appropriate use of these two macros.
   - Added implementation of PowerPC atomic read and set.
   - Added Darwin implementations of all atomic operations.
   - Fixed the Solaris 10 implementations to use the _nv
     variants wherever necessary.
   - Added memory barriers to Solaris 10 read32/set32.
   - Added Solaris 10 add32/sub32 implementations.
   - Changed generic implementations to force consistent
     per-object locking by either only using the mutex,
     or only using cas32/read32 (whenever available).
   - Changed num_atomic_hash from macro to constant.
   - Renamed CHECK macro to APR_CHECK; less likely collisions.
   - Split the generic mutex implementations from the generic
     implementations that use cas32/read32.
   - Added Sparc v9 implementations of read32/set32/cas32.

   Notes:
   - This version was tested on the following systems:
   - The Sparc assembler implementations require a Sparc v9 and,
     on older GCCs, also an appropriate compile switch; requires
     changes to the build system for detecting the cpu type, and
     passing -mcpu=v9 to the compiler.
   - The Darwin implementations require Mac OS X 10.4 or greater;
     the build system must be changed to set the DARWIN macro to
     some value that can then be tested in this file...
   - Building on Solaris 8 initially failed because a getpass()
     function was redefined in the apr source.
   - Testing on Linux initially had USE_GENERIC_ATOMICS defined,
     I am no expert on automake/autoconf etc. so I can't tell why.
   - Mixing atomic functions that use a mutex, and atomic
     functions that use assembler or other 'direct' locking, is NOT
     in general safe.
   - For the consistent locking to work, every specific implementation
     MUST either provide both cas32 and read32, or MUST be completely
     removed.
   - The PowerPC inline implemenations can produce livelocks:
http://www-306.ibm.com/chips/techlib/techlib.nsf/techdocs/79B6E24422AA101287256E93006C957E/$file/PowerPC_970FX_errata_DD3.X_V1.7.pdf

   Tested:
   A test consisted of compiling apr and running testatomic.
   - 4-way PPC 970/Apple GCC 5363/Darwin 8.8.0/32bit
   - 8-way UltraSparc II/GCC 4.0.2/Solaris 8/32bit
     (requires CFLAGS="-mcpu=v9" ./configure)
   - 2/4-way Intel Xeon/GCC 4.0.2/Linux 2.4.20-8smp/32bit
*/


#if !defined(APR_HAS_THREADS)

/*****************************************/

APR_DECLARE(apr_status_t) apr_atomic_init(apr_pool_t *p)
{
   return APR_SUCCESS;
}

APR_DECLARE(apr_uint32_t) apr_atomic_read32(apr_uint32_t *mem)
{
   return *mem;
}

APR_DECLARE(void) apr_atomic_set32(apr_uint32_t *mem, apr_uint32_t val)
{
   *mem = val;
}

APR_DECLARE(apr_uint32_t) apr_atomic_add32(apr_uint32_t *mem, apr_uint32_t val)
{
   apr_uint32_t nrv = *mem;
   *mem += val;
   return nrv;
}

APR_DECLARE(void) apr_atomic_sub32(apr_uint32_t *mem, apr_uint32_t val)
{
   *mem -= val;
}

APR_DECLARE(apr_uint32_t) apr_atomic_inc32(apr_uint32_t *mem)
{
   apr_uint32_t nrv = *mem;
   ++*mem;
   return nrv;
}

APR_DECLARE(int) apr_atomic_dec32(apr_uint32_t *mem)
{
   return --*mem;
}

APR_DECLARE(apr_uint32_t) apr_atomic_cas32(apr_uint32_t *mem,
                                           apr_uint32_t with,
                                           apr_uint32_t cmp)
{
   apr_uint32_t nrv = *mem;

   if (nrv == cmp)
      *mem = with;

   return nrv;
}

APR_DECLARE(apr_uint32_t) apr_atomic_xchg32(apr_uint32_t *mem,
                                            apr_uint32_t val)
{
   apr_uint32_t nrv = *mem;
   *mem = val;
   return nrv;
}

APR_DECLARE(void*) apr_atomic_casptr(void **mem,
                                     void *with,
                                     void *cmp)
{
   void *nrv = *mem;

   if (nrv == cmp)
      *mem = with;

   return nrv;
}

#else /* APR_HAS_THREADS */

/*****************************************/

#if (defined(__GNUC__) && defined(__STRICT_ANSI__)) || defined(USE_GENERIC_ATOMICS)
/* force use of generic atomics if building e.g. with -std=c89, which
 * doesn't allow inline asm */
#define FORBID_INLINE_ASSEMBLER
#endif

/*****************************************/

#if (defined(__i386__) || defined(__x86_64__)) \
    && defined(__GNUC__) && !defined(FORBID_INLINE_ASSEMBLER)

#if !defined(APR_OVERRIDE_ATOMIC_READ32)
APR_DECLARE(apr_uint32_t) apr_atomic_read32(apr_uint32_t *mem)
{
   return *(volatile apr_uint32_t *)mem;
}
#define APR_OVERRIDE_ATOMIC_READ32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_SET32)
APR_DECLARE(void) apr_atomic_set32(apr_uint32_t *mem,
                                   apr_uint32_t val)
{
   *(volatile apr_uint32_t *)mem = val;
}
#define APR_OVERRIDE_ATOMIC_SET32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_CAS32)
APR_DECLARE(apr_uint32_t) apr_atomic_cas32(apr_uint32_t *mem, 
                                           apr_uint32_t with,
                                           apr_uint32_t cmp)
{
    apr_uint32_t prev;

    asm volatile ("lock; cmpxchgl %1,%2"
                  : "=a" (prev)
                  : "r" (with), "m" (*mem), "0"(cmp) 
                  : "memory", "cc");
    return prev;
}
#define APR_OVERRIDE_ATOMIC_CAS32
#endif

static apr_uint32_t inline intel_atomic_add32(apr_uint32_t *mem, 
                                              apr_uint32_t val)
{
    asm volatile ("lock; xaddl %0,%1"
                  : "=r"(val), "=m"(*mem)
                  : "0"(val), "m"(*mem)
                  : "memory", "cc");
    return val;
}

#if !defined(APR_OVERRIDE_ATOMIC_ADD32)
APR_DECLARE(apr_uint32_t) apr_atomic_add32(apr_uint32_t *mem, 
                                           apr_uint32_t val)
{
    return intel_atomic_add32(mem, val);
}
#define APR_OVERRIDE_ATOMIC_ADD32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_SUB32)
APR_DECLARE(void) apr_atomic_sub32(apr_uint32_t *mem, apr_uint32_t val)
{
    asm volatile ("lock; subl %1,%0"
                  :
                  : "m" (*mem), "r" (val)
                  : "memory", "cc");
}
#define APR_OVERRIDE_ATOMIC_SUB32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_DEC32)
APR_DECLARE(int) apr_atomic_dec32(apr_uint32_t *mem)
{
    unsigned char prev;

    asm volatile ("lock; decl %1;\n\t"
                  "setnz %%al"
                  : "=a" (prev)
                  : "m" (*(mem))
                  : "memory", "cc");
    return prev;
}
#define APR_OVERRIDE_ATOMIC_DEC32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_INC32)
APR_DECLARE(apr_uint32_t) apr_atomic_inc32(apr_uint32_t *mem)
{
    return intel_atomic_add32(mem, 1);
}
#define APR_OVERRIDE_ATOMIC_INC32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_XCHG32)
APR_DECLARE(apr_uint32_t) apr_atomic_xchg32(apr_uint32_t *mem, apr_uint32_t val)
{
    apr_uint32_t prev = val;

    asm volatile ("lock; xchgl %0, %1"
                  : "=r" (prev)
                  : "m" (*mem), "0"(prev)
                  : "memory");
    return prev;
}
#define APR_OVERRIDE_ATOMIC_XCHG32
#endif

#endif /* __i386__ || __x86_64__ && __GNUC__ */

/*****************************************/

#if defined(__sparc__) && defined(__GNUC__)     \
      && !defined(FORBID_INLINE_ASSEMBLER)

#if !defined(APR_OVERRIDE_ATOMIC_READ32)
APR_DECLARE(apr_uint32_t) apr_atomic_read32(apr_uint32_t *mem)
{
   asm volatile( "membar #StoreLoad | #LoadLoad" : : : "memory" );
   return *(volatile apr_uint32_t*)mem;
}
#define APR_OVERRIDE_ATOMIC_READ32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_SET32)
APR_DECLARE(void) apr_atomic_set32(apr_uint32_t *mem, apr_uint32_t val)
{
   *(volatile apr_uint32_t*)mem = val;
   asm volatile( "membar #StoreStore | #StoreLoad" : : : "memory" );
}
#define APR_OVERRIDE_ATOMIC_SET32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_CAS32)
APR_DECLARE(apr_uint32_t) apr_atomic_cas32(apr_uint32_t *mem,
                                           apr_uint32_t with,
                                           apr_uint32_t cmp)
{
   apr_uint32_t prev;

   asm volatile( "cas [%1],%2,%0"
                 : "=r" (prev)
                 : "r" (mem), "r" (cmp), "0" (with)
                 : "memory" );

   return prev;
}
#define APR_OVERRIDE_ATOMIC_CAS32
#endif

#endif /* defined(__sparc__) && defined(__GNUC__)
          && !defined(FORBID_INLINE_ASSEMBLER) */

/*****************************************/

#if defined(__POWERPC__ ) && defined(__GNUC__) \
   && !defined(FORBID_INLINE_ASSEMBLER)

#if !defined(APR_OVERRIDE_ATOMIC_READ32)
APR_DECLARE(apr_uint32_t) apr_atomic_read32(apr_uint32_t *mem)
{
   asm volatile ("sync" ::: "memory" );
   return *(volatile apr_uint32_t*)mem;
}
#define APR_OVERRIDE_ATOMIC_READ32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_SET32)
APR_DECLARE(void) apr_atomic_set32(apr_uint32_t *mem, apr_uint32_t val)
{
   *(volatile apr_uint32_t*)mem = val;
   asm volatile ("sync" ::: "memory" );
}
#define APR_OVERRIDE_ATOMIC_SET32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_CAS32)
APR_DECLARE(apr_uint32_t) apr_atomic_cas32(apr_uint32_t *mem,
                                           apr_uint32_t swap,
                                           apr_uint32_t cmp)
{
    apr_uint32_t prev;
                                                                                
    asm volatile ("0:\n\t"                   /* retry local label     */
                  "lwarx  %0,0,%1\n\t"       /* load prev and reserve */
                  "cmpw   %0,%3\n\t"         /* does it match cmp?    */
                  "bne-   1f\n\t"            /* ...no, bail out       */
                  "stwcx. %2,0,%1\n\t"       /* ...yes, conditionally
                                                store swap            */
                  "bne-   0b\n\t"            /* start over if we lost
                                                the reservation       */
                  "1:"                       /* exit local label      */

                  : "=&r"(prev)                        /* output      */
                  : "b" (mem), "r" (swap), "r"(cmp)    /* inputs      */
                  : "memory", "cc");                   /* clobbered   */
    return prev;
}
#define APR_OVERRIDE_ATOMIC_CAS32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_ADD32)
APR_DECLARE(apr_uint32_t) apr_atomic_add32(apr_uint32_t *mem,
                                           apr_uint32_t delta)
{
    apr_uint32_t prev, temp;
                                                                                
    asm volatile ("0:\n\t"                   /* retry local label     */
                  "lwarx  %0,0,%2\n\t"       /* load prev and reserve */
                  "add    %1,%0,%3\n\t"      /* temp = prev + delta   */
                  "stwcx. %1,0,%2\n\t"       /* conditionally store   */
                  "bne-   0b"                /* start over if we lost
                                                the reservation       */

                  /*XXX find a cleaner way to define the temp         
                   *    it's not an output
                   */
                  : "=&r" (prev), "=&r" (temp)        /* output, temp */
                  : "b" (mem), "r" (delta)            /* inputs       */
                  : "memory", "cc");                  /* clobbered    */
    return prev;
}
#define APR_OVERRIDE_ATOMIC_ADD32
#endif

#endif /* __POWERPC__ */

/*****************************************/

// DARWIN >= 4
#if defined(DARWIN) && !defined(USE_GENERIC_ATOMICS)

#include <inttypes.h>
#include <libkern/OSAtomic.h>

#if !defined(APR_OVERRIDE_ATOMIC_READ32)
APR_DECLARE(apr_uint32_t) apr_atomic_read32(apr_uint32_t *mem)
{
   OSMemoryBarrier();
   return *(volatile apr_uint32_t*)mem;
}
#define APR_OVERRIDE_ATOMIC_READ32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_SET32)
APR_DECLARE(void) apr_atomic_set32(apr_uint32_t *mem, apr_uint32_t val)
{
   *(volatile apr_uint32_t*)mem = val;
   OSMemoryBarrier();
}
#define APR_OVERRIDE_ATOMIC_SET32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_ADD32)
APR_DECLARE(apr_uint32_t) apr_atomic_add32(apr_uint32_t *mem, apr_uint32_t val)
{
   return OSAtomicAdd32(val, (int32_t *)mem) - val;
}
#define APR_OVERRIDE_ATOMIC_ADD32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_SUB32)
APR_DECLARE(void) apr_atomic_sub32(apr_uint32_t *mem, apr_uint32_t val)
{
   OSAtomicAdd32(-val, (int32_t *)mem);
}
#define APR_OVERRIDE_ATOMIC_SUB32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_INC32)
APR_DECLARE(apr_uint32_t) apr_atomic_inc32(apr_uint32_t *mem)
{
   return apr_atomic_add32(mem, 1);
}
#define APR_OVERRIDE_ATOMIC_INC32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_DEC32)
APR_DECLARE(int) apr_atomic_dec32(apr_uint32_t *mem)
{
   return OSAtomicAdd32(0xffffffff, (int32_t *)mem);
}
#define APR_OVERRIDE_ATOMIC_DEC32
#endif

/* Apple's OSAtomicCompareAndSwap is a pain because it doesn't
   return the old value, which is however needed for the return
   value of apr_atomic_cas32 if the compare fails...
   The following implementation is ugly as hell, but should
   never be used as long as the PowerPC and i386/x86_64 versions
   are kept above.
*/

#if !defined(APR_OVERRIDE_ATOMIC_CAS32)
APR_DECLARE(apr_uint32_t) apr_atomic_cas32(apr_uint32_t *mem,
                                           apr_uint32_t with,
                                           apr_uint32_t cmp)
{
   apr_uint32_t tmp;

   do {
      if (OSAtomicCompareAndSwap32(cmp, with, (int32_t *)mem))
         return cmp;
      else if ((tmp = apr_atomic_read32(mem)) == with)
         return cmp;  /* pretend */

   } while ( tmp == cmp );

   return tmp;
}
#define APR_OVERRIDE_ATOMIC_CAS32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_XCHG32)
APR_DECLARE(apr_uint32_t) apr_atomic_xchg32(apr_uint32_t *mem,
                                            apr_uint32_t val)
{
   apr_uint32_t tmp;

   do {
      tmp = apr_atomic_read32(mem);
   } while (!OSAtomicCompareAndSwap32(tmp, val, (int32_t *)mem));

   return tmp;
}
#define APR_OVERRIDE_ATOMIC_XCHG32
#endif

#endif /* DARWIN && DARWIN >= 4 */

/*****************************************/

#if (defined(SOLARIS2) && SOLARIS2 >= 10) \
    && !defined(USE_GENERIC_ATOMICS)

#include <atomic.h>

#if !defined(APR_OVERRIDE_ATOMIC_CAS32)
APR_DECLARE(apr_uint32_t) apr_atomic_cas32(apr_uint32_t *mem,
                                           apr_uint32_t with,
                                           apr_uint32_t cmp)
{
    return atomic_cas_32(mem, cmp, with);
}
#define APR_OVERRIDE_ATOMIC_CAS32
#endif /* APR_OVERRIDE_ATOMIC_CAS32 */

#if !defined(APR_OVERRIDE_ATOMIC_ADD32)
APR_DECLARE(apr_uint32_t) apr_atomic_add32(apr_uint32_t *mem,
                                           apr_uint32_t val)
{
   return atomic_add_32_nv(mem, val) - val;
}
#define APR_OVERRIDE_ATOMIC_ADD32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_SUB32)
APR_DECLARE(void) apr_atomic_sub32(apr_uint32_t *mem,
                                   apr_uint32_t val)
{
   atomic_add_32(mem, -val);
}
#define APR_OVERRIDE_ATOMIC_SUB32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_DEC32)
APR_DECLARE(int) apr_atomic_dec32(apr_uint32_t *mem)
{
   return atomic_dec_32_nv(mem);
}
#define APR_OVERRIDE_ATOMIC_DEC32
#endif /* APR_OVERRIDE_ATOMIC_DEC32 */

#if !defined(APR_OVERRIDE_ATOMIC_INC32)
APR_DECLARE(apr_uint32_t) apr_atomic_inc32(apr_uint32_t *mem)
{
    return atomic_inc_32_nv(mem) - 1;
}
#define APR_OVERRIDE_ATOMIC_INC32
#endif /* APR_OVERRIDE_ATOMIC_INC32 */

#if !defined(APR_OVERRIDE_ATOMIC_READ32)
APR_DECLARE(apr_uint32_t) apr_atomic_read32(apr_uint32_t *mem)
{
   membar_enter();
   return *(volatile apr_uint32_t *)mem;
}
#define APR_OVERRIDE_ATOMIC_READ32
#endif

#if !defined(APR_OVERRIDE_ATOMIC_SET32)
APR_DECLARE(void) apr_atomic_set32(apr_uint32_t *mem, apr_uint32_t val)
{
   *(volatile apr_uint32_t *)mem = val;
   membar_exit();
}
#define APR_OVERRIDE_ATOMIC_SET32
#endif /* APR_OVERRIDE_ATOMIC_SET32 */

#if !defined(APR_OVERRIDE_ATOMIC_XCHG32)
APR_DECLARE(apr_uint32_t) apr_atomic_xchg32(apr_uint32_t *mem,
                                            apr_uint32_t val) 
{
    return atomic_swap_32(mem, val);
}
#define APR_OVERRIDE_ATOMIC_XCHG32
#endif /* APR_OVERRIDE_ATOMIC_XCHG32 */

#if !define(APR_OVERRIDE_ATOMIC_CASPTR)
APR_DECLARE(void*) apr_atomic_casptr(void **mem,
                                     void *with,
                                     void *cmp)
{
   return atomic_cas_ptr(mem, cmp, with);
}
#define APR_OVERRIDE_ATOMIC_CASPTR
#endif

#endif /* SOLARIS2 && SOLARIS2 >= 10 */

/*****************************************/

#if defined(APR_OVERRIDE_ATOMIC_INIT)
#error Unallowed definition of apr_atomic_init.
#endif

#if defined(APR_OVERRIDE_ATOMIC_CAS32) && !defined(APR_OVERRIDE_ATOMIC_READ32)
#error Inconsistent set of functions defined.
#endif

#if !defined(APR_OVERRIDE_ATOMIC_CAS32) && defined(APR_OVERRIDE_ATOMIC_READ32)
#error Inconsistent set of functions defined.
#endif

/*****************************************/

#if !defined(APR_OVERRIDE_ATOMIC_CASPTR) || !defined(APR_OVERRIDE_ATOMIC_CAS32) || !defined(APR_OVERRIDE_ATOMIC_READ32)

static const unsigned num_atomic_hash = 7;
/* shift by 2 or 3 to get rid of alignment issues */
#define ATOMIC_HASH(x) (unsigned int)(((unsigned long)(x)>>((sizeof(void *)<8)?2:3))%(unsigned int)num_atomic_hash)
static apr_thread_mutex_t **hash_mutex;
#define APR_CHECK(x) do { if ((x) != APR_SUCCESS) abort(); } while (0)

apr_status_t apr_atomic_init(apr_pool_t *p)
{
    int i;
    apr_status_t rv;
    hash_mutex = apr_palloc(p, sizeof(apr_thread_mutex_t*) * num_atomic_hash);

    for (i = 0; i < num_atomic_hash; i++) {
        rv = apr_thread_mutex_create(&(hash_mutex[i]),
                                     APR_THREAD_MUTEX_DEFAULT, p);
        if (rv != APR_SUCCESS) {
           return rv;
        }
    }
    return APR_SUCCESS;
}
#define APR_OVERRIDE_ATOMIC_INIT
#endif /* !defined(APR_OVERRIDE_ATOMIC_CASPTR) || !defined(APR_OVERRIDE_ATOMIC_CAS32) || !defined(APR_OVERRIDE_ATOMIC_READ32) /*

/*****************************************/

#if !defined(APR_OVERRIDE_ATOMIC_INC32)
apr_uint32_t apr_atomic_inc32(apr_uint32_t *mem) 
{
    return apr_atomic_add32(mem, 1);
}
#define APR_OVERRIDE_ATOMIC_INC32
#endif /* !defined(APR_OVERRIDE_ATOMIC_INC32) */

/*****************************************/

#if defined(APR_OVERRIDE_ATOMIC_CAS32) && defined(APR_OVERRIDE_ATOMIC_READ32)

#if !defined(APR_OVERRIDE_ATOMIC_ADD32)
apr_uint32_t apr_atomic_add32(apr_uint32_t *mem, apr_uint32_t val)
{
    apr_uint32_t old_value, new_value;
    
    do {
       old_value = apr_atomic_read32(mem);
        new_value = old_value + val;
    } while (apr_atomic_cas32(mem, new_value, old_value) != old_value);
    return old_value;
}
#define APR_OVERRIDE_ATOMIC_ADD32
#endif /* !defined(APR_OVERRIDE_ATOMIC_ADD32) */

#if !defined(APR_OVERRIDE_ATOMIC_SUB32)
void apr_atomic_sub32(apr_uint32_t *mem, apr_uint32_t val)
{
    apr_uint32_t old_value, new_value;
    
    do {
       old_value = apr_atomic_read32(mem);
        new_value = old_value - val;
    } while (apr_atomic_cas32(mem, new_value, old_value) != old_value);
}
#define APR_OVERRIDE_ATOMIC_SUB32
#endif /* !defined(APR_OVERRIDE_ATOMIC_SUB32) */

#if !defined(APR_OVERRIDE_ATOMIC_DEC32)
int apr_atomic_dec32(apr_uint32_t *mem)
{
    apr_uint32_t old_value, new_value;
    
    do {
       old_value = apr_atomic_read32(mem);
        new_value = old_value - 1;
    } while (apr_atomic_cas32(mem, new_value, old_value) != old_value);
    return old_value != 1;
}
#define APR_OVERRIDE_ATOMIC_DEC32
#endif /* !defined(APR_OVERRIDE_ATOMIC_DEC32) */

#if !defined(APR_OVERRIDE_ATOMIC_XCHG32)
apr_uint32_t apr_atomic_xchg32(apr_uint32_t *mem, apr_uint32_t val)
{
    apr_uint32_t prev;
    do {
       prev = apr_atomic_read32(mem);
    } while (apr_atomic_cas32(mem, val, prev) != prev);
    return prev;
}
#define APR_OVERRIDE_ATOMIC_XCHG32
#endif /* !defined(APR_OVERRIDE_ATOMIC_XCHG32) */

#if !defined(APR_OVERRIDE_ATOMIC_SET32)
void apr_atomic_set32(apr_uint32_t *mem, apr_uint32_t val) 
{
   apr_uint32_t prev;
   do {
      prev = apr_atomic_read32(mem);
   } while (apr_atomic_cas32(mem, val, prev) != prev);
}
#define APR_OVERRIDE_ATOMIC_SET32
#endif /* !defined(APR_OVERRIDE_ATOMIC_SET32) */

#else /* defined(APR_OVERRIDE_ATOMIC_CAS32) && defined(APR_OVERRIDE_ATOMIC_READ32) */

/*****************************************/

#if defined(APR_OVERRIDE_ATOMIC_ADD32)
#error Found function that should not exist.
#endif
apr_uint32_t apr_atomic_add32(apr_uint32_t *mem, apr_uint32_t val)
{
    apr_uint32_t old_value;

    apr_thread_mutex_t *lock = hash_mutex[ATOMIC_HASH(mem)];
       
    APR_CHECK(apr_thread_mutex_lock(lock));
    old_value = *mem;
    *mem += val;
    APR_CHECK(apr_thread_mutex_unlock(lock));

    return old_value;
}
#define APR_OVERRIDE_ATOMIC_ADD32

#if defined(APR_OVERRIDE_ATOMIC_SUB32)
#error Found function that should not exist.
#endif
void apr_atomic_sub32(apr_uint32_t *mem, apr_uint32_t val) 
{
    apr_thread_mutex_t *lock = hash_mutex[ATOMIC_HASH(mem)];
       
    APR_CHECK(apr_thread_mutex_lock(lock));
    *mem -= val;
    APR_CHECK(apr_thread_mutex_unlock(lock));
}
#define APR_OVERRIDE_ATOMIC_SUB32

#if defined(APR_OVERRIDE_ATOMIC_DEC32)
#error Found function that should not exist.
#endif
int apr_atomic_dec32(apr_uint32_t *mem) 
{
    apr_thread_mutex_t *lock = hash_mutex[ATOMIC_HASH(mem)];
    apr_uint32_t new;

    APR_CHECK(apr_thread_mutex_lock(lock));
    (*mem)--;
    new = *mem;
    APR_CHECK(apr_thread_mutex_unlock(lock));
    return new;
}
#define APR_OVERRIDE_ATOMIC_DEC32

#if defined(APR_OVERRIDE_ATOMIC_CAS32)
#error Found function that should not exist.
#endif
apr_uint32_t apr_atomic_cas32(apr_uint32_t *mem,
                              apr_uint32_t with,
			      apr_uint32_t cmp)
{
    apr_uint32_t prev;
    apr_thread_mutex_t *lock = hash_mutex[ATOMIC_HASH(mem)];

    APR_CHECK(apr_thread_mutex_lock(lock));
    prev = *mem;
    if (prev == cmp) {
        *mem = with;
    }
    APR_CHECK(apr_thread_mutex_unlock(lock));
    return prev;
}
#define APR_OVERRIDE_ATOMIC_CAS32

#if defined(APR_OVERRIDE_ATOMIC_XCHG32)
#error Found function that should not exist.
#endif
apr_uint32_t apr_atomic_xchg32(apr_uint32_t *mem, apr_uint32_t val)
{
    apr_uint32_t prev;
    apr_thread_mutex_t *lock = hash_mutex[ATOMIC_HASH(mem)];

    APR_CHECK(apr_thread_mutex_lock(lock));
    prev = *mem;
    *mem = val;
    APR_CHECK(apr_thread_mutex_unlock(lock));
    return prev;
}
#define APR_OVERRIDE_ATOMIC_XCHG32

#if defined(APR_OVERRIDE_ATOMIC_SET32)
#error Found function that should not exist.
#endif
void apr_atomic_set32(apr_uint32_t *mem, apr_uint32_t val) 
{
    apr_thread_mutex_t *lock = hash_mutex[ATOMIC_HASH(mem)];

    APR_CHECK(apr_thread_mutex_lock(lock));
    *mem = val;
    APR_CHECK(apr_thread_mutex_unlock(lock));
}
#define APR_OVERRIDE_ATOMIC_SET32

#if defined(APR_OVERRIDE_ATOMIC_READ32)
#error Found function that should not exist.
#endif
APR_DECLARE(apr_uint32_t) apr_atomic_read32(apr_uint32_t *mem)
{
   apr_uint32_t nrv;
   apr_thread_mutex_t *lock = hash_mutex[ATOMIC_HASH(mem)];

   APR_CHECK(apr_thread_mutex_lock(lock));
   nrv = *mem;
   APR_CHECK(apr_thread_mutex_unlock(lock));

   return nrv;
}

#endif /* else of defined(APR_OVERRIDE_ATOMIC_CAS32) && defined(APR_OVERRIDE_ATOMIC_READ32) */

/*****************************************/

#if !defined(APR_OVERRIDE_ATOMIC_CASPTR)
void *apr_atomic_casptr(void **mem, void *with, void *cmp)
{
    void *prev;
    apr_thread_mutex_t *lock = hash_mutex[ATOMIC_HASH(mem)];

    APR_CHECK(apr_thread_mutex_lock(lock));
    prev = *(void **)mem;
    if (prev == cmp) {
        *mem = with;
    }
    APR_CHECK(apr_thread_mutex_unlock(lock));
    return prev;
}
#endif /* !defined(APR_OVERRIDE_ATOMIC_CASPTR) */

/*****************************************/

#if !defined(APR_OVERRIDE_ATOMIC_INIT)
APR_DECLARE(apr_status_t) apr_atomic_init(apr_pool_t *p)
{
   return APR_SUCCESS;
}
#define APR_OVERRIDE_ATOMIC_INIT
#endif

#endif /* APR_HAS_THREADS */
