// Aurora API.

package client2

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
)

// GetJobSummary gets a summary of all jobs matching the request from aurora.
func GetJobSummary(url string, request *JobSummaryRequest) (result *JobSummaryResult, err error) {
	response, err := doRequest(url+"/getJobSummary", request)
	return &response.Result.JobSummaryResult, err
}

// GetRoleSummary gets an overview of all active roles from aurora.
func GetRoleSummary(url string) (result *RoleSummaryResult, err error) {
	response, err := doRequest(url+"/getRoleSummary", nil)
	return &response.Result.RoleSummaryResult, err
}

// GetTaskStatus gets detailed info about the scheduler state of a specific job.
func GetTaskStatus(url string, request *TaskStatusRequest) (result *TaskStatusResult, err error) {
	response, err := doRequest(url+"/getTasksStatus", request)
	return &response.Result.ScheduleStatusResult, err
}

func doRequest(url string, request interface{}) (response *Response, err error) {
	// Prepare request data.
	var data = make([]byte, 0)
	if request != nil {
		if data, err = json.Marshal(request); err != nil {
			return nil, err
		}
	}

	// Do the post.
	postResponse, err := http.Post(url, "text/json", bytes.NewReader(data))
	defer postResponse.Body.Close()
	if err != nil {
		return nil, err
	}
	if postResponse.StatusCode != 200 {
		return nil, fmt.Errorf(
			"Received response code %d from scheduler", postResponse.StatusCode)
	}

	// Parse response.
	body, err := ioutil.ReadAll(postResponse.Body)
	if err != nil {
		return nil, err
	}

	// Ensure that the request was successful.
	response = new(Response)
	err = json.Unmarshal(body, response)
	if response.ResponseCode != ResponseOK {
		return response, fmt.Errorf(string(body))
	}
	return response, err
}
