/**
 *
 *  CONEXANT PROPRIETARY AND CONFIDENTIAL SOFTWARE
 *  Conexant Systems Inc. (c) 2007
 *  Austin, TX
 *  All Rights Reserved
 * 
 *  $Id: ClientToProxyIoHandler.java 2719 2007-09-17 23:34:15Z scott.carter $
 * 
 */

package com.conexant.scm.server;

import java.net.InetSocketAddress;

import org.apache.mina.common.ByteBuffer;
import org.apache.mina.common.ConnectFuture;
import org.apache.mina.common.IoConnector;
import org.apache.mina.common.IoFuture;
import org.apache.mina.common.IoFutureListener;
import org.apache.mina.common.IoSession;
import org.apache.mina.common.RuntimeIOException;
import org.apache.mina.common.TrafficMask;
import org.apache.mina.common.WriteFuture;
import org.apache.mina.util.SessionLog;

import com.conexant.scm.util.ClientVersion;
import com.conexant.scm.util.ClientVersionFormatException;

/**
 * Handles the client to proxy part of the proxied connection.
 */
public final class ClientToProxyIoHandler extends AbstractProxyIoHandler
{
    private final ServerToProxyIoHandler connectorHandler; 
    private final IoConnector connector; 
    private final InetSocketAddress address;
    private final ServerProperties proxyServerProperties;

    public ClientToProxyIoHandler(ServerToProxyIoHandler connectorHandler, 
                                  IoConnector connector,
                                  InetSocketAddress address,
                                  ServerProperties proxyServerProperties)
    {
        this.connectorHandler = connectorHandler;
        this.connector = connector;
        this.address = address;
        this.proxyServerProperties = proxyServerProperties;
    }

    /**
     * Overrides sessionOpened() handler to register an 
     * operationComplete() handler. 
     */
    @Override
    public void sessionOpened(final IoSession session) throws Exception 
    {
        SessionLog.debug(session, "*** New client connected");
        connector.connect(address, connectorHandler).addListener(new IoFutureListener() {
                public void operationComplete(IoFuture f)
                {
                    onOperationComplete(f, session);
                }
            });
    }

    /**
     * Overrides AbstractProxyIoHandler's 
     * preprocessReceivedMessage() method to provide the primary 
     * function of the SCM proxy server - i.e., version ID 
     * verification against clients that connect. 
     */
    @Override
    protected ByteBuffer preprocessReceivedMessage(ProxySessionState toProxySessionState, ProxySessionState fromProxySessionState, ByteBuffer messageBuffer)
    {
        SessionLog.debug(toProxySessionState.session, "in preprocessReceivedMessage()");
        ByteBuffer rb = messageBuffer;
        if (!toProxySessionState.isAuthorizedClientVersion)
        {
            rb.mark();

            // TODO: Consider refactoring all of the version ID decoding into a separate class

            // TODO: Consider reading an MD5 hash from the buffer, and then comparing that against our own
            //       generated hash of the version string.  This will allow for a little more security in
            //       the protocol.

            // TODO: the hard-coded '4' should really be, at a minimum, a constant ... 
            byte[] header = new byte[4];
            for (int i = 0; i < header.length; i++)
                header[i] = rb.get();
            String headerString = new String(header);
            try
            {
                // TODO: the hard-coded '5' should really be, at a minimum, a constant ... 
                byte[] versionData = new byte[5];
                for (int i = 0; i < versionData.length; i++)
                    versionData[i] = rb.get();

                toProxySessionState.clientVersionID = new ClientVersion(versionData);
            }
            catch (ClientVersionFormatException cvfe)
            {
                SessionLog.error(toProxySessionState.session, "Could not parse client version format, disconnecting client.  Exception:", cvfe);
            }
            SessionLog.info(toProxySessionState.session, "*** Client connected, attempting to validate...");
            SessionLog.debug(toProxySessionState.session, "Client header string = " + headerString);
            SessionLog.debug(toProxySessionState.session, "Client version number = " + toProxySessionState.clientVersionID);

            // TODO: The hard-coding of the 'CSCM' header string should really be refactored... (ScmProxyProtocol class?)
            if (headerString.equals("CSCM") &&
                (toProxySessionState.clientVersionID.compareTo(this.proxyServerProperties.getMinimumClientVersion()) >= 0))
            {
                toProxySessionState.isAuthorizedClientVersion = true;
                SessionLog.info(toProxySessionState.session, "*** Authorized client connected with version " + toProxySessionState.clientVersionID);
            }
            else
            {
                // TODO: It is really rude to just disconnect clients without telling them why!
                //       In order to fix this, we really need to know how to speak Subversion,
                //       so that we can write a formatted error message back to the client before
                //       disconnecting them.

                String minClientVersion = this.proxyServerProperties.getMinimumClientVersion().toString();
                String errorMessage = "Unauthorized client connected; minimum required version is " + minClientVersion;
                String errorResponse = "( failure ( " + errorMessage.length() + ":" + errorMessage + " ) )";

                ByteBuffer wb = ByteBuffer.allocate(errorResponse.length());
                wb.put(errorResponse.getBytes());
                wb.flip();
                WriteFuture wf = fromProxySessionState.session.write(wb);
                wf.join();
                if (!wf.isWritten())
                {
                    SessionLog.debug(fromProxySessionState.session, "error writing failure response");
                }

                SessionLog.debug(fromProxySessionState.session, "writing error response '"+errorResponse+"'");

                SessionLog.info(fromProxySessionState.session, "*** Unauthorized client connected, closing connection...");
                fromProxySessionState.session.setAttachment(null);
                fromProxySessionState.session.close();
                toProxySessionState.session.setAttachment(null);
                toProxySessionState.session.close();
            }
        }
        return rb;
    }

    /**
     * Handler for the MINA onOperationComplete() event. 
     * Essentially saves a session state variable as the attachment 
     * for the session. 
     *  
     * @param f
     * @param session
     */
    private void onOperationComplete(IoFuture f, IoSession session)
    {
        ConnectFuture future = (ConnectFuture) f;
        try
        {
            // session is the IoSession representing the client-to-proxy (CSCM Client) connection
            SessionLog.debug(session, "toProxy attachment address = " + session.getRemoteAddress());
            ProxySessionState toProxySessionState = new ProxySessionState();
            toProxySessionState.session = session;
            future.getSession().setAttachment(toProxySessionState);

            // future.getSession() is the IoSession representing the proxy-to-server (WANdisco) connection
            ProxySessionState fromProxySessionState = new ProxySessionState();
            fromProxySessionState.session = future.getSession();
            session.setAttachment(fromProxySessionState);
            SessionLog.debug(session, "fromProxy attachment address = " + future.getSession().getRemoteAddress());

            future.getSession().setTrafficMask(TrafficMask.ALL);
        }
        catch (RuntimeIOException e)
        {
            // Connect failed
            SessionLog.error(session, "Error: " + e.getMessage());
            e.printStackTrace();
            session.close();
        }
        finally
        {
            session.setTrafficMask(TrafficMask.ALL);
        }
    }
}

class ProxySessionState
{
    public IoSession session;
    public boolean isAuthorizedClientVersion = false;
    public ClientVersion clientVersionID;
}
