/**
 * 
 * CONEXANT PROPRIETARY AND CONFIDENTIAL SOFTWARE
 * Conexant Systems Inc. (c) 2007
 * Austin, TX
 * All Rights Reserved
 * 
 * 
 * $Id: ScmProxyServer.java 2719 2007-09-17 23:34:15Z scott.carter $
 *
 */

package com.conexant.scm.server;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.rmi.server.UnicastRemoteObject;

import org.apache.mina.common.IoAcceptor;
import org.apache.mina.common.IoConnector;
import org.apache.mina.common.IoConnectorConfig;
import org.apache.mina.transport.socket.nio.SocketAcceptor;
import org.apache.mina.transport.socket.nio.SocketAcceptorConfig;
import org.apache.mina.transport.socket.nio.SocketConnector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Entrypoint class for the SCM Proxy Server.  This proxy server
 * serves as a way to only allow 'authorized' versions of CSCM
 * access to the underlying repository.
 */
public class ScmProxyServer implements IProxyManagerService
{
    private static final Logger logger = LoggerFactory.getLogger(ScmProxyServer.class);

    public static final String SERVICE_NAME = "ScmProxyServer";

    private IoAcceptor ioAcceptor = null;
    private boolean isStarted = false;

    ServerProperties serverProperties;

    public ScmProxyServer() throws ScmProxyServerException
    {
        // Load properties
        try
        {
            this.serverProperties = ServerProperties.getProperties();
        }
        catch (FileNotFoundException fnfe)
        {
            this.logger.error("Properties file not found", fnfe);
            throw new ScmProxyServerException("Could not find properties file!", fnfe);
        }
        catch (IOException ioe)
        {
            this.logger.error("IOException while loading properties", ioe);
            throw new ScmProxyServerException("I/O Error while attempting to load properties file!", ioe);
        }
        catch (ServerPropertiesFormatException spfe)
        {
            this.logger.error("Configuration error in properties file", spfe);
            throw new ScmProxyServerException("Error in properties file: " + spfe.getMessage(), spfe);
        }

        Runtime.getRuntime().addShutdownHook(new Thread() {
                // VM Shutdown hook that stops the server before quitting.
                public void run() {
                    stopProxyServer(false);
                }
            });

        // Create TCP/IP acceptor.
        ioAcceptor = new SocketAcceptor();
        ((SocketAcceptorConfig)ioAcceptor.getDefaultConfig()).setReuseAddress(true);
    }

    public void startProxyServer() throws RemoteException
    {
        this.logger.debug("START command received");

        if (this.isStarted)
        {
            this.logger.warn("Server already started; ignoring start request.");
            return;
        }

        if (null == this.serverProperties)
        {
            try
            {
                this.serverProperties = ServerProperties.getProperties();
                this.serverProperties.reloadProperties();
            }
            catch (FileNotFoundException fnfe)
            {
                this.logger.error("Properties file not found", fnfe);
                throw new RemoteException("Could not find properties file!", fnfe);
            }
            catch (IOException ioe)
            {
                this.logger.error("IOException while loading properties", ioe);
                throw new RemoteException("I/O Error while attempting to load properties file!", ioe);
            }
            catch (ServerPropertiesFormatException spfe)
            {
                this.logger.error("Configuration error in properties file", spfe);
                throw new RemoteException("Error in properties file: " + spfe.getMessage(), spfe);
            }
        }

        // Create TCP/IP connector.
        IoConnector connector = new SocketConnector();

        // Set connect timeout.
        ((IoConnectorConfig)connector.getDefaultConfig()).setConnectTimeout(this.serverProperties.getConnectTimeout());

        ClientToProxyIoHandler handler = new ClientToProxyIoHandler(new ServerToProxyIoHandler(),
                                                                    connector,
                                                                    new InetSocketAddress(this.serverProperties.getServerAddress(),
                                                                                          this.serverProperties.getServerPort()),
                                                                    this.serverProperties);
        // Start proxy.
        int proxyPort = this.serverProperties.getProxyPort();
        String proxyAddress = this.serverProperties.getProxyAddress();
        try
        {
            this.ioAcceptor.bind(new InetSocketAddress(proxyAddress, proxyPort), handler);
            this.isStarted = true;
            this.logger.info("SCM Proxy Server started, listening on {}:{} ", proxyAddress, proxyPort);
        }
        catch (IOException ioe)
        {
            this.logger.error("IOException occurred while trying to bind acceptor!", ioe);
            throw new RemoteException(ioe.getMessage(), ioe);
        }
    }

    public void stopProxyServer(boolean bQuitAfterStop)
    {
        this.logger.debug("STOP command received");

        if (this.isStarted)
        {
            this.ioAcceptor.unbindAll();

            // Set the properties instance to null.  startProxyServer() will conditionally
            // reload the properties based on this, in order to ensure that the first time
            // the app is launched, we do not load the properties twice.
            this.serverProperties = null;
            this.isStarted = false;

            this.logger.info("SCM Proxy Server has shutdown.");

            if (bQuitAfterStop)
                System.exit(0);
        }
    }

    public void restartProxyServer() throws RemoteException
    {
        this.logger.debug("RESTART command received");

        stopProxyServer(false);
        startProxyServer();
    }

    public static void main(String[] args)
    {
        try
        {
            IProxyManagerService proxyServer = new ScmProxyServer();
            IProxyManagerService stub = (IProxyManagerService) UnicastRemoteObject.exportObject(proxyServer, 0);
            Registry registry = LocateRegistry.getRegistry();
            registry.rebind(SERVICE_NAME, stub);

            proxyServer.startProxyServer();
        }
        catch (ScmProxyServerException se)
        {
            String errorMessage = "Proxy server could not be initialized.  Cause: " + se.getMessage();
            ScmProxyServer.logger.error(errorMessage);
            logger.debug(errorMessage);
            System.exit(1);
        }
        catch (RemoteException re)
        {
            String errorMessage = "Proxy server service could not be registered in RMI registry, aborting.  Cause: " + re.getMessage();
            ScmProxyServer.logger.error(errorMessage);
            logger.debug(errorMessage);
            System.exit(1);
        }
    }
}

