/**
 * 
 * CONEXANT PROPRIETARY AND CONFIDENTIAL SOFTWARE
 * Conexant Systems Inc. (c) 2007
 * Austin, TX
 * All Rights Reserved
 * 
 * 
 * $Id: ServerProperties.java 2719 2007-09-17 23:34:15Z scott.carter $
 *
 */

package com.conexant.scm.server;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.conexant.scm.util.ClientVersion;
import com.conexant.scm.util.ClientVersionFormatException;

public class ServerProperties
{
    final Logger logger = LoggerFactory.getLogger(ServerProperties.class);

    private static final int DEFAULT_PROXY_PORT = 3690;
    //private static final String DEFAULT_PROXY_ADDRESS = "au-svn-01.au.eng.conexant.com";
    private static final int DEFAULT_SERVER_PORT = 13690;
    //private static final String DEFAULT_SERVER_ADDRESS = "au-svn-01.au.eng.conexant.com";
    private static final int DEFAULT_CONNECT_TIMEOUT = 30;
    //private static final String DEFAULT_MINIMUM_CLIENT_VERSION = "1.0.0.1";

    private static final String PROPS_KEY_PROXY_PORT = "proxyPort";
    private static final String PROPS_KEY_PROXY_ADDRESS = "proxyAddress";
    private static final String PROPS_KEY_SERVER_PORT = "serverPort";
    private static final String PROPS_KEY_SERVER_ADDRESS = "serverAddress";
    private static final String PROPS_KEY_CONNECT_TIMEOUT = "connectTimeout";
    private static final String PROPS_KEY_MINIMUM_CLIENT_VERSION = "minimumClientVersion";

    private static final Properties DEFAULT_PROPERTIES = new Properties()
    {
        {
            setProperty(PROPS_KEY_PROXY_PORT, ""+DEFAULT_PROXY_PORT);
            //setProperty(PROPS_KEY_PROXY_ADDRESS, DEFAULT_PROXY_ADDRESS);
            setProperty(PROPS_KEY_SERVER_PORT, ""+DEFAULT_SERVER_PORT);
            //setProperty(PROPS_KEY_SERVER_ADDRESS, DEFAULT_SERVER_ADDRESS);
            setProperty(PROPS_KEY_CONNECT_TIMEOUT, ""+DEFAULT_CONNECT_TIMEOUT);
            //setProperty(PROPS_KEY_MINIMUM_CLIENT_VERSION, DEFAULT_MINIMUM_CLIENT_VERSION);
        }
    };

    private static final String PROPERTIES_FILE_NAME = "scmproxy.properties";
    private static final File propsFile = new File(PROPERTIES_FILE_NAME);
    private static ServerProperties instance = null;

    private int proxyPort;
    private String proxyAddress;
    private int serverPort;
    private String serverAddress;
    private int connectTimeout;
    private ClientVersion minimumClientVersion;

    private ServerProperties() throws FileNotFoundException, IOException, ServerPropertiesFormatException
    {
        reloadProperties();
        this.logger.debug("ServerProperties() initialized");
    }

    public static synchronized ServerProperties getProperties() throws FileNotFoundException, IOException, ServerPropertiesFormatException
    {
        if (null == ServerProperties.instance)
            ServerProperties.instance = new ServerProperties();

        return ServerProperties.instance;
    }

    /**
     * Loads the server's properties settings from the
     * properties file.  Assumes that the prefs file 
     * already exists. 
     */
    public void reloadProperties() throws FileNotFoundException, IOException, ServerPropertiesFormatException
    {
        Properties props = new Properties(DEFAULT_PROPERTIES);

        FileReader fr = null;
        try
        {
            fr = new FileReader(this.propsFile);
            props.load(fr);
        }
        finally
        {
            if (null != fr)
            {
                try
                {
                    fr.close();
                }
                catch (IOException ioe)
                {
                    this.logger.error("IOException occurred while trying to close properties file", ioe);
                }
            }
        }

        String keyValue;

        // validate port number for the proxy server
        keyValue = props.getProperty(PROPS_KEY_PROXY_PORT);
        try
        {
            // no null check - revert to defaults (as stored in DEFAULT_PROPERTIES)
            // if no port is specified for the proxy server
            this.proxyPort = Integer.parseInt(keyValue);
            this.logger.debug("proxyPort = {}", this.proxyPort);
        }
        catch (NumberFormatException nfe)
        {
            throw new ServerPropertiesFormatException(ServerPropertiesFormatException.ErrorType.INVALID_VALUE, PROPS_KEY_PROXY_PORT);
        }

        // validate proxy server address
        keyValue = props.getProperty(PROPS_KEY_PROXY_ADDRESS);
        try
        {
            if (null == keyValue)
                throw new ServerPropertiesFormatException(ServerPropertiesFormatException.ErrorType.MISSING_KEY, PROPS_KEY_PROXY_ADDRESS);

            // do host lookup to validate host name
            InetAddress.getByName(keyValue);
            this.proxyAddress = keyValue;
            this.logger.debug("proxyAddress = {}", this.proxyAddress);
        }
        catch (UnknownHostException uhe)
        {
            throw new ServerPropertiesFormatException(ServerPropertiesFormatException.ErrorType.UNKNOWN_HOST, PROPS_KEY_PROXY_ADDRESS, keyValue);
        }

        // validate port number for the SVN server
        keyValue = props.getProperty(PROPS_KEY_SERVER_PORT);
        try
        {
            // no null check - revert to defaults (as stored in DEFAULT_PROPERTIES)
            // if no port is specified for the SVN server
            this.serverPort = Integer.parseInt(keyValue);
            this.logger.debug("serverPort = {}", this.serverPort);
        }
        catch (NumberFormatException nfe)
        {
            throw new ServerPropertiesFormatException(ServerPropertiesFormatException.ErrorType.INVALID_VALUE, PROPS_KEY_PROXY_PORT);
        }

        // validate SVN server address
        keyValue = props.getProperty(PROPS_KEY_SERVER_ADDRESS);
        try
        {
            if (null == keyValue)
                throw new ServerPropertiesFormatException(ServerPropertiesFormatException.ErrorType.MISSING_KEY, PROPS_KEY_SERVER_ADDRESS);

            // do host lookup to validate host name
            InetAddress.getByName(keyValue);
            this.serverAddress = keyValue;
            this.logger.debug("serverAddress = {}", this.serverAddress);
        }
        catch (UnknownHostException uhe)
        {
            throw new ServerPropertiesFormatException(ServerPropertiesFormatException.ErrorType.UNKNOWN_HOST, PROPS_KEY_SERVER_ADDRESS, keyValue);
        }

        // validate connect timeout parameter
        keyValue = props.getProperty(PROPS_KEY_CONNECT_TIMEOUT);
        try
        {
            if (null == keyValue)
                throw new ServerPropertiesFormatException(ServerPropertiesFormatException.ErrorType.MISSING_KEY, PROPS_KEY_PROXY_PORT);

            this.connectTimeout = Integer.parseInt(keyValue);
            this.logger.debug("connectTimeout = {}", this.connectTimeout);
        }
        catch (NumberFormatException nfe)
        {
            throw new ServerPropertiesFormatException(ServerPropertiesFormatException.ErrorType.INVALID_VALUE, PROPS_KEY_CONNECT_TIMEOUT);
        }

        // validate minimum client version string
        keyValue = props.getProperty(PROPS_KEY_MINIMUM_CLIENT_VERSION);
        if (keyValue.matches("\\d+\\.\\d+\\.\\d+\\.\\d+"))
        {
            try
            {
                this.minimumClientVersion = new ClientVersion(keyValue);
                this.logger.debug("minimumClientVersion = {}", this.minimumClientVersion);
            }
            catch (ClientVersionFormatException cvfe)
            {
                throw new ServerPropertiesFormatException(ServerPropertiesFormatException.ErrorType.INVALID_VERSION, PROPS_KEY_MINIMUM_CLIENT_VERSION);
            }
        }
        else
        {
            throw new ServerPropertiesFormatException(ServerPropertiesFormatException.ErrorType.INVALID_VERSION, PROPS_KEY_MINIMUM_CLIENT_VERSION);
        }
            
        this.logger.info("Reloaded properties.");
    }

    public int getProxyPort()
    {
        return this.proxyPort;
    }

    public String getProxyAddress()
    {
        return this.proxyAddress;
    }

    public int getServerPort()
    {
        return this.serverPort;
    }

    public String getServerAddress()
    {
        return this.serverAddress;
    }

    public int getConnectTimeout()
    {
        return this.connectTimeout;
    }

    public ClientVersion getMinimumClientVersion()
    {
        return this.minimumClientVersion;
    }
}

